<?php

use App\Models\Ad;
use App\Models\User;
use App\Models\UserPlan;
use App\Models\Category;
use App\Models\AlgorithmConfig;
use Illuminate\Support\Facades\DB;

require __DIR__.'/vendor/autoload.php';
$app = require_once __DIR__.'/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

// Cleanup
Ad::where('title', 'LIKE', 'TEST_VIP_%')->forceDelete();
User::where('email', 'LIKE', 'test_vip_%')->delete();
UserPlan::where('plan_name', 'LIKE', 'TEST_VIP_%')->delete();
echo "--- CLEANUP COMPLETE ---\n";

// 1. Create Test Plans
$planA = UserPlan::create(['plan_name' => 'TEST_VIP_PLAN_A', 'price' => 10, 'duration_days' => 30, 'status' => 'Active']);
$planB = UserPlan::create(['plan_name' => 'TEST_VIP_PLAN_B', 'price' => 50, 'duration_days' => 30, 'status' => 'Active']);
$planC = UserPlan::create(['plan_name' => 'TEST_VIP_PLAN_C', 'price' => 100, 'duration_days' => 30, 'status' => 'Active']);

echo "Created Plans: {$planA->id}, {$planB->id}, {$planC->id}\n";

// 2. Create Users
$userA = User::factory()->create(['email' => 'test_vip_a@example.com', 'current_plan_id' => $planA->id]);
$userB = User::factory()->create(['email' => 'test_vip_b@example.com', 'current_plan_id' => $planB->id]);
$userC = User::factory()->create(['email' => 'test_vip_c@example.com', 'current_plan_id' => $planC->id]);

// 3. Create Identical Ads
$category = Category::first() ?? Category::factory()->create();
$baseData = [
    'category_id' => $category->id,
    'description' => 'Test', 'price' => 100, 'currency' => 'INR',
    'location_city' => 'Center', 'location_latitude' => 12.9716, 'location_longitude' => 77.5946,
    'created_at' => now(), 'status' => 'Active', 'ad_type' => 'standard'
];

$adA = Ad::create(array_merge($baseData, ['user_id' => $userA->id, 'title' => 'TEST_VIP_AD_A (Low Boost)']));
$adB = Ad::create(array_merge($baseData, ['user_id' => $userB->id, 'title' => 'TEST_VIP_AD_B (Med Boost)']));
$adC = Ad::create(array_merge($baseData, ['user_id' => $userC->id, 'title' => 'TEST_VIP_AD_C (High Boost)']));

echo "Ads Created.\n";

// 4. Set Dynamic Plan Weights
$config = AlgorithmConfig::getDefault();
if (!$config) $config = AlgorithmConfig::create(['config_key'=>'default']);

// Reset standard weights
$config->updateWeights([
    'location_relevance' => 0, 
    'freshness_factor' => 0, 
    'ad_performance' => 0
]);

// Set Dynamic VIP Settings
// Plan A: 10, Plan B: 50, Plan C: 100
$vipTiers = [
    'priority_ranking' => true,
    "plan_{$planA->id}_boost" => 10,
    "plan_{$planB->id}_boost" => 50,
    "plan_{$planC->id}_boost" => 100,
];

// Explicitly clear legacy keys from this test payload to prove independence
// (The Ad model uses what's in 'vip_tiers' array)
$config->updateVIPSettings(['vip_tiers' => $vipTiers, 'priority_ranking' => true]);

echo "Weights Set (Dynamic Plan IDs). Testing Search...\n";

// 5. Run Query
$results = Ad::active()
    ->where('title', 'LIKE', 'TEST_VIP_%')
    ->orderByInterestAndLocation(12.9716, 77.5946)
    ->get();

echo "\n--- RESULTS ---\n";
foreach ($results as $i => $ad) {
    echo ($i+1) . ". {$ad->title} (Score: " . number_format($ad->relevance_score, 2) . ")\n";
}

// Cleanup
Ad::where('title', 'LIKE', 'TEST_VIP_%')->forceDelete();
User::where('email', 'LIKE', 'test_vip_%')->delete();
$planA->delete();
$planB->delete();
$planC->delete();
echo "--- TEST COMPLETE ---\n";
