<?php

// Test the complete flow from backend to ensure everything is working
require __DIR__.'/vendor/autoload.php';
$app = require_once __DIR__.'/bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use Illuminate\Support\Facades\DB;

echo "=== DEFAULT PLAN IMPLEMENTATION TEST ===\n\n";

// 1. Check default plan setting
echo "1. Checking Default Plan Setting:\n";
$defaultPlan = DB::table('user_plans')->where('is_default', true)->first();
if ($defaultPlan) {
    echo "   ✅ Default Plan: {$defaultPlan->plan_name} (ID: {$defaultPlan->id})\n";
    echo "   ✅ Price: ₹{$defaultPlan->price}\n";
    echo "   ✅ Validity: {$defaultPlan->validity_days} days\n";
} else {
    echo "   ❌ No default plan set!\n";
    exit(1);
}

// 2. Check if users have the default plan assigned
echo "\n2. Checking User Plan Assignments:\n";
$usersWithPlan = DB::table('users')
    ->where('current_plan_id', $defaultPlan->id)
    ->count();
echo "   ✅ Users with default plan: {$usersWithPlan}\n";

// 3. Simulate API response for /plans endpoint
echo "\n3. Simulating API Response (/api/plans):\n";
$plans = DB::table('user_plans')
    ->where('status', 'Active')
    ->orderBy('price', 'asc')
    ->get(['id', 'plan_name', 'price', 'is_default']);

echo "   Plans returned:\n";
foreach ($plans as $plan) {
    $isDefault = $plan->is_default ? ' (DEFAULT)' : '';
    echo "   - {$plan->plan_name} - ₹{$plan->price}{$isDefault}\n";
}

// 4. Simulate getting a user's current plan
echo "\n4. Testing User's Current Plan Detection:\n";
$testUser = DB::table('users')->first();
if ($testUser) {
    echo "   Test User: {$testUser->name} (ID: {$testUser->id})\n";
    echo "   Current Plan ID: " . ($testUser->current_plan_id ?? 'NULL') . "\n";
    
    if ($testUser->current_plan_id) {
        $userPlan = DB::table('user_plans')->find($testUser->current_plan_id);
        if ($userPlan) {
            echo "   ✅ Plan: {$userPlan->plan_name}\n";
            echo "   ✅ Expires: " . ($testUser->plan_expires_at ?? 'N/A') . "\n";
        }
    } else {
        echo "   ⚠️  User has no plan assigned (probably old user)\n";
        echo "   💡 Fix: Assign default plan to existing users\n";
    }
}

// 5. Test what the mobile app will receive
echo "\n5. Simulating Mobile App API Response:\n";
$apiResponse = [
    'plans' => $plans->map(function($p) use ($defaultPlan) {
        return [
            'id' => $p->id,
            'name' => $p->plan_name,
            'price' => $p->price,
            'is_default' => $p->is_default
        ];
    })->toArray(),
    'current_plan_id' => $testUser->current_plan_id ?? null,
    'currency_symbol' => '₹'
];

echo "   Response:\n";
echo "   - Plans count: " . count($apiResponse['plans']) . "\n";
echo "   - Current plan ID: " . ($apiResponse['current_plan_id'] ?? 'NULL') . "\n";
echo "   - Currency: {$apiResponse['currency_symbol']}\n";

// 6. Check what the app will display
echo "\n6. What the App Will Show:\n";
foreach ($plans as $plan) {
    $isCurrent = $testUser && $testUser->current_plan_id == $plan->id;
    $buttonText = $isCurrent ? 'Current Plan ✓' : "Upgrade to {$plan->plan_name}";
    $disabled = $isCurrent ? '(disabled)' : '';
    echo "   - {$plan->plan_name}: [{$buttonText}] {$disabled}\n";
}

echo "\n=== TEST COMPLETE ===\n";

// 7. Recommendations
echo "\n📋 Recommendations:\n";
if (!$testUser->current_plan_id) {
    echo "⚠️  Action needed: Update existing users to have default plan:\n";
    echo "   Run: UPDATE users SET current_plan_id = {$defaultPlan->id} WHERE current_plan_id IS NULL;\n";
}
echo "✅ New users will automatically get '{$defaultPlan->plan_name}' plan on registration\n";
echo "✅ Mobile app will show 'Current Plan ✓' for their active plan\n";
