
    /**
     * Get public ads for a specific user (Seller Profile)
     */
    public function getAdsByUser($userId)
    {
        try {
            $user = \App\Models\User::with(['trustScore', 'currentPlan'])->findOrFail($userId);
            
            // Get stats
            $activeAdsCount = Ad::where('user_id', $userId)->where('status', 'Active')->count();
            $soldAdsCount = Ad::where('user_id', $userId)->where('status', 'Sold')->count();
            
            // Get Ads
            $ads = Ad::with(['category', 'images'])
                ->where('user_id', $userId)
                ->where('status', 'Active')
                ->where('is_active', true)
                ->orderBy('is_featured', 'desc') // Featured first
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            $transformedAds = $ads->getCollection()->map(function($ad) {
                // Image logic
                $mainImage = null;
                if ($ad->relationLoaded('images') && $ad->images->isNotEmpty()) {
                    $path = $ad->images->first()->image_url;
                    $mainImage = str_starts_with($path, 'http') ? $path : url('storage/' . $path);
                }

                return [
                    'id' => $ad->id,
                    'title' => $ad->title,
                    'price' => (string) ($ad->price ?? 0),
                    'currency' => $ad->currency ?? 'USD',
                    'city' => $ad->location_city ?? $ad->city ?? null,
                    'area' => $ad->location_address ?? null,
                    'main_image' => $mainImage,
                    'slug' => 'ad-' . $ad->id,
                    'created_at' => $ad->created_at ? $ad->created_at->toDiffForHumans() : 'Just now',
                    'is_featured' => (bool) ($ad->is_featured ?? false),
                    'is_boosted' => (bool) ($ad->is_boosted ?? false),
                    'views' => $ad->view_count ?? 0,
                ];
            });

            $ads->setCollection($transformedAds);

            return response()->json([
                'success' => true,
                'data' => [
                    'profile' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'avatar' => $user->profile_image ? url('storage/' . $user->profile_image) : null,
                        'city' => $user->city ?? 'Unknown',
                        'country' => $user->country ?? 'Unknown',
                        'member_since' => $user->created_at->format('M Y'),
                        'response_rate' => '98%', // Placeholder - calculate real if available
                        'response_time' => '1 hour', // Placeholder
                        'rating' => 4.9, // Placeholder or fetch from Reviews model
                        'rating_count' => 124, // Placeholder
                        'trust_score' => optional($user->trustScore)->score ?? 100,
                        'trust_tier' => optional($user->trustScore)->tier ?? 'new',
                        'is_verified' => (bool) $user->hasPlanFeature('verified_badge'),
                        'plan_name' => optional($user->currentPlan)->plan_name ?? 'Standard',
                        'active_ads_count' => $activeAdsCount,
                        'sold_ads_count' => $soldAdsCount,
                    ],
                    'ads' => $ads->items(),
                    'meta' => [
                        'current_page' => $ads->currentPage(),
                        'last_page' => $ads->lastPage(),
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch seller profile',
                'error' => $e->getMessage(),
            ], 500);
        }
    }
