<?php

// Simple database setup script
$databasePath = __DIR__ . '/database/database.sqlite';

// Check if database file exists
if (!file_exists($databasePath)) {
    echo "Creating database file...\n";
    touch($databasePath);
}

try {
    // Connect to database
    $pdo = new PDO("sqlite:$databasePath");
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    echo "✅ Connected to database successfully!\n";
    
    // Create ads table
    $sql = "CREATE TABLE IF NOT EXISTS ads (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        title TEXT NOT NULL,
        description TEXT,
        price DECIMAL(10,2) DEFAULT 0.00,
        currency VARCHAR(3) DEFAULT 'USD',
        category_id INTEGER,
        subcategory_id INTEGER,
        user_id INTEGER,
        status VARCHAR(20) DEFAULT 'pending',
        is_featured BOOLEAN DEFAULT 0,
        views INTEGER DEFAULT 0,
        location_city TEXT,
        location_state TEXT,
        location_country TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        deleted_at DATETIME,
        FOREIGN KEY (category_id) REFERENCES categories(id),
        FOREIGN KEY (subcategory_id) REFERENCES subcategories(id),
        FOREIGN KEY (user_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created ads table successfully!\n";
    
    // Add deleted_at column if it doesn't exist
    try {
        $pdo->exec("ALTER TABLE ads ADD COLUMN deleted_at DATETIME");
        echo "✅ Added deleted_at column to ads table\n";
    } catch (PDOException $e) {
        // Column already exists, which is fine
    }
    
    // Create users table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        email TEXT UNIQUE NOT NULL,
        email_verified_at DATETIME,
        password TEXT NOT NULL,
        remember_token TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        is_admin BOOLEAN DEFAULT 0,
        status VARCHAR(20) DEFAULT 'active',
        mobile_number TEXT,
        profile_image TEXT,
        is_verified BOOLEAN DEFAULT 0,
        verification_token TEXT,
        verification_expires_at DATETIME,
        trust_score DECIMAL(5,2) DEFAULT 50.00,
        total_ads INTEGER DEFAULT 0,
        total_views INTEGER DEFAULT 0,
        total_reports INTEGER DEFAULT 0,
        is_suspended BOOLEAN DEFAULT 0,
        suspended_until DATETIME,
        is_banned BOOLEAN DEFAULT 0,
        banned_reason TEXT,
        banned_at DATETIME
    )";
    
    $pdo->exec($sql);
    echo "✅ Created users table successfully!\n";
    
    // Add missing columns to users table if they don't exist
    $columns = [
        'status VARCHAR(20) DEFAULT "active"',
        'mobile_number TEXT',
        'profile_image TEXT',
        'is_verified BOOLEAN DEFAULT 0',
        'verification_token TEXT',
        'verification_expires_at DATETIME',
        'trust_score DECIMAL(5,2) DEFAULT 50.00',
        'total_ads INTEGER DEFAULT 0',
        'total_views INTEGER DEFAULT 0',
        'total_reports INTEGER DEFAULT 0',
        'is_suspended BOOLEAN DEFAULT 0',
        'suspended_until DATETIME',
        'is_banned BOOLEAN DEFAULT 0',
        'banned_reason TEXT',
        'banned_at DATETIME'
    ];
    
    foreach ($columns as $column) {
        try {
            $pdo->exec("ALTER TABLE users ADD COLUMN $column");
            echo "✅ Added column to users table\n";
        } catch (PDOException $e) {
            // Column already exists, which is fine
        }
    }
    
    // Create categories table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS categories (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        description TEXT,
        icon TEXT,
        color VARCHAR(7) DEFAULT '#3b82f6',
        is_active BOOLEAN DEFAULT 1,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )";
    
    $pdo->exec($sql);
    echo "✅ Created categories table successfully!\n";
    
    // Create subcategories table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS subcategories (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        category_id INTEGER,
        description TEXT,
        icon TEXT,
        color VARCHAR(7) DEFAULT '#3b82f6',
        is_active BOOLEAN DEFAULT 1,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (category_id) REFERENCES categories(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created subcategories table successfully!\n";
    
    // Create safety reports table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS safety_reports (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ad_id INTEGER,
        reporter_id INTEGER,
        report_type VARCHAR(50) NOT NULL,
        reason TEXT NOT NULL,
        status VARCHAR(20) DEFAULT 'pending',
        admin_notes TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (ad_id) REFERENCES ads(id),
        FOREIGN KEY (reporter_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created safety_reports table successfully!\n";
    
    // Create user trust scores table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS user_trust_scores (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        trust_score DECIMAL(5,2) DEFAULT 50.00,
        score_reason TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created user_trust_scores table successfully!\n";
    
    // Create safety actions table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS safety_actions (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        ad_id INTEGER,
        admin_user_id INTEGER,
        action_type VARCHAR(50) NOT NULL,
        reason TEXT,
        status VARCHAR(20) DEFAULT 'completed',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id),
        FOREIGN KEY (ad_id) REFERENCES ads(id),
        FOREIGN KEY (admin_user_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created safety_actions table successfully!\n";
    
    // Create analytics events table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS analytics_events (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ad_id INTEGER,
        user_id INTEGER,
        event_type VARCHAR(50) NOT NULL,
        event_data TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (ad_id) REFERENCES ads(id),
        FOREIGN KEY (user_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created analytics_events table successfully!\n";
    
    // Create notifications table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS notifications (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        type VARCHAR(50) NOT NULL,
        title TEXT NOT NULL,
        message TEXT NOT NULL,
        data TEXT,
        is_read BOOLEAN DEFAULT 0,
        read_at DATETIME,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (user_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created notifications table successfully!\n";
    
    // Create chat messages table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS chat_messages (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ad_id INTEGER,
        sender_id INTEGER,
        receiver_id INTEGER,
        message TEXT NOT NULL,
        is_read BOOLEAN DEFAULT 0,
        read_at DATETIME,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        FOREIGN KEY (ad_id) REFERENCES ads(id),
        FOREIGN KEY (sender_id) REFERENCES users(id),
        FOREIGN KEY (receiver_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created chat_messages table successfully!\n";
    
    // Create user plans table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS user_plans (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        plan_name TEXT NOT NULL,
        price DECIMAL(10,2) DEFAULT 0.00,
        duration_days INTEGER DEFAULT 30,
        post_limit INTEGER NULL,
        lead_limit INTEGER NULL,
        analytics_tier TEXT DEFAULT 'basic',
        validity_days INTEGER DEFAULT 30,
        ad_free BOOLEAN DEFAULT 0,
        has_analytics BOOLEAN DEFAULT 0,
        verified_badge BOOLEAN DEFAULT 0,
        theme TEXT NULL,
        status TEXT DEFAULT 'Active',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )";
    
    $pdo->exec($sql);
    echo "✅ Created user_plans table successfully!\n";
    
    // Create ad images table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS ad_images (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ad_id INTEGER,
        image_url TEXT NOT NULL,
        image_path TEXT,
        image_name TEXT,
        image_size INTEGER,
        image_type TEXT,
        is_primary BOOLEAN DEFAULT 0,
        sort_order INTEGER DEFAULT 0,
        alt_text TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        deleted_at DATETIME,
        FOREIGN KEY (ad_id) REFERENCES ads(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created ad_images table successfully!\n";
    
    // Create ad analytics table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS ad_analytics (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ad_id INTEGER,
        user_id INTEGER,
        event_type VARCHAR(50) NOT NULL,
        event_data TEXT,
        ip_address VARCHAR(45),
        user_agent TEXT,
        session_id VARCHAR(255),
        referrer TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        deleted_at DATETIME,
        FOREIGN KEY (ad_id) REFERENCES ads(id),
        FOREIGN KEY (user_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created ad_analytics table successfully!\n";
    
    // Create ad post statistics table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS ad_post_statistics (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        ad_id INTEGER UNIQUE,
        total_views INTEGER DEFAULT 0,
        unique_views INTEGER DEFAULT 0,
        total_impressions INTEGER DEFAULT 0,
        unique_impressions INTEGER DEFAULT 0,
        total_clicks INTEGER DEFAULT 0,
        unique_clicks INTEGER DEFAULT 0,
        conversion_rate DECIMAL(5,2) DEFAULT 0.00,
        bounce_rate DECIMAL(5,2) DEFAULT 0.00,
        avg_time_on_page DECIMAL(6,2) DEFAULT 0.00,
        last_viewed_at DATETIME,
        last_clicked_at DATETIME,
        view_count_today INTEGER DEFAULT 0,
        view_count_week INTEGER DEFAULT 0,
        view_count_month INTEGER DEFAULT 0,
        click_count_today INTEGER DEFAULT 0,
        click_count_week INTEGER DEFAULT 0,
        click_count_month INTEGER DEFAULT 0,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        deleted_at DATETIME,
        FOREIGN KEY (ad_id) REFERENCES ads(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created ad_post_statistics table successfully!\n";
    
    // Create user appeals table if it doesn't exist
    $sql = "CREATE TABLE IF NOT EXISTS user_appeals (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        user_id INTEGER,
        appeal_type VARCHAR(50) NOT NULL,
        appeal_reason TEXT NOT NULL,
        status VARCHAR(20) DEFAULT 'pending',
        admin_notes TEXT,
        admin_id INTEGER,
        reviewed_at DATETIME,
        resolution_notes TEXT,
        is_urgent BOOLEAN DEFAULT 0,
        attachments TEXT,
        appeal_data TEXT,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        deleted_at DATETIME,
        FOREIGN KEY (user_id) REFERENCES users(id),
        FOREIGN KEY (admin_id) REFERENCES users(id)
    )";
    
    $pdo->exec($sql);
    echo "✅ Created user_appeals table successfully!\n";
    
    // Insert sample data
    echo "\n📝 Inserting sample data...\n";
    
    // Insert sample categories
    $categories = [
        ['Electronics', 'Electronic devices and gadgets', '📱', '#3b82f6'],
        ['Vehicles', 'Cars, bikes, and other vehicles', '🚗', '#10b981'],
        ['Real Estate', 'Properties and real estate', '🏠', '#f59e0b'],
        ['Fashion', 'Clothing and accessories', '👗', '#7c3aed'],
        ['Home & Garden', 'Home items and garden supplies', '🌿', '#ef4444']
    ];
    
    foreach ($categories as $category) {
        $stmt = $pdo->prepare("INSERT INTO categories (name, description, icon, color) VALUES (?, ?, ?, ?)");
        $stmt->execute($category);
    }
    echo "✅ Inserted sample categories\n";
    
    // Insert sample subcategories
    $subcategories = [
        ['Smartphones', 1, 'Mobile phones and accessories', '📱', '#3b82f6'],
        ['Laptops', 1, 'Computers and laptops', '💻', '#3b82f6'],
        ['Cars', 2, 'New and used cars', '🚗', '#10b981'],
        ['Motorcycles', 2, 'Bikes and motorcycles', '🏍️', '#10b981'],
        ['Apartments', 3, 'Apartments for sale/rent', '🏢', '#f59e0b'],
        ['Houses', 3, 'Houses and villas', '🏠', '#f59e0b'],
        ['Clothing', 4, 'Men and women clothing', '👕', '#7c3aed'],
        ['Shoes', 4, 'Footwear and accessories', '👟', '#7c3aed'],
        ['Furniture', 5, 'Home furniture and decor', '🛋️', '#ef4444'],
        ['Tools', 5, 'Garden tools and equipment', '🛠️', '#ef4444']
    ];
    
    foreach ($subcategories as $subcategory) {
        $stmt = $pdo->prepare("INSERT INTO subcategories (name, category_id, description, icon, color) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute($subcategory);
    }
    echo "✅ Inserted sample subcategories\n";
    
    // Insert sample user plans
    $plans = [
        ['Standard', 0.00, 30, 5, 10, 'basic', 30, 0, 1, 0, 'standard', 'Active'],
        ['VIP Buyer', 19.99, 30, null, null, 'advanced', 30, 0, 1, 1, 'vip_buyer_theme', 'Active'],
        ['VIP Seller', 24.99, 30, 50, 100, 'premium', 30, 0, 1, 1, 'vip_seller_theme', 'Active'],
        ['VIP Pro', 39.99, 30, 200, 500, 'enterprise', 30, 1, 1, 1, 'vip_pro_theme', 'Active']
    ];
    
    foreach ($plans as $plan) {
        $stmt = $pdo->prepare("INSERT INTO user_plans (plan_name, price, duration_days, post_limit, lead_limit, analytics_tier, validity_days, ad_free, has_analytics, verified_badge, theme, status) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute($plan);
    }
    echo "✅ Inserted sample user plans\n";
    
    // Insert sample ads
    $ads = [
        ['iPhone 15 Pro Max', 'Latest iPhone in excellent condition', 1199.99, 'USD', 1, 1, 1, 'active', 1, 1234, 'San Francisco', 'California', 'USA'],
        ['2023 Tesla Model Y', 'Electric SUV with low mileage', 52990.00, 'USD', 2, 3, 2, 'pending', 0, 892, 'Los Angeles', 'California', 'USA'],
        ['Designer Watch Collection', 'Luxury watches from top brands', 2999.50, 'USD', 4, 7, 3, 'sold', 1, 1567, 'New York', 'New York', 'USA'],
        ['Modern Sofa Set', 'Comfortable sofa for living room', 1299.00, 'USD', 5, 9, 4, 'active', 0, 456, 'Chicago', 'Illinois', 'USA'],
        ['1 Bedroom Apartment', 'Cozy apartment in downtown', 250000.00, 'USD', 3, 5, 5, 'active', 1, 2156, 'Miami', 'Florida', 'USA']
    ];
    
    foreach ($ads as $ad) {
        $stmt = $pdo->prepare("INSERT INTO ads (title, description, price, currency, category_id, subcategory_id, user_id, status, is_featured, views, location_city, location_state, location_country) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute($ad);
    }
    echo "✅ Inserted sample ads\n";
    
    // Insert sample users
    $users = [
        ['Admin User', 'admin@example.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 1, 'active', null, null, 1, 100.00, 50, 1000, 5, 0, null, 0, null, null],
        ['John Doe', 'john@example.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 0, 'active', '123-456-7890', null, 0, 85.50, 25, 500, 2, 0, null, 0, null, null],
        ['Jane Smith', 'jane@example.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi', 0, 'active', '098-765-4321', null, 0, 92.75, 15, 300, 1, 0, null, 0, null, null]
    ];
    
    foreach ($users as $user) {
        $stmt = $pdo->prepare("INSERT INTO users (name, email, password, is_admin, status, mobile_number, profile_image, is_verified, trust_score, total_ads, total_views, total_reports, is_suspended, suspended_until, is_banned, banned_reason, banned_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute($user);
    }
    echo "✅ Inserted sample users\n";
    
    echo "\n🎉 Database setup completed successfully!\n";
    echo "📊 You now have:\n";
    echo "   - 5 sample categories\n";
    echo "   - 10 sample subcategories\n";
    echo "   - 4 user plans (Standard, VIP Buyer, VIP Seller, VIP Pro)\n";
    echo "   - 5 sample ads\n";
    echo "   - 3 sample users (1 admin, 2 regular users)\n";
    echo "\n🚀 You can now access the Enhanced Individual Ad Analytics system!\n";
    
} catch (PDOException $e) {
    echo "❌ Database error: " . $e->getMessage() . "\n";
} catch (Exception $e) {
    echo "❌ Error: " . $e->getMessage() . "\n";
}
