<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\CategoryController;
use App\Http\Controllers\Api\AdController;
use App\Http\Controllers\Api\ChatController;
use App\Http\Controllers\Api\NotificationController;
/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

Route::middleware('auth:sanctum')->group(function () {
    Route::get('/user', function (Request $request) {
        return $request->user()->load('currentPlan');
    });
    Route::get('/user/stats', [App\Http\Controllers\Api\UserStatsController::class, 'index']);
    Route::get('/my-ads', [App\Http\Controllers\Api\AdController::class, 'getMyAds']);
    Route::get('/favorites', [App\Http\Controllers\Api\AdController::class, 'getFavorites']);
    Route::post('/ads/{id}/favorite', [App\Http\Controllers\Api\AdController::class, 'toggleFavorite']);
    Route::get('/my-orders', [App\Http\Controllers\Api\OrderController::class, 'index']);
    Route::delete('/ads/{id}', [App\Http\Controllers\Api\AdController::class, 'destroy']);
    Route::post('/ads/{id}/toggle-status', [App\Http\Controllers\Api\AdController::class, 'toggleStatus']);
    
    // Package Routes
    Route::get('/my-packages', [App\Http\Controllers\Api\PackageController::class, 'getMyPackages']);

    // Plans API (requires auth to get current plan)
    Route::get('/plans', [App\Http\Controllers\Api\PlansController::class, 'index']);
    Route::post('/plans/{planId}/activate', [App\Http\Controllers\Api\PlansController::class, 'activate']);
});

// Packages API (public/auth mixed)
Route::get('/packages', [App\Http\Controllers\Api\PackageController::class, 'index']);
Route::get('/packages/{id}', [App\Http\Controllers\Api\PackageController::class, 'show']);
Route::middleware('auth:sanctum')->post('/packages/{id}/purchase', [App\Http\Controllers\Api\PackageController::class, 'purchase']);

// Blogs API (public)
Route::get('/blogs', [App\Http\Controllers\Api\BlogController::class, 'index']);
Route::get('/blogs/categories', [App\Http\Controllers\Api\BlogController::class, 'categories']);
Route::get('/blogs/{slug}', [App\Http\Controllers\Api\BlogController::class, 'show']);
Route::post('/blogs/{id}/views', [App\Http\Controllers\Api\BlogController::class, 'incrementViews']);

// Pages API (public - for legal policies)
Route::get('/pages', [App\Http\Controllers\Api\PageController::class, 'index']);
Route::get('/pages/{slug}', [App\Http\Controllers\Api\PageController::class, 'show']);

// Analytics API Endpoints
Route::prefix('analytics')->middleware('auth:sanctum')->group(function () {
    // Business Intelligence Dashboard
    Route::get('/business-intelligence', [App\Http\Controllers\Admin\AnalyticsController::class, 'businessIntelligence']);
    Route::get('/ai-insights', [App\Http\Controllers\Admin\AnalyticsController::class, 'getAIInsights']);
    Route::get('/executive-summary', [App\Http\Controllers\Admin\AnalyticsController::class, 'getExecutiveSummary']);
    
    // Ad Post Analytics
    Route::get('/ad-post', [App\Http\Controllers\Admin\AnalyticsController::class, 'adPostAnalytics']);
    
    // Individual Ad Analytics
    Route::get('/individual-ads/{adId}', [App\Http\Controllers\Admin\AnalyticsController::class, 'individualAdAnalytics']);
    
    // User Analytics
    Route::get('/users', [App\Http\Controllers\Admin\AnalyticsController::class, 'userAnalytics']);
    
    // Funnels Analytics
    Route::get('/funnels', [App\Http\Controllers\Admin\AnalyticsController::class, 'funnelsAnalytics']);
    
    // Export Analytics
    Route::get('/export', [App\Http\Controllers\Admin\AnalyticsController::class, 'exportAnalytics']);
    Route::get('/export-data', [App\Http\Controllers\Admin\AnalyticsController::class, 'exportAnalytics']);
    
    // Download endpoint
    Route::get('/download/{filename}', [App\Http\Controllers\Admin\AnalyticsController::class, 'downloadExport']);
});

// Admin Ads Statistics API Routes
Route::prefix('admin/ads')->middleware(['web', 'admin'])->group(function () {
    Route::get('/stats', [App\Http\Controllers\Admin\AdStatisticsController::class, 'getStats'])->name('api.admin.ads.stats');
    Route::get('/charts', [App\Http\Controllers\Admin\AdStatisticsController::class, 'getCharts'])->name('api.admin.ads.charts');
    Route::get('/real-time-stats', [App\Http\Controllers\Admin\AdStatisticsController::class, 'getRealTimeStats'])->name('api.admin.ads.real-time-stats');
    Route::get('/data', [App\Http\Controllers\Admin\AdDataController::class, 'getAdsData'])->name('api.admin.ads.data');
});

// Category Routes
Route::get('categories', [CategoryController::class, 'index']);
Route::get('categories/{id}', [CategoryController::class, 'show']);
Route::get('categories/{id}/attributes', [CategoryController::class, 'getAttributes']);
Route::get('categories/{id}/subcategories', [CategoryController::class, 'getSubcategories']);

// Ad Routes - Public endpoints (no authentication required for viewing)
Route::get('ads', [AdController::class, 'index']); // Guest can view all ads
Route::get('ads/search', [AdController::class, 'search']); // Guest can search ads
Route::get('ads/user/{userId}', [AdController::class, 'getSellerProfile']); // Public Seller Profile - MOVED UP
Route::get('ads/{id}', [AdController::class, 'show']); // Guest can view ad details
Route::get('ads/{id}/related', [AdController::class, 'getRelated']); // Guest can view related ads

// Ad Routes - Protected endpoints (require authentication)
Route::middleware('auth:sanctum')->prefix('ads')->group(function () {
    Route::get('/my-ads', [AdController::class, 'getMyAds']); // Get user's own ads
    Route::post('/', [AdController::class, 'store']);
    Route::put('/{id}', [AdController::class, 'update']);
    Route::delete('/{id}', [AdController::class, 'destroy']);
    Route::post('/{id}/images', [AdController::class, 'uploadImages']);
    Route::delete('/{id}/images/{imageId}', [AdController::class, 'deleteImage']);
    Route::post('/{id}/report', [App\Http\Controllers\Api\AdReportingController::class, 'store']);
    Route::get('/{id}/analytics', [AdController::class, 'getAnalytics']);
    Route::post('/{id}/boost', [AdController::class, 'boostAd']);
    Route::get('/{id}/recommendations', [AdController::class, 'getRecommendations']);
    Route::post('/{id}/contact', [App\Http\Controllers\Api\LeadController::class, 'contactSeller']);
    Route::post('/{id}/sold', [AdController::class, 'markAsSold']);
});

// Chat Routes
Route::middleware('auth:sanctum')->prefix('chat')->group(function () {
    Route::get('/', [ChatController::class, 'index']); // Get conversations list
    Route::get('/{userId}', [ChatController::class, 'show']); // Get messages with user
    Route::post('/{userId}/messages', [ChatController::class, 'sendMessage']); // Send message
    Route::put('/messages/{messageId}/read', [ChatController::class, 'markAsRead']); // Mark as read
    Route::delete('/messages/{messageId}', [ChatController::class, 'deleteMessage']); // Delete message
    
    // Firebase Chat Routes
    Route::post('/firebase/init', [ChatController::class, 'initFirebaseChat']); // Initialize Firebase chat
    Route::post('/firebase/send', [ChatController::class, 'sendFirebaseMessage']); // Send Firebase message
    Route::post('/firebase/typing', [ChatController::class, 'updateTypingStatus']); // Update typing status
});

// Support Routes
Route::prefix('support')->group(function () {
    Route::get('/settings', [App\Http\Controllers\Api\SupportController::class, 'getSettings']);
    Route::get('/faqs', [App\Http\Controllers\Api\SupportController::class, 'getFAQs']);
    Route::get('/categories', [App\Http\Controllers\Api\SupportController::class, 'getCategories']);
});

Route::middleware('auth:sanctum')->prefix('support')->group(function () {
    Route::post('/tickets', [App\Http\Controllers\Api\SupportController::class, 'createTicket']);
    Route::get('/tickets', [App\Http\Controllers\Api\SupportController::class, 'getMyTickets']);
    Route::get('/tickets/{id}', [App\Http\Controllers\Api\SupportController::class, 'getTicket']);
    Route::post('/tickets/{id}/reply', [App\Http\Controllers\Api\SupportController::class, 'replyToTicket']);
});

// Notification Routes
Route::middleware('auth:sanctum')->prefix('notifications')->group(function () {
    Route::get('/', [NotificationController::class, 'index']);
    Route::get('/stats', [NotificationController::class, 'getStats']);
    Route::get('/{id}', [NotificationController::class, 'show']);
    Route::post('/{id}/mark-read', [NotificationController::class, 'markAsRead']);
    Route::post('/{id}/mark-clicked', [NotificationController::class, 'markAsClicked']);
    Route::delete('/{id}', [NotificationController::class, 'delete']);
    Route::get('/preferences', [NotificationController::class, 'getPreferences']);
    Route::put('/preferences', [NotificationController::class, 'updatePreferences']);
    Route::post('/register-token', [NotificationController::class, 'registerDeviceToken']);
    Route::post('/test', [NotificationController::class, 'sendTestNotification']);
});

// Review Routes
Route::middleware('auth:sanctum')->prefix('reviews')->group(function () {
    Route::get('/received', [App\Http\Controllers\Api\ReviewController::class, 'getReceivedReviews']);
    Route::get('/given', [App\Http\Controllers\Api\ReviewController::class, 'getGivenReviews']);
    Route::get('/user/{userId}', [App\Http\Controllers\Api\ReviewController::class, 'getUserReviewStats']);
    Route::post('/', [App\Http\Controllers\Api\ReviewController::class, 'submitReview']);
    Route::put('/{id}', [App\Http\Controllers\Api\ReviewController::class, 'updateReview']);
    Route::delete('/{id}', [App\Http\Controllers\Api\ReviewController::class, 'deleteReview']);
});


// Payment Routes
Route::middleware('auth:sanctum')->prefix('payments')->group(function () {
    Route::post('/create-intent', [App\Http\Controllers\Api\PaymentController::class, 'createPaymentIntent']);
    Route::post('/create-subscription', [App\Http\Controllers\Api\PaymentController::class, 'createSubscription']);
    Route::get('/transactions', [App\Http\Controllers\Api\PaymentController::class, 'getTransactions']);
    Route::get('/transactions/{transactionId}', [App\Http\Controllers\Api\PaymentController::class, 'getTransaction']);
    Route::post('/transactions/{transactionId}/refund', [App\Http\Controllers\Api\PaymentController::class, 'requestRefund']);
});

// Banner Placement API (public - for displaying banners on different screens)
Route::prefix('banners')->group(function () {
    Route::get('/placement/{screenName}', [App\Http\Controllers\Api\BannerPlacementController::class, 'getBannersForPlacement']);
    Route::get('/placements/settings', [App\Http\Controllers\Api\BannerPlacementController::class, 'getPlacementSettings'])->middleware('auth:sanctum');
    Route::get('/placements/analytics', [App\Http\Controllers\Api\BannerPlacementController::class, 'getPlacementAnalytics'])->middleware('auth:sanctum');
});

// User Banner Management Routes
Route::middleware('auth:sanctum')->prefix('user/banners')->group(function () {
    Route::get('/packages', [App\Http\Controllers\Api\UserBannerController::class, 'getPackages']);
    Route::get('/', [App\Http\Controllers\Api\UserBannerController::class, 'index']);
    Route::post('/', [App\Http\Controllers\Api\UserBannerController::class, 'store']);
    Route::get('/{bannerId}/analytics', [App\Http\Controllers\Api\UserBannerController::class, 'analytics']);
    Route::get('/real-time-stats', [App\Http\Controllers\Api\UserBannerController::class, 'realTimeStats']);
    Route::put('/{bannerId}/pause', [App\Http\Controllers\Api\UserBannerController::class, 'pause']);
    Route::put('/{bannerId}/resume', [App\Http\Controllers\Api\UserBannerController::class, 'resume']);
    Route::delete('/{bannerId}', [App\Http\Controllers\Api\UserBannerController::class, 'destroy']);
});

// Payment Webhooks (no authentication - gateways will send webhooks)
Route::post('/webhooks/stripe', [App\Http\Controllers\Api\PaymentWebhookController::class, 'stripeWebhook']);
Route::post('/webhooks/razorpay', [App\Http\Controllers\Api\PaymentWebhookController::class, 'razorpayWebhook']);
Route::post('/webhooks/phonepe', [App\Http\Controllers\Api\PaymentWebhookController::class, 'phonepeWebhook']);
Route::post('/webhooks/paypal', [App\Http\Controllers\Api\PaymentWebhookController::class, 'paypalWebhook']);

// Analytics Routes
Route::middleware('auth:sanctum')->prefix('analytics')->group(function () {
    Route::get('/dashboard', [AnalyticsController::class, 'dashboard']);
    Route::get('/ad-post', [AnalyticsController::class, 'adPostAnalytics']);
    Route::get('/ad/{adId}', [App\Http\Controllers\Admin\IndividualAdAnalyticsController::class, 'getEnhancedAnalyticsData']);
    Route::get('/user/{userId}', [AnalyticsController::class, 'userAnalytics']);
    Route::get('/engagement', [AnalyticsController::class, 'engagementAnalytics']);
    
    // Business Intelligence Analytics
    Route::get('/business-intelligence', [AnalyticsController::class, 'businessIntelligence']);
    Route::get('/ai-insights', [AnalyticsController::class, 'aiInsights']);
    Route::get('/executive-summary', [AnalyticsController::class, 'executiveSummary']);
    Route::get('/export', [AnalyticsController::class, 'exportReport']);
    Route::get('/business-reports/geo', [AnalyticsController::class, 'geoDistributionReport']);
    
    // New VIP Analytics Dashboard Routes
    Route::get('/overview', [AnalyticsController::class, 'overview']); // User analytics overview
    Route::post('/compare', [AnalyticsController::class, 'compare']); // Compare ads
    Route::get('/realtime/overview', [AnalyticsController::class, 'realtimeOverview']); // User realtime overview (ALL ADS)
    Route::get('/ad/{adId}', [AnalyticsController::class, 'adAnalytics']); // Individual ad analytics
    Route::get('/realtime/{adId}', [AnalyticsController::class, 'realtimeAnalytics']); // Real-time analytics
    Route::get('/suggestions', [AnalyticsController::class, 'suggestions']); // AI suggestions for all ads
    Route::get('/ad/{adId}/suggestions', [AnalyticsController::class, 'adSuggestions']); // AI suggestions for specific ad
    Route::get('/packages', [AnalyticsController::class, 'packagePerformance']); // Package performance
    Route::get('/export/csv', [AnalyticsController::class, 'exportCSV']); // Export as CSV
    Route::get('/export/data', [AnalyticsController::class, 'exportData']); // Export data for PDF
});

// Auth Routes (Public)
Route::prefix('auth')->group(function () {
    Route::get('/config', [AuthController::class, 'config']); // Get auth config
    Route::post('/login', [AuthController::class, 'login']); // Email login
    Route::post('/register', [AuthController::class, 'register']); // Register
    Route::post('/google-login', [AuthController::class, 'googleLogin']); // Google login
    Route::post('/forgot-password', [AuthController::class, 'forgotPassword']);
    Route::post('/reset-password', [AuthController::class, 'resetPassword']);
});

// Auth Routes (Protected)
Route::middleware('auth:sanctum')->prefix('auth')->group(function () {
    Route::post('/logout', [AuthController::class, 'logout']);
    Route::post('/refresh', [AuthController::class, 'refresh']);
    Route::post('/change-password', [AuthController::class, 'changePassword']);
    Route::post('/update-profile', [AuthController::class, 'updateProfile']);
    Route::delete('/account', [AuthController::class, 'deleteAccount']);
    Route::post('/device-token', [AuthController::class, 'registerDeviceToken']);
});

// Admin Analytics Routes
Route::middleware(['auth:sanctum', 'admin'])->prefix('admin/analytics')->group(function () {
    Route::get('/business-intelligence', [App\Http\Controllers\Admin\AdvancedAnalyticsController::class, 'businessIntelligence']);
    Route::get('/ai-insights', [App\Http\Controllers\Admin\AdvancedAnalyticsController::class, 'aiInsights']);
    Route::get('/executive-summary', [App\Http\Controllers\Admin\AdvancedAnalyticsController::class, 'executiveSummary']);
    Route::get('/export', [App\Http\Controllers\Admin\AdvancedAnalyticsController::class, 'exportReport']);
});


// Admin API Routes
Route::middleware('auth:sanctum')->prefix('admin')->group(function () {
    // Plans API
    Route::prefix('plans')->group(function () {
        Route::get('/{plan}', [App\Http\Controllers\Admin\PlansController::class, 'apiShow']);
        Route::get('/{plan}/analytics', [App\Http\Controllers\Admin\PlansController::class, 'apiAnalytics']);
        Route::post('/{plan}/toggle-status', [App\Http\Controllers\Admin\PlansController::class, 'apiToggleStatus']);
        Route::post('/bulk-action', [App\Http\Controllers\Admin\PlansController::class, 'apiBulkAction']);
    });

    // User Management API
    Route::prefix('users')->group(function () {
        Route::get('/{user}/details', [App\Http\Controllers\Admin\UserManagementController::class, 'getUserDetails']);
        Route::post('/{user}/ban', [App\Http\Controllers\Admin\UserManagementController::class, 'banUser']);
        Route::post('/{user}/suspend', [App\Http\Controllers\Admin\UserManagementController::class, 'suspendUser']);
        Route::post('/{user}/verify', [App\Http\Controllers\Admin\UserManagementController::class, 'verifyUser']);
        Route::post('/{user}/unban', [App\Http\Controllers\Admin\UserManagementController::class, 'unbanUser']);
        Route::post('/{user}/unsuspend', [App\Http\Controllers\Admin\UserManagementController::class, 'unsuspendUser']);
    });

    // Safety Reports API
    Route::prefix('safety')->group(function () {
        Route::get('/ads/{ad}/reports', [App\Http\Controllers\Admin\UserManagementController::class, 'getAdSafetyReports']);
        Route::post('/appeals/{appeal}/handle', [App\Http\Controllers\Admin\UserManagementController::class, 'handleAppeal']);
    });
});

// Public routes (no authentication required)
Route::get('ads/public/{id}', [AdController::class, 'showPublic']);
Route::get('ads/public/search', [AdController::class, 'searchPublic']);
Route::get('categories/public', [CategoryController::class, 'indexPublic']);

// Pages API (public access for mobile app)
Route::get('pages', [App\Http\Controllers\Api\PageController::class, 'index']);
Route::get('pages/{slug}', [App\Http\Controllers\Api\PageController::class, 'show']);

// Blog API (public access for mobile app)
Route::get('blogs', [App\Http\Controllers\Api\BlogController::class, 'index']);
Route::get('blogs/{slug}', [App\Http\Controllers\Api\BlogController::class, 'show']);
Route::get('blog-categories', [App\Http\Controllers\Api\BlogController::class, 'categories']);
Route::post('blogs/{id}/view', [App\Http\Controllers\Api\BlogController::class, 'incrementViews'])->middleware('auth:sanctum');

// Banner Routes (Public - for display)
Route::prefix('banners')->group(function () {
    Route::get('/active', [App\Http\Controllers\Api\BannerDisplayController::class, 'getActiveBanners']);
    Route::post('/{id}/impression', [App\Http\Controllers\Api\BannerDisplayController::class, 'recordImpression']);
    Route::post('/{id}/click', [App\Http\Controllers\Api\BannerDisplayController::class, 'recordClick']);
});

// Monetization Config API (Public)
Route::get('/monetization/config', [App\Http\Controllers\Api\MonetizationController::class, 'getConfig']);
