@extends('admin.layouts.app')

@section('title', 'Create Advanced Notification')

@section('content')
    <div class="card">
        <div class="card-header">
            <h3 style="margin: 0; font-size: 1.25rem; color: var(--text-primary);">Create Advanced Notification</h3>
            <div style="font-size: 0.875rem; color: var(--text-secondary);">Create and schedule notifications with advanced targeting and A/B testing</div>
        </div>

        <div class="card-body">
            <form id="notificationForm" action="{{ route('notifications.store') }}" method="POST">
                @csrf

                <!-- Basic Information -->
                <div class="form-section">
                    <h4 style="margin-bottom: 16px; color: var(--text-primary);">Basic Information</h4>
                    
                    <div class="form-grid">
                        <div class="form-group">
                            <label for="title" class="form-label">Notification Title</label>
                            <input type="text" id="title" name="title" class="form-input" required maxlength="100" placeholder="Enter notification title">
                            <div class="form-hint">Max 100 characters</div>
                        </div>

                        <div class="form-group">
                            <label for="message" class="form-label">Notification Message</label>
                            <textarea id="message" name="message" class="form-input" rows="3" required maxlength="300" placeholder="Enter notification message"></textarea>
                            <div class="form-hint">Max 300 characters</div>
                        </div>

                        <div class="form-group">
                            <label for="priority" class="form-label">Priority Level</label>
                            <select id="priority" name="priority" class="form-input">
                                <option value="normal">Normal</option>
                                <option value="high">High</option>
                                <option value="urgent">Urgent</option>
                            </select>
                            <div class="form-hint">Higher priority notifications get more attention</div>
                        </div>

                        <div class="form-group">
                            <label for="category" class="form-label">Category</label>
                            <select id="category" name="category" class="form-input">
                                <option value="general">General</option>
                                <option value="promotions">Promotions</option>
                                <option value="security">Security</option>
                                <option value="updates">Updates</option>
                                <option value="reminders">Reminders</option>
                            </select>
                            <div class="form-hint">Used for notification channels and filtering</div>
                        </div>

                        <div class="form-group">
                            <label for="deep_link_url" class="form-label">Deep Link URL (Optional)</label>
                            <input type="url" id="deep_link_url" name="deep_link_url" class="form-input" placeholder="https://example.com/deeplink/...">
                            <div class="form-hint">URL to open when notification is tapped</div>
                        </div>
                    </div>
                </div>

                <!-- Targeting Options -->
                <div class="form-section">
                    <h4 style="margin-bottom: 16px; color: var(--text-primary);">Targeting Options</h4>
                    
                    <div class="form-group">
                        <label class="form-label">Target Type</label>
                        <div class="radio-group">
                            <label class="radio-option">
                                <input type="radio" name="target_type" value="all" checked>
                                <span>All Users</span>
                            </label>
                            <label class="radio-option">
                                <input type="radio" name="target_type" value="single_user">
                                <span>Specific Users</span>
                            </label>
                            <label class="radio-option">
                                <input type="radio" name="target_type" value="premium_plan">
                                <span>Premium Plan Users</span>
                            </label>
                            <label class="radio-option">
                                <input type="radio" name="target_type" value="custom">
                                <span>Custom Targeting</span>
                            </label>
                        </div>
                    </div>

                    <!-- Specific Users Targeting -->
                    <div id="specificUsersSection" class="form-group" style="display: none;">
                        <label for="user_ids" class="form-label">User IDs (comma-separated)</label>
                        <input type="text" id="user_ids" name="target_value[user_ids]" class="form-input" placeholder="1, 2, 3, 4">
                        <div class="form-hint">Enter user IDs separated by commas</div>
                    </div>

                    <!-- Premium Plan Targeting -->
                    <div id="premiumPlanSection" class="form-group" style="display: none;">
                        <label class="form-label">Select Plan Types</label>
                        <div class="checkbox-group">
                            <label class="checkbox-option">
                                <input type="checkbox" name="target_value[plan_types][]" value="standard">
                                <span>Standard (Free)</span>
                            </label>
                            <label class="checkbox-option">
                                <input type="checkbox" name="target_value[plan_types][]" value="basic">
                                <span>Basic</span>
                            </label>
                            <label class="checkbox-option">
                                <input type="checkbox" name="target_value[plan_types][]" value="premium">
                                <span>Premium</span>
                            </label>
                            <label class="checkbox-option">
                                <input type="checkbox" name="target_value[plan_types][]" value="enterprise">
                                <span>Enterprise</span>
                            </label>
                        </div>
                    </div>

                    <!-- Custom Targeting -->
                    <div id="customTargetingSection" class="form-group" style="display: none;">
                        <div class="alert alert-info">
                            <strong>Custom targeting requires additional configuration.</strong>
                            Please contact your system administrator for advanced targeting rules.
                        </div>
                    </div>
                </div>

                <!-- A/B Testing -->
                <div class="form-section">
                    <h4 style="margin-bottom: 16px; color: var(--text-primary);">A/B Testing</h4>
                    
                    <div class="form-group">
                        <label class="checkbox-option">
                            <input type="checkbox" id="ab_test_enabled" name="ab_test_enabled">
                            <span>Enable A/B Testing</span>
                        </label>
                        <div class="form-hint">Create multiple variants to test different messages</div>
                    </div>

                    <div id="abTestSection" style="display: none;">
                        <div class="form-grid">
                            <div class="form-group">
                                <label for="variant_title_1" class="form-label">Variant A Title</label>
                                <input type="text" id="variant_title_1" name="ab_test_variants[variant_a][title]" class="form-input" placeholder="Original title">
                            </div>
                            <div class="form-group">
                                <label for="variant_message_1" class="form-label">Variant A Message</label>
                                <textarea id="variant_message_1" name="ab_test_variants[variant_a][message]" class="form-input" rows="2" placeholder="Original message"></textarea>
                            </div>
                            <div class="form-group">
                                <label for="variant_title_2" class="form-label">Variant B Title</label>
                                <input type="text" id="variant_title_2" name="ab_test_variants[variant_b][title]" class="form-input" placeholder="Alternative title">
                            </div>
                            <div class="form-group">
                                <label for="variant_message_2" class="form-label">Variant B Message</label>
                                <textarea id="variant_message_2" name="ab_test_variants[variant_b][message]" class="form-input" rows="2" placeholder="Alternative message"></textarea>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Scheduling -->
                <div class="form-section">
                    <h4 style="margin-bottom: 16px; color: var(--text-primary);">Scheduling</h4>
                    
                    <div class="form-group">
                        <label class="radio-option">
                            <input type="radio" name="schedule_type" value="immediate" checked>
                            <span>Send Immediately</span>
                        </label>
                        <label class="radio-option">
                            <input type="radio" name="schedule_type" value="scheduled">
                            <span>Schedule for Later</span>
                        </label>
                    </div>

                    <div id="scheduleSection" class="form-group" style="display: none;">
                        <label for="scheduled_at" class="form-label">Scheduled Date & Time</label>
                        <input type="datetime-local" id="scheduled_at" name="scheduled_at" class="form-input">
                        <div class="form-hint">Select date and time for notification delivery</div>
                    </div>
                </div>

                <!-- Preview -->
                <div class="form-section">
                    <h4 style="margin-bottom: 16px; color: var(--text-primary);">Preview</h4>
                    
                    <div class="notification-preview">
                        <div class="preview-header">
                            <span>Mobile Notification Preview</span>
                            <button type="button" id="refreshPreview" class="btn-secondary">Refresh Preview</button>
                        </div>
                        <div class="preview-content">
                            <div class="preview-device">
                                <div class="preview-screen">
                                    <div class="preview-notification">
                                        <div class="preview-title" id="previewTitle">Notification Title</div>
                                        <div class="preview-message" id="previewMessage">Notification message goes here...</div>
                                        <div class="preview-meta">
                                            <span class="preview-time">Just now</span>
                                            <span class="preview-app">CLX Pro</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Actions -->
                <div class="form-actions">
                    <button type="submit" class="btn-primary">Create Notification</button>
                    <button type="button" id="saveDraft" class="btn-secondary">Save as Draft</button>
                    <button type="button" id="testNotification" class="btn-info">Test Notification</button>
                    <a href="{{ route('notifications.index') }}" class="btn-text">Cancel</a>
                </div>
            </form>
        </div>
    </div>

    <!-- Test Notification Modal -->
    <div id="testModal" class="modal" style="display: none;">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Test Notification</h3>
                <button onclick="closeTestModal()" class="modal-close">×</button>
            </div>
            <div class="modal-body">
                <p>Send a test notification to verify your configuration.</p>
                <div class="form-group">
                    <label for="testUser" class="form-label">Test User</label>
                    <select id="testUser" class="form-input">
                        <option value="">Select a user...</option>
                        <!-- Users will be loaded via JavaScript -->
                    </select>
                </div>
                <div class="form-actions">
                    <button onclick="sendTestNotification()" class="btn-primary">Send Test</button>
                    <button onclick="closeTestModal()" class="btn-secondary">Cancel</button>
                </div>
            </div>
        </div>
    </div>

    <style>
        .form-section {
            margin-bottom: 32px;
            padding-bottom: 24px;
            border-bottom: 1px solid var(--border-color);
        }

        .form-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 16px;
        }

        .form-group {
            margin-bottom: 16px;
        }

        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: var(--text-primary);
            font-size: 0.875rem;
        }

        .form-input {
            width: 100%;
            padding: 12px 16px;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            font-size: 1rem;
            transition: border-color 0.2s ease;
            background: white;
        }

        .form-input:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }

        .form-hint {
            font-size: 0.75rem;
            color: var(--text-secondary);
            margin-top: 4px;
        }

        .radio-group, .checkbox-group {
            display: flex;
            flex-wrap: wrap;
            gap: 16px;
        }

        .radio-option, .checkbox-option {
            display: flex;
            align-items: center;
            gap: 8px;
            cursor: pointer;
            padding: 8px 12px;
            border-radius: 6px;
            border: 1px solid var(--border-color);
            background: white;
            transition: all 0.2s ease;
        }

        .radio-option:hover, .checkbox-option:hover {
            border-color: var(--primary-color);
            background: #f8fafc;
        }

        .radio-option input, .checkbox-option input {
            margin: 0;
        }

        .alert {
            padding: 16px;
            border-radius: 8px;
            border: 1px solid;
            margin-bottom: 16px;
        }

        .alert-info {
            background-color: #e0f2fe;
            border-color: #0ea5e9;
            color: #0c4a6e;
        }

        .notification-preview {
            background: #f8fafc;
            border-radius: 12px;
            padding: 20px;
            border: 1px solid var(--border-color);
        }

        .preview-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 16px;
            font-weight: 600;
            color: var(--text-primary);
        }

        .preview-content {
            display: flex;
            justify-content: center;
        }

        .preview-device {
            width: 300px;
            height: 600px;
            background: #1f2937;
            border-radius: 24px;
            padding: 20px;
            box-shadow: 0 10px 25px rgba(0,0,0,0.2);
        }

        .preview-screen {
            background: white;
            border-radius: 16px;
            height: 100%;
            padding: 16px;
            position: relative;
        }

        .preview-notification {
            background: #f1f5f9;
            border-radius: 12px;
            padding: 16px;
            margin-top: 20px;
            border-left: 4px solid var(--primary-color);
        }

        .preview-title {
            font-weight: 700;
            color: #1e293b;
            margin-bottom: 8px;
            font-size: 1rem;
        }

        .preview-message {
            color: #475569;
            font-size: 0.875rem;
            line-height: 1.4;
        }

        .preview-meta {
            display: flex;
            justify-content: space-between;
            margin-top: 12px;
            font-size: 0.75rem;
            color: #64748b;
        }

        .form-actions {
            display: flex;
            gap: 12px;
            align-items: center;
            margin-top: 24px;
        }

        .btn-primary {
            background: var(--primary-color);
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.2s ease;
        }

        .btn-primary:hover {
            background: #2563eb;
        }

        .btn-secondary {
            background: #e2e8f0;
            color: #334155;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.2s ease;
        }

        .btn-secondary:hover {
            background: #cbd5e1;
        }

        .btn-info {
            background: #38bdf8;
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: background-color 0.2s ease;
        }

        .btn-info:hover {
            background: #0ea5e9;
        }

        .btn-text {
            color: var(--text-secondary);
            text-decoration: none;
            padding: 12px 24px;
            border-radius: 8px;
            font-weight: 600;
            transition: color 0.2s ease;
        }

        .btn-text:hover {
            color: var(--text-primary);
            background: #f8fafc;
        }

        /* Modal Styles */
        .modal {
            position: fixed;
            z-index: 1000;
            left: 0;
            top: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0,0,0,0.5);
            display: flex;
            justify-content: center;
            align-items: center;
        }

        .modal-content {
            background-color: white;
            border-radius: 12px;
            width: 500px;
            max-width: 90%;
            box-shadow: 0 20px 40px rgba(0,0,0,0.3);
        }

        .modal-header {
            padding: 20px 24px;
            border-bottom: 1px solid var(--border-color);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .modal-header h3 {
            margin: 0;
            color: var(--text-primary);
        }

        .modal-close {
            background: none;
            border: none;
            font-size: 24px;
            cursor: pointer;
            color: var(--text-secondary);
        }

        .modal-body {
            padding: 24px;
        }

        /* Responsive */
        @media (max-width: 768px) {
            .form-grid {
                grid-template-columns: 1fr;
            }
            
            .preview-device {
                width: 250px;
                height: 500px;
                padding: 16px;
            }
        }
    </style>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            setupFormInteractions();
            loadTestUsers();
        });

        function setupFormInteractions() {
            // Target type switching
            document.querySelectorAll('input[name="target_type"]').forEach(radio => {
                radio.addEventListener('change', function() {
                    document.getElementById('specificUsersSection').style.display = 'none';
                    document.getElementById('premiumPlanSection').style.display = 'none';
                    document.getElementById('customTargetingSection').style.display = 'none';

                    if (this.value === 'single_user') {
                        document.getElementById('specificUsersSection').style.display = 'block';
                    } else if (this.value === 'premium_plan') {
                        document.getElementById('premiumPlanSection').style.display = 'block';
                    } else if (this.value === 'custom') {
                        document.getElementById('customTargetingSection').style.display = 'block';
                    }
                });
            });

            // A/B testing toggle
            document.getElementById('ab_test_enabled').addEventListener('change', function() {
                document.getElementById('abTestSection').style.display = this.checked ? 'block' : 'none';
            });

            // Scheduling toggle
            document.querySelectorAll('input[name="schedule_type"]').forEach(radio => {
                radio.addEventListener('change', function() {
                    document.getElementById('scheduleSection').style.display = this.value === 'scheduled' ? 'block' : 'none';
                });
            });

            // Live preview updates
            ['title', 'message'].forEach(field => {
                document.getElementById(field).addEventListener('input', updatePreview);
            });

            document.getElementById('refreshPreview').addEventListener('click', updatePreview);

            // Form validation
            document.getElementById('notificationForm').addEventListener('submit', function(e) {
                if (!validateForm()) {
                    e.preventDefault();
                }
            });
        }

        function updatePreview() {
            const title = document.getElementById('title').value || 'Notification Title';
            const message = document.getElementById('message').value || 'Notification message goes here...';
            
            document.getElementById('previewTitle').textContent = title;
            document.getElementById('previewMessage').textContent = message;
        }

        function validateForm() {
            const title = document.getElementById('title').value.trim();
            const message = document.getElementById('message').value.trim();
            const targetType = document.querySelector('input[name="target_type"]:checked').value;

            if (title.length < 5) {
                alert('Title must be at least 5 characters long.');
                return false;
            }

            if (message.length < 10) {
                alert('Message must be at least 10 characters long.');
                return false;
            }

            if (targetType === 'single_user') {
                const userIds = document.getElementById('user_ids').value.trim();
                if (!userIds) {
                    alert('Please enter user IDs for specific user targeting.');
                    return false;
                }
            }

            if (targetType === 'premium_plan') {
                const planTypes = document.querySelectorAll('input[name="target_value[plan_types][]"]:checked');
                if (planTypes.length === 0) {
                    alert('Please select at least one plan type.');
                    return false;
                }
            }

            return true;
        }

        function loadTestUsers() {
            // This would typically load from an API endpoint
            // For now, we'll use a static list or leave it empty
            const select = document.getElementById('testUser');
            // Add some sample users for testing
            const users = [
                { id: 1, name: 'John Doe (Admin)' },
                { id: 2, name: 'Jane Smith (User)' },
                { id: 3, name: 'Bob Wilson (Premium)' }
            ];

            users.forEach(user => {
                const option = document.createElement('option');
                option.value = user.id;
                option.textContent = user.name;
                select.appendChild(option);
            });
        }

        function openTestModal() {
            document.getElementById('testModal').style.display = 'flex';
        }

        function closeTestModal() {
            document.getElementById('testModal').style.display = 'none';
        }

        function sendTestNotification() {
            const userId = document.getElementById('testUser').value;
            if (!userId) {
                alert('Please select a test user.');
                return;
            }

            // This would send a test notification via AJAX
            alert('Test notification sent to user ' + userId);
            closeTestModal();
        }

        // Firebase Configuration Status Check
        function checkFirebaseStatus() {
            // This would check Firebase configuration via AJAX
            // For now, we'll show a simple status
            console.log('Firebase status check would happen here');
        }
    </script>
@endsection
