@extends('admin.layouts.app')

@section('title', 'Ad Post Analytics')

@section('content')
    <!-- Ad Post Analytics Dashboard -->
    <div class="analytics-dashboard">
        <!-- Ad Performance Overview Section -->
        <div class="dashboard-grid">
            <!-- Total Ads -->
            <div class="card">
                <div class="card-header">
                    <div>
                        <div class="card-title">Total Ads</div>
                        <div class="card-value" id="total-ads">0</div>
                    </div>
                    <div class="stat-icon">📊</div>
                </div>
                <div class="card-change" id="ads-growth">+0% from last month</div>
            </div>

            <!-- Active Ads -->
            <div class="card">
                <div class="card-header">
                    <div>
                        <div class="card-title">Active Ads</div>
                        <div class="card-value" id="active-ads">0</div>
                    </div>
                    <div class="stat-icon">✅</div>
                </div>
                <div class="card-change" id="active-growth">+0% from last month</div>
            </div>

            <!-- Ad Views -->
            <div class="card">
                <div class="card-header">
                    <div>
                        <div class="card-title">Total Views</div>
                        <div class="card-value" id="total-views">0</div>
                    </div>
                    <div class="stat-icon">👁️</div>
                </div>
                <div class="card-change" id="views-growth">+0% from last month</div>
            </div>

            <!-- Ad CTR -->
            <div class="card">
                <div class="card-header">
                    <div>
                        <div class="card-title">Avg CTR</div>
                        <div class="card-value" id="avg-ctr">0%</div>
                    </div>
                    <div class="stat-icon">📈</div>
                </div>
                <div class="card-change" id="ctr-change">+0% from last month</div>
            </div>
        </div>

        <!-- Ad Engagement and Monetization Section -->
        <div class="content-grid">
            <!-- Ad Engagement Chart -->
            <div class="chart-card">
                <div class="card-header">
                    <h3 style="margin: 0; font-size: 1.1rem; color: var(--text-primary);">Ad Engagement Metrics</h3>
                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Messages, clicks, and interactions</div>
                </div>
                
                <div style="margin-top: 20px; height: 300px; position: relative;">
                    <canvas id="adEngagementChart" width="600" height="300"></canvas>
                </div>
            </div>

            <!-- Ad Monetization -->
            <div class="table-card">
                <div class="card-header">
                    <h3 style="margin: 0; font-size: 1.1rem; color: var(--text-primary);">Ad Monetization</h3>
                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Revenue from ad features</div>
                </div>
                
                <div style="margin-top: 20px;">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Feature</th>
                                <th>Revenue</th>
                                <th>Transactions</th>
                            </tr>
                        </thead>
                        <tbody id="monetization-table">
                            <!-- Data loaded via JavaScript -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Ad Conversion Funnels and Timing -->
        <div class="content-grid">
            <!-- Conversion Funnels -->
            <div class="chart-card">
                <div class="card-header">
                    <h3 style="margin: 0; font-size: 1.1rem; color: var(--text-primary);">Ad Conversion Funnels</h3>
                    <div style="font-size: 0.875rem; color: var(--text-secondary);">User journey analysis</div>
                </div>
                
                <div style="margin-top: 20px; height: 300px; position: relative;">
                    <canvas id="adConversionChart" width="600" height="300"></canvas>
                </div>
            </div>

            <!-- Optimal Posting Times -->
            <div class="table-card">
                <div class="card-header">
                    <h3 style="margin: 0; font-size: 1.1rem; color: var(--text-primary);">Optimal Posting Times</h3>
                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Peak engagement hours</div>
                </div>
                
                <div style="margin-top: 20px;">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Hour</th>
                                <th>Views</th>
                                <th>Best Time</th>
                            </tr>
                        </thead>
                        <tbody id="timing-table">
                            <!-- Data loaded via JavaScript -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Ad Category Performance and User Behavior -->
        <div class="content-grid">
            <!-- Category Performance -->
            <div class="chart-card">
                <div class="card-header">
                    <h3 style="margin: 0; font-size: 1.1rem; color: var(--text-primary);">Category Performance</h3>
                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Top performing categories</div>
                </div>
                
                <div style="margin-top: 20px; height: 300px; position: relative;">
                    <canvas id="categoryPerformanceChart" width="600" height="300"></canvas>
                </div>
            </div>

            <!-- User Behavior -->
            <div class="table-card">
                <div class="card-header">
                    <h3 style="margin: 0; font-size: 1.1rem; color: var(--text-primary);">User Behavior</h3>
                    <div style="font-size: 0.875rem; color: var(--text-secondary);">Buyer and seller patterns</div>
                </div>
                
                <div style="margin-top: 20px;">
                    <table class="table">
                        <thead>
                            <tr>
                                <th>Behavior</th>
                                <th>Value</th>
                                <th>Insight</th>
                            </tr>
                        </thead>
                        <tbody id="user-behavior-table">
                            <!-- Data loaded via JavaScript -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Ad Optimization Recommendations -->
        <div class="card" style="margin-bottom: 24px;">
            <div class="card-header">
                <h3 style="margin: 0; font-size: 1.1rem; color: var(--text-primary);">Ad Optimization Recommendations</h3>
                <div style="font-size: 0.875rem; color: var(--text-secondary);">AI-powered insights for better performance</div>
            </div>
            
            <div style="margin-top: 20px; display: grid; grid-template-columns: repeat(3, 1fr); gap: 20px;">
                <div class="recommendation-card">
                    <div style="font-weight: 600; color: var(--text-primary); margin-bottom: 8px;">High Priority</div>
                    <div style="color: var(--text-secondary); font-size: 0.875rem;" id="high-priority-recommendations">Loading...</div>
                </div>
                <div class="recommendation-card">
                    <div style="font-weight: 600; color: var(--text-primary); margin-bottom: 8px;">Medium Priority</div>
                    <div style="color: var(--text-secondary); font-size: 0.875rem;" id="medium-priority-recommendations">Loading...</div>
                </div>
                <div class="recommendation-card">
                    <div style="font-weight: 600; color: var(--text-primary); margin-bottom: 8px;">Low Priority</div>
                    <div style="color: var(--text-secondary); font-size: 0.875rem;" id="low-priority-recommendations">Loading...</div>
                </div>
            </div>
        </div>

        <!-- Ad Analytics Controls -->
        <div class="card" style="margin-bottom: 24px;">
            <div class="card-header">
                <h3 style="margin: 0; font-size: 1.1rem; color: var(--text-primary);">Ad Analytics Controls</h3>
                <div style="font-size: 0.875rem; color: var(--text-secondary);">Filter and export ad data</div>
            </div>
            
            <div style="margin-top: 20px; display: flex; gap: 20px; align-items: center; flex-wrap: wrap;">
                <div>
                    <label style="display: block; font-size: 0.875rem; color: var(--text-secondary); margin-bottom: 4px;">Start Date</label>
                    <input type="date" id="start-date" style="padding: 8px; border: 1px solid var(--border-color); border-radius: 4px; background: var(--card-bg); color: var(--text-primary);">
                </div>
                <div>
                    <label style="display: block; font-size: 0.875rem; color: var(--text-secondary); margin-bottom: 4px;">End Date</label>
                    <input type="date" id="end-date" style="padding: 8px; border: 1px solid var(--border-color); border-radius: 4px; background: var(--card-bg); color: var(--text-primary);">
                </div>
                <div>
                    <label style="display: block; font-size: 0.875rem; color: var(--text-secondary); margin-bottom: 4px;">Report Type</label>
                    <select id="report-type" style="padding: 8px; border: 1px solid var(--border-color); border-radius: 4px; background: var(--card-bg); color: var(--text-primary);">
                        <option value="comprehensive">Comprehensive</option>
                        <option value="performance">Performance Only</option>
                        <option value="monetization">Monetization Only</option>
                        <option value="engagement">Engagement Only</option>
                    </select>
                </div>
                <div style="margin-top: 24px;">
                    <button onclick="exportAdAnalytics()" style="padding: 10px 20px; background: var(--primary-color); color: white; border: none; border-radius: 6px; cursor: pointer; font-weight: 600;">Export Report</button>
                    <button onclick="refreshAdAnalytics()" style="padding: 10px 20px; background: var(--border-color); color: var(--text-primary); border: none; border-radius: 6px; cursor: pointer; font-weight: 600; margin-left: 10px;">Refresh Data</button>
                    <button onclick="getAdOptimizationRecommendations()" style="padding: 10px 20px; background: #10b981; color: white; border: none; border-radius: 6px; cursor: pointer; font-weight: 600; margin-left: 10px;">Get Recommendations</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Chart.js Library -->
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

    <!-- Ad Analytics JavaScript -->
    <script>
        // Chart instances
        let adEngagementChart, adConversionChart, categoryPerformanceChart;

        // Initialize dashboard
        document.addEventListener('DOMContentLoaded', function() {
            initializeCharts();
            loadAdAnalyticsData();
            
            // Set default date range
            const today = new Date();
            const lastMonth = new Date();
            lastMonth.setMonth(lastMonth.getMonth() - 1);
            
            document.getElementById('start-date').value = lastMonth.toISOString().split('T')[0];
            document.getElementById('end-date').value = today.toISOString().split('T')[0];
            
            // Add event listeners
            document.getElementById('start-date').addEventListener('change', loadAdAnalyticsData);
            document.getElementById('end-date').addEventListener('change', loadAdAnalyticsData);
            document.getElementById('report-type').addEventListener('change', loadAdAnalyticsData);
            
            // Add date range preset listeners
            setupDateRangePresets();
        });

        function setupDateRangePresets() {
            // Listen for date range presets
            const dateRangePresets = document.querySelectorAll('.date-range-preset, select[id*="date-range"], select[name*="date-range"]');
            dateRangePresets.forEach(preset => {
                preset.addEventListener('change', function() {
                    const presetValue = this.value;
                    const dates = getDateRangeFromPreset(presetValue);
                    
                    document.getElementById('start-date').value = dates.start;
                    document.getElementById('end-date').value = dates.end;
                    
                    loadAdAnalyticsData();
                });
            });
        }

        function getDateRangeFromPreset(preset) {
            const today = new Date();
            const startOfMonth = new Date(today.getFullYear(), today.getMonth(), 1);
            const endOfMonth = new Date(today.getFullYear(), today.getMonth() + 1, 0);
            
            switch(preset) {
                case 'today':
                    return {
                        start: today.toISOString().split('T')[0],
                        end: today.toISOString().split('T')[0]
                    };
                case 'yesterday':
                    const yesterday = new Date(today);
                    yesterday.setDate(yesterday.getDate() - 1);
                    return {
                        start: yesterday.toISOString().split('T')[0],
                        end: yesterday.toISOString().split('T')[0]
                    };
                case 'last-7-days':
                    const last7Days = new Date(today);
                    last7Days.setDate(last7Days.getDate() - 7);
                    return {
                        start: last7Days.toISOString().split('T')[0],
                        end: today.toISOString().split('T')[0]
                    };
                case 'last-30-days':
                    const last30Days = new Date(today);
                    last30Days.setDate(last30Days.getDate() - 30);
                    return {
                        start: last30Days.toISOString().split('T')[0],
                        end: today.toISOString().split('T')[0]
                    };
                case 'this-month':
                    return {
                        start: startOfMonth.toISOString().split('T')[0],
                        end: endOfMonth.toISOString().split('T')[0]
                    };
                case 'last-month':
                    const lastMonthStart = new Date(today.getFullYear(), today.getMonth() - 1, 1);
                    const lastMonthEnd = new Date(today.getFullYear(), today.getMonth(), 0);
                    return {
                        start: lastMonthStart.toISOString().split('T')[0],
                        end: lastMonthEnd.toISOString().split('T')[0]
                    };
                default:
                    return {
                        start: '',
                        end: ''
                    };
            }
        }

        function initializeCharts() {
            // Ad Engagement Chart
            const engagementCtx = document.getElementById('adEngagementChart').getContext('2d');
            adEngagementChart = new Chart(engagementCtx, {
                type: 'bar',
                data: {
                    labels: ['Messages', 'Clicks', 'Views', 'Shares'],
                    datasets: [{
                        label: 'Ad Engagement',
                        data: [0, 0, 0, 0],
                        backgroundColor: ['#7c3aed', '#10b981', '#f59e0b', '#ef4444']
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    }
                }
            });

            // Ad Conversion Chart
            const conversionCtx = document.getElementById('adConversionChart').getContext('2d');
            adConversionChart = new Chart(conversionCtx, {
                type: 'doughnut',
                data: {
                    labels: ['Post to View', 'View to Click', 'Click to Message', 'Message to Conversion'],
                    datasets: [{
                        data: [100, 80, 60, 40],
                        backgroundColor: ['#7c3aed', '#10b981', '#f59e0b', '#ef4444']
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            position: 'bottom'
                        }
                    }
                }
            });

            // Category Performance Chart
            const categoryCtx = document.getElementById('categoryPerformanceChart').getContext('2d');
            categoryPerformanceChart = new Chart(categoryCtx, {
                type: 'line',
                data: {
                    labels: [],
                    datasets: [{
                        label: 'Category Performance',
                        data: [],
                        borderColor: '#7c3aed',
                        backgroundColor: 'rgba(124, 58, 237, 0.1)',
                        borderWidth: 2,
                        fill: true
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    }
                }
            });
        }

        async function loadAdAnalyticsData() {
            const startDate = document.getElementById('start-date').value;
            const endDate = document.getElementById('end-date').value;
            const reportType = document.getElementById('report-type').value;

            try {
                // Load Ad Post Performance
                const response = await fetch(`/api/analytics/ad-post/performance?start_date=${startDate}&end_date=${endDate}`);
                const data = await response.json();

                if (data.success) {
                    updateAdPerformanceOverview(data.data.ad_performance_overview);
                    updateAdEngagementChart(data.data.ad_engagement_metrics);
                    updateAdMonetization(data.data.ad_monetization_analytics);
                    updateAdConversionFunnels(data.data.ad_conversion_funnels);
                    updateAdTimingOptimization(data.data.ad_timing_optimization);
                    updateAdCategoryPerformance(data.data.ad_category_performance);
                    updateAdUserBehavior(data.data.ad_user_behavior);
                    
                    // Update chart labels with selected date range
                    updateChartLabels(startDate, endDate);
                }

            } catch (error) {
                console.error('Error loading ad analytics data:', error);
                alert('Error loading ad analytics data. Please try again.');
            }
        }

        function updateChartLabels(startDate, endDate) {
            const dateRangeText = startDate && endDate ? `${startDate} to ${endDate}` : 'All Time';
            
            // Update chart titles
            const chartTitles = document.querySelectorAll('.card-header h3');
            chartTitles.forEach(title => {
                if (title.textContent.includes('Metrics') || title.textContent.includes('Analytics')) {
                    title.textContent = `${title.textContent.split(' (')[0]} (${dateRangeText})`;
                }
            });
        }

        function updateAdPerformanceOverview(overviewData) {
            document.getElementById('total-ads').textContent = overviewData.total_ads.toLocaleString();
            document.getElementById('active-ads').textContent = overviewData.active_ads.toLocaleString();
            document.getElementById('total-views').textContent = overviewData.total_views.toLocaleString();
            document.getElementById('avg-ctr').textContent = `${overviewData.ctr.toFixed(2)}%`;
            
            document.getElementById('ads-growth').textContent = `${overviewData.total_ads > 0 ? '+' : ''}${overviewData.total_ads}% from last month`;
            document.getElementById('active-growth').textContent = `${overviewData.active_ads > 0 ? '+' : ''}${overviewData.active_ads}% from last month`;
            document.getElementById('views-growth').textContent = `${overviewData.total_views > 0 ? '+' : ''}${overviewData.total_views}% from last month`;
            document.getElementById('ctr-change').textContent = `${overviewData.ctr > 0 ? '+' : ''}${overviewData.ctr.toFixed(1)}% from last month`;
        }

        function updateAdEngagementChart(engagementData) {
            if (engagementData.message_engagement) {
                const data = [
                    engagementData.message_engagement.total_messages || 0,
                    engagementData.call_to_action_performance.call_button_clicks || 0,
                    engagementData.message_engagement.total_messages || 0,
                    engagementData.ad_sharing_metrics || 0
                ];

                adEngagementChart.data.datasets[0].data = data;
                adEngagementChart.update();
            }
        }

        function updateAdMonetization(monetizationData) {
            const tableBody = document.getElementById('monetization-table');
            tableBody.innerHTML = '';

            if (monetizationData.ad_package_performance) {
                monetizationData.ad_package_performance.forEach(feature => {
                    const row = document.createElement('tr');
                    row.innerHTML = `
                        <td style="font-weight: 600;">${feature.transaction_type}</td>
                        <td>₹${feature.revenue || 0}</td>
                        <td>${feature.count || 0}</td>
                    `;
                    tableBody.appendChild(row);
                });
            }
        }

        function updateAdConversionFunnels(funnelData) {
            // Use actual funnel data from API response
            if (funnelData.conversion_funnels && funnelData.conversion_funnels.length > 0) {
                const data = funnelData.conversion_funnels;
                
                adConversionChart.data.datasets[0].data = data;
                adConversionChart.update();
            } else {
                // Fallback to dynamic data based on date range if no API data
                const startDate = document.getElementById('start-date').value;
                const endDate = document.getElementById('end-date').value;
                
                // Generate dynamic data based on date range
                const dateRange = new Date(endDate) - new Date(startDate);
                const days = Math.ceil(dateRange / (1000 * 60 * 60 * 24));
                
                // Adjust data based on time period
                let data;
                if (days <= 7) {
                    data = [100, 85, 70, 55]; // Higher conversion for short periods
                } else if (days <= 30) {
                    data = [100, 75, 60, 45]; // Medium conversion for monthly
                } else {
                    data = [100, 65, 50, 35]; // Lower conversion for long periods
                }

                adConversionChart.data.datasets[0].data = data;
                adConversionChart.update();
            }
        }

        function updateAdTimingOptimization(timingData) {
            const tableBody = document.getElementById('timing-table');
            tableBody.innerHTML = '';

            if (timingData.optimal_posting_times) {
                timingData.optimal_posting_times.forEach(time => {
                    const row = document.createElement('tr');
                    row.innerHTML = `
                        <td style="font-weight: 600;">${time.hour}:00</td>
                        <td>${time.views || 0}</td>
                        <td><span class="status-badge status-active">Peak</span></td>
                    `;
                    tableBody.appendChild(row);
                });
            }
        }

        function updateAdCategoryPerformance(categoryData) {
            if (categoryData.category_trends && categoryData.category_trends.length > 0) {
                const categories = categoryData.category_trends.map(cat => formatCategoryName(cat.category_id));
                const counts = categoryData.category_trends.map(cat => cat.count || 0);

                categoryPerformanceChart.data.labels = categories;
                categoryPerformanceChart.data.datasets[0].data = counts;
                categoryPerformanceChart.update();
            } else if (categoryData.category_popularity) {
                // Fallback to category_popularity if category_trends not available
                const categories = categoryData.category_popularity.map(cat => formatCategoryName(cat.category_id));
                const counts = categoryData.category_popularity.map(cat => cat.ad_count || 0);

                categoryPerformanceChart.data.labels = categories;
                categoryPerformanceChart.data.datasets[0].data = counts;
                categoryPerformanceChart.update();
            }
        }

        function updateAdUserBehavior(behaviorData) {
            const tableBody = document.getElementById('user-behavior-table');
            tableBody.innerHTML = '';

            if (behaviorData.feature_usage && behaviorData.feature_usage.length > 0) {
                // Use feature usage data from API response
                behaviorData.feature_usage.forEach(feature => {
                    const row = document.createElement('tr');
                    row.innerHTML = `
                        <td style="font-weight: 600;">${formatEventType(feature.event_type)}</td>
                        <td>${feature.count || 0}</td>
                        <td>Feature engagement</td>
                    `;
                    tableBody.appendChild(row);
                });
            } else if (behaviorData.buyer_behavior_patterns) {
                // Fallback to buyer behavior patterns if feature_usage not available
                const patterns = behaviorData.buyer_behavior_patterns;
                
                const rows = [
                    ['Avg Search Sessions', patterns.avg_search_sessions || 0, 'Session frequency'],
                    ['Avg Time per Session', `${patterns.avg_time_per_session || 0} min`, 'Engagement depth'],
                    ['Favorite Categories', patterns.favorite_categories?.length || 0, 'Category preference'],
                    ['Purchase Timing', patterns.purchase_timing?.length || 0, 'Buying patterns']
                ];

                rows.forEach(rowData => {
                    const row = document.createElement('tr');
                    row.innerHTML = `
                        <td style="font-weight: 600;">${rowData[0]}</td>
                        <td>${rowData[1]}</td>
                        <td>${rowData[2]}</td>
                    `;
                    tableBody.appendChild(row);
                });
            }
        }

        function formatEventType(eventType) {
            const eventMap = {
                'ad_view': 'Ad Views',
                'ad_click': 'Ad Clicks',
                'ad_message': 'Ad Messages',
                'ad_share': 'Ad Shares',
                'ad_save': 'Ad Saves',
                'ad_report': 'Ad Reports',
                'ad_purchase': 'Ad Purchases',
                'ad_search': 'Ad Searches',
                'ad_filter': 'Ad Filters',
                'ad_sort': 'Ad Sorts'
            };
            return eventMap[eventType] || eventType.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
        }

        function formatCategoryName(categoryId) {
            const categoryMap = {
                'electronics': 'Electronics',
                'vehicles': 'Vehicles',
                'real_estate': 'Real Estate',
                'fashion': 'Fashion',
                'home_garden': 'Home & Garden',
                'services': 'Services',
                'jobs': 'Jobs',
                'pets': 'Pets',
                'sports': 'Sports',
                'hobbies': 'Hobbies'
            };
            return categoryMap[categoryId] || categoryId.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
        }

        async function getAdOptimizationRecommendations() {
            const startDate = document.getElementById('start-date').value;
            const endDate = document.getElementById('end-date').value;

            try {
                const response = await fetch(`/api/analytics/ad-post/recommendations?start_date=${startDate}&end_date=${endDate}`);
                const data = await response.json();

                if (data.success) {
                    updateRecommendations(data.data);
                }
            } catch (error) {
                console.error('Error loading recommendations:', error);
                alert('Error loading recommendations. Please try again.');
            }
        }

        function updateRecommendations(recommendations) {
            const highPriority = document.getElementById('high-priority-recommendations');
            const mediumPriority = document.getElementById('medium-priority-recommendations');
            const lowPriority = document.getElementById('low-priority-recommendations');

            if (recommendations.high_priority) {
                highPriority.innerHTML = recommendations.high_priority.map(rec => 
                    `<div style="margin-bottom: 8px; padding: 8px; background: #fef3c7; border-radius: 4px;">${rec.title}: ${rec.description}</div>`
                ).join('');
            }

            if (recommendations.medium_priority) {
                mediumPriority.innerHTML = recommendations.medium_priority.map(rec => 
                    `<div style="margin-bottom: 8px; padding: 8px; background: #e0f2fe; border-radius: 4px;">${rec.title}: ${rec.description}</div>`
                ).join('');
            }

            if (recommendations.low_priority) {
                lowPriority.innerHTML = recommendations.low_priority.map(rec => 
                    `<div style="margin-bottom: 8px; padding: 8px; background: #dcfce7; border-radius: 4px;">${rec.title}: ${rec.description}</div>`
                ).join('');
            }
        }

        async function exportAdAnalytics() {
            const startDate = document.getElementById('start-date').value;
            const endDate = document.getElementById('end-date').value;
            const reportType = document.getElementById('report-type').value;

            try {
                const response = await fetch('/api/analytics/ad-post/export', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        start_date: startDate,
                        end_date: endDate,
                        format: 'pdf',
                        report_type: reportType
                    })
                });

                if (response.ok) {
                    alert('Ad analytics report exported successfully!');
                } else {
                    alert('Error exporting ad analytics report. Please try again.');
                }
            } catch (error) {
                console.error('Error exporting ad analytics:', error);
                alert('Error exporting ad analytics report. Please try again.');
            }
        }

        function refreshAdAnalytics() {
            loadAdAnalyticsData();
        }
    </script>

    <style>
        .recommendation-card {
            background: var(--bg-color);
            padding: 16px;
            border-radius: 8px;
            border: 1px solid var(--border-color);
        }

        .ad-analytics-dashboard {
            animation: fadeIn 0.5s ease-in;
        }

        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
    </style>
@endsection
