<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\Ad;
use App\Models\Category;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class RealisticAdsSeeder extends Seeder
{
    public function run()
    {
        // 0. Safety Check: If no categories exist, we cannot seed ads.
        if (\App\Models\Category::count() === 0) {
            $this->command->info('No categories found. Skipping Realistic Ads seeding.');
            return;
        }

        // 1. Create Users with different Tiers
        $vipProUser = User::firstOrCreate(
            ['email' => 'pro_seller@clx.com'],
            [
                'name' => 'Apex Motors',
                'password' => Hash::make('password'),
                'membership_type' => 'VIP Pro',
                'email_verified_at' => now(),
            ]
        );

        $vipUser = User::firstOrCreate(
            ['email' => 'vip_seller@clx.com'],
            [
                'name' => 'Elite Gadgets',
                'password' => Hash::make('password'),
                'membership_type' => 'VIP', 
                'email_verified_at' => now(),
            ]
        );

        $standardUser = User::firstOrCreate(
            ['email' => 'john_doe@clx.com'],
            [
                'name' => 'John Doe',
                'password' => Hash::make('password'),
                'membership_type' => 'Standard',
                'email_verified_at' => now(),
            ]
        );

        // Get Categories
        $carsCat = Category::where('slug', 'cars')->first() ?? Category::first();
        $electronicsCat = Category::where('slug', 'electronics')->first() ?? Category::first();
        $furnitureCat = Category::where('slug', 'furniture')->first() ?? Category::first();
        $propertiesCat = Category::where('slug', 'properties')->first() ?? Category::first();

        // 2. Create Ads (Expanded to 50 items)
        $count = 50;
        $faker = \Faker\Factory::create();
        
        $titles = [
            'cars' => ['Tesla Model S', 'BMW M4', 'Audi RS7', 'Mercedes AMG', 'Porsche 911', 'Ford Mustang', 'Chevrolet Camaro', 'Toyota Supra', 'Honda Civic Type R', 'Lamborghini Huracan'],
            'electronics' => ['iPhone 15 Pro', 'Samsung S24 Ultra', 'MacBook Pro M3', 'Sony A7IV', 'Dell XPS 15', 'iPad Pro 12.9', 'Nvidia RTX 4090', 'PlayStation 5', 'Xbox Series X', 'Canon R5'],
            'properties' => ['Luxury Penthouse', 'Modern Villa', 'Cozy Apartment', 'Beach House', 'City Studio', 'Suburban Home', 'Mountain Cabin', 'Lake Cottage', 'Downtown Loft', 'Historic Manor'],
            'furniture' => ['Modern Sofa', 'Oak Dining Table', 'King Size Bed', 'Ergonomic Chair', 'Bookshelf', 'Coffee Table', 'Wardrobe', 'TV Stand', 'Recliner', 'Office Desk']
        ];

        $images = [
            'cars' => [
                'https://images.unsplash.com/photo-1617788138017-80ad40651399?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1503376763036-066120622c74?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1552519507-da3b142c6e3d?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1583121274602-3e2820c698d9?auto=format&fit=crop&q=80&w=800'
            ],
            'electronics' => [
                'https://images.unsplash.com/photo-1517336714731-489689fd1ca4?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1516035069371-29a1b244cc32?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1593341646261-4fa3198031f3?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1511384065330-6306a4aaf4a6?auto=format&fit=crop&q=80&w=800'
            ],
            'properties' => [
                'https://images.unsplash.com/photo-1512917774080-9991f1c4c750?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1600596542815-e32870110029?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1600585154340-be6161a56a0c?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1600607687939-ce8a6c25118c?auto=format&fit=crop&q=80&w=800'
            ],
            'furniture' => [
                'https://images.unsplash.com/photo-1555041469-a586c61ea9bc?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1577140917170-285929d5575c?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1524758631624-e2822e304c36?auto=format&fit=crop&q=80&w=800',
                'https://images.unsplash.com/photo-1586023492125-27b2c045efd7?auto=format&fit=crop&q=80&w=800'
            ]
        ];

        $locations = [
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'Indiranagar', 'lat' => 12.9719, 'lng' => 77.6412],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'Koramangala', 'lat' => 12.9352, 'lng' => 77.6245],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'Whitefield', 'lat' => 12.9698, 'lng' => 77.7500],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'Jayanagar', 'lat' => 12.9250, 'lng' => 77.5938],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'HSR Layout', 'lat' => 12.9121, 'lng' => 77.6446],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'MG Road', 'lat' => 12.9766, 'lng' => 77.5993],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'Electronic City', 'lat' => 12.8452, 'lng' => 77.6602],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'Hebbal', 'lat' => 13.0354, 'lng' => 77.5988],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'Yelahanka', 'lat' => 13.1007, 'lng' => 77.5963],
            ['city' => 'Bangalore', 'state' => 'Karnataka', 'area' => 'Malleshwaram', 'lat' => 13.0031, 'lng' => 77.5643]
        ];

        for ($i = 0; $i < $count; $i++) {
            // Pick random category type
            $type = $faker->randomElement(['cars', 'electronics', 'properties', 'furniture']);
            
            // Resolve Category ID
            $catId = match($type) {
                'cars' => $carsCat->id,
                'electronics' => $electronicsCat->id,
                'properties' => $propertiesCat->id,
                'furniture' => $furnitureCat->id,
            };

            // Pick User (Weighted towards VIPs)
            $user = $faker->randomElement([$vipProUser, $vipProUser, $vipUser, $standardUser]);
            
            // Pick Location
            $location = $faker->randomElement($locations);

            $ad = Ad::create([
                'user_id' => $user->id,
                'category_id' => $catId,
                'title' => $faker->randomElement($titles[$type]) . ' - ' . $faker->words(3, true),
                'description' => $faker->paragraph(3),
                'price' => $faker->randomFloat(2, 50, 500000),
                'currency' => 'INR', // Changed to INR for Bangalore context
                'city' => $location['city'],
                'state' => $location['state'],
                'location_address' => $location['area'],
                'latitude' => $location['lat'] + ($faker->randomFloat(6, -0.01, 0.01)), // Add slight jitter
                'longitude' => $location['lng'] + ($faker->randomFloat(6, -0.01, 0.01)),
                'country' => 'India',
                'status' => 'Active',
                'is_active' => true,
                'is_featured' => $faker->boolean(20), // 20% chance of being featured
                'view_count' => rand(10, 5000),
            ]);

            // Add 2-10 Images
            $numImages = rand(2, 10);
            $catImages = $images[$type];
            // Ensure we have enough unique images or reuse them
            
            for ($j = 0; $j < $numImages; $j++) {
                // Loop through available images if we need more than available
                $imgUrl = $catImages[$j % count($catImages)];
                
                \DB::table('ad_images')->insert([
                    'ad_id' => $ad->id,
                    'image_url' => $imgUrl,
                    'is_primary' => ($j === 0),
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }
        }
    }
}
