<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class InitialDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     *
     * @return void
     */
    public function run()
    {
        // Create default user plans
        $this->createUserPlans();
        
        // Create default categories
        $this->createCategories();
        
        // Create admin user
        $this->createAdminUser();
        
        // Create default settings
        $this->createSettings();
        
        // Create payment gateways
        $this->createPaymentGateways();
        
        // Create ad packages
        $this->createAdPackages();
    }

    private function createUserPlans()
    {
        $plans = [
            [
                'plan_name' => 'Standard',
                'price' => 0.00,
                'price_multiplier' => 1.0,
                'duration_days' => 30,
                'city_multiplier' => 1.0,
                'post_limit' => 5,
                'lead_limit' => 10,
                'analytics_tier' => 'Basic',
                'validity_days' => 30,
                'ad_free' => false,
                'has_analytics' => false,
                'verified_badge' => false,
                'badge_style' => null,
                'theme' => 'light',
                'halo_effect' => null,
                'ad_border' => null,
                'status' => 'Active',
            ],
            [
                'plan_name' => 'VIP Buyer',
                'price' => 9.99,
                'price_multiplier' => 1.0,
                'duration_days' => 30,
                'city_multiplier' => 1.0,
                'post_limit' => null,
                'lead_limit' => 50,
                'analytics_tier' => 'Advanced',
                'validity_days' => 30,
                'ad_free' => true,
                'has_analytics' => true,
                'verified_badge' => true,
                'badge_style' => 'gold',
                'theme' => 'premium',
                'halo_effect' => 'blue',
                'ad_border' => 'gold',
                'status' => 'Active',
            ],
            [
                'plan_name' => 'VIP Seller',
                'price' => 19.99,
                'price_multiplier' => 1.0,
                'duration_days' => 30,
                'city_multiplier' => 1.0,
                'post_limit' => 50,
                'lead_limit' => 100,
                'analytics_tier' => 'Advanced',
                'validity_days' => 30,
                'ad_free' => true,
                'has_analytics' => true,
                'verified_badge' => true,
                'badge_style' => 'silver',
                'theme' => 'premium',
                'halo_effect' => 'green',
                'ad_border' => 'silver',
                'status' => 'Active',
            ],
            [
                'plan_name' => 'VIP Pro',
                'price' => 29.99,
                'price_multiplier' => 1.0,
                'duration_days' => 30,
                'city_multiplier' => 1.0,
                'post_limit' => null,
                'lead_limit' => null,
                'analytics_tier' => 'Premium',
                'validity_days' => 30,
                'ad_free' => true,
                'has_analytics' => true,
                'verified_badge' => true,
                'badge_style' => 'platinum',
                'theme' => 'premium',
                'halo_effect' => 'purple',
                'ad_border' => 'platinum',
                'status' => 'Active',
            ],
        ];

        foreach ($plans as $plan) {
            DB::table('user_plans')->insert($plan);
        }
    }

    private function createCategories()
    {
        $categories = [
            [
                'name' => 'Real Estate',
                'slug' => 'real-estate',
                'icon' => '🏠',
            ],
            [
                'name' => 'Vehicles',
                'slug' => 'vehicles',
                'icon' => '🚗',
            ],
            [
                'name' => 'Electronics',
                'slug' => 'electronics',
                'icon' => '📱',
            ],
            [
                'name' => 'Jobs',
                'slug' => 'jobs',
                'icon' => '💼',
            ],
            [
                'name' => 'Services',
                'slug' => 'services',
                'icon' => '🔧',
            ],
            [
                'name' => 'Fashion',
                'slug' => 'fashion',
                'icon' => '👗',
            ],
            [
                'name' => 'Home & Garden',
                'slug' => 'home-garden',
                'icon' => '🏡',
            ],
            [
                'name' => 'Sports & Hobbies',
                'slug' => 'sports-hobbies',
                'icon' => '⚽',
            ],
        ];

        foreach ($categories as $category) {
            $categoryId = DB::table('categories')->insertGetId($category);
            
            // Create subcategories for some categories
            $this->createSubcategories($categoryId, $category['slug']);
        }
    }

    private function createSubcategories($categoryId, $categorySlug)
    {
        $subcategories = [];
        
        switch ($categorySlug) {
            case 'real-estate':
                $subcategories = [
                    ['name' => 'Apartments', 'slug' => 'apartments', 'icon' => '🏢'],
                    ['name' => 'Houses', 'slug' => 'houses', 'icon' => '🏠'],
                    ['name' => 'Land', 'slug' => 'land', 'icon' => '🌳'],
                    ['name' => 'Commercial', 'slug' => 'commercial', 'icon' => '🏢'],
                ];
                break;
                
            case 'vehicles':
                $subcategories = [
                    ['name' => 'Cars', 'slug' => 'cars', 'icon' => '🚗'],
                    ['name' => 'Motorcycles', 'slug' => 'motorcycles', 'icon' => '🏍️'],
                    ['name' => 'Bikes', 'slug' => 'bikes', 'icon' => '🚲'],
                    ['name' => 'Accessories', 'slug' => 'vehicle-accessories', 'icon' => '🔧'],
                ];
                break;
                
            case 'electronics':
                $subcategories = [
                    ['name' => 'Mobile Phones', 'slug' => 'mobile-phones', 'icon' => '📱'],
                    ['name' => 'Laptops', 'slug' => 'laptops', 'icon' => '💻'],
                    ['name' => 'TV & Audio', 'slug' => 'tv-audio', 'icon' => '📺'],
                    ['name' => 'Gaming', 'slug' => 'gaming', 'icon' => '🎮'],
                ];
                break;
        }

        foreach ($subcategories as $subcategory) {
            $subcategory['category_id'] = $categoryId;
            DB::table('subcategories')->insert($subcategory);
        }
    }

    private function createAdminUser()
    {
        DB::table('users')->insert([
            'name' => 'Admin User',
            'email' => 'admin@clx.com',
            'password' => Hash::make('admin123'),
            'is_admin' => true,
            'email_verified_at' => now(),
            'created_at' => now(),
            'updated_at' => now(),
        ]);
    }

    private function createSettings()
    {
        $settings = [
            [
                'key' => 'allow_free_ad_posting',
                'value' => 'true',
                'description' => 'Allow free ad posting or restrict to paid only',
            ],
            [
                'key' => 'map_provider',
                'value' => 'OpenStreetMap',
                'description' => 'Current map provider (OpenStreetMap or GoogleMaps)',
            ],
            [
                'key' => 'google_maps_api_key',
                'value' => null,
                'description' => 'Google Maps API Key if Google Maps is selected',
            ],
            [
                'key' => 'blue_tick_paid_feature',
                'value' => 'false',
                'description' => 'Is Blue Tick verification a paid feature?',
            ],
            [
                'key' => 'default_currency',
                'value' => 'INR',
                'description' => 'Default currency for the platform',
            ],
            [
                'key' => 'max_images_per_ad',
                'value' => '5',
                'description' => 'Maximum number of images allowed per ad',
            ],
            [
                'key' => 'ad_approval_required',
                'value' => 'false',
                'description' => 'Require admin approval for ads',
            ],
        ];

        foreach ($settings as $setting) {
            DB::table('settings')->insert($setting);
        }
    }

    private function createPaymentGateways()
    {
        $gateways = [
            [
                'name' => 'Razorpay',
                'slug' => 'razorpay',
                'api_key' => env('RAZORPAY_KEY_ID', 'rzp_test_your_key_id'),
                'api_secret' => env('RAZORPAY_KEY_SECRET', 'rzp_test_your_key_secret'),
                'webhook_secret' => env('RAZORPAY_WEBHOOK_SECRET', 'your_webhook_secret'),
                'is_active' => false, // Set to true when you have real credentials
                'is_test_mode' => true,
                'supported_currencies' => json_encode(['INR']),
                'supported_methods' => json_encode(['credit_card', 'debit_card', 'upi', 'net_banking', 'wallet']),
                'transaction_fee_percent' => 2.00,
                'transaction_fee_fixed' => 0.00,
                'description' => 'Razorpay payment gateway for Indian payments',
                'sort_order' => 1,
            ],
            [
                'name' => 'PhonePe',
                'slug' => 'phonepe',
                'api_key' => env('PHONEPE_MERCHANT_ID', 'your_merchant_id'),
                'api_secret' => env('PHONEPE_SALT_KEY', 'your_salt_key'),
                'webhook_secret' => env('PHONEPE_SALT_INDEX', '1'),
                'is_active' => false, // Set to true when you have real credentials
                'is_test_mode' => true,
                'supported_currencies' => json_encode(['INR']),
                'supported_methods' => json_encode(['upi', 'credit_card', 'debit_card', 'net_banking']),
                'transaction_fee_percent' => 1.99,
                'transaction_fee_fixed' => 0.00,
                'description' => 'PhonePe payment gateway for UPI and card payments',
                'sort_order' => 2,
            ],
            [
                'name' => 'Stripe',
                'slug' => 'stripe',
                'api_key' => env('STRIPE_PUBLISHABLE_KEY', 'pk_test_your_key'),
                'api_secret' => env('STRIPE_SECRET_KEY', 'sk_test_your_key'),
                'webhook_secret' => env('STRIPE_WEBHOOK_SECRET', 'whsec_your_webhook_secret'),
                'is_active' => false, // Set to true when you have real credentials
                'is_test_mode' => true,
                'supported_currencies' => json_encode(['USD', 'EUR', 'GBP', 'INR']),
                'supported_methods' => json_encode(['credit_card', 'debit_card', 'apple_pay', 'google_pay']),
                'transaction_fee_percent' => 2.90,
                'transaction_fee_fixed' => 0.30,
                'description' => 'Stripe payment gateway for international payments',
                'sort_order' => 3,
            ],
            [
                'name' => 'PayPal',
                'slug' => 'paypal',
                'api_key' => env('PAYPAL_CLIENT_ID', 'your_client_id'),
                'api_secret' => env('PAYPAL_CLIENT_SECRET', 'your_client_secret'),
                'webhook_secret' => null,
                'is_active' => false, // Set to true when you have real credentials
                'is_test_mode' => true,
                'supported_currencies' => json_encode(['USD', 'EUR', 'GBP', 'INR']),
                'supported_methods' => json_encode(['paypal', 'credit_card']),
                'transaction_fee_percent' => 3.50,
                'transaction_fee_fixed' => 0.30,
                'description' => 'PayPal payment gateway for international payments',
                'sort_order' => 4,
            ],
        ];

        foreach ($gateways as $gateway) {
            $gatewayId = DB::table('payment_gateways')->insertGetId($gateway);
            
            // Create payment methods for each gateway
            $this->createPaymentMethods($gatewayId, $gateway['slug']);
        }
    }

    private function createPaymentMethods($gatewayId, $gatewaySlug)
    {
        $methods = [];

        switch ($gatewaySlug) {
            case 'razorpay':
                $methods = [
                    ['name' => 'Credit Card', 'slug' => 'credit_card', 'icon' => '💳', 'sort_order' => 1],
                    ['name' => 'Debit Card', 'slug' => 'debit_card', 'icon' => '💳', 'sort_order' => 2],
                    ['name' => 'UPI', 'slug' => 'upi', 'icon' => '📱', 'sort_order' => 3],
                    ['name' => 'Net Banking', 'slug' => 'net_banking', 'icon' => '🏦', 'sort_order' => 4],
                    ['name' => 'Wallet', 'slug' => 'wallet', 'icon' => '💰', 'sort_order' => 5],
                ];
                break;

            case 'phonepe':
                $methods = [
                    ['name' => 'UPI', 'slug' => 'upi', 'icon' => '📱', 'sort_order' => 1],
                    ['name' => 'Credit Card', 'slug' => 'credit_card', 'icon' => '💳', 'sort_order' => 2],
                    ['name' => 'Debit Card', 'slug' => 'debit_card', 'icon' => '💳', 'sort_order' => 3],
                    ['name' => 'Net Banking', 'slug' => 'net_banking', 'icon' => '🏦', 'sort_order' => 4],
                ];
                break;

            case 'stripe':
                $methods = [
                    ['name' => 'Credit Card', 'slug' => 'credit_card', 'icon' => '💳', 'sort_order' => 1],
                    ['name' => 'Debit Card', 'slug' => 'debit_card', 'icon' => '💳', 'sort_order' => 2],
                    ['name' => 'Apple Pay', 'slug' => 'apple_pay', 'icon' => '🍎', 'sort_order' => 3],
                    ['name' => 'Google Pay', 'slug' => 'google_pay', 'icon' => 'G', 'sort_order' => 4],
                ];
                break;

            case 'paypal':
                $methods = [
                    ['name' => 'PayPal', 'slug' => 'paypal', 'icon' => '🅿️', 'sort_order' => 1],
                    ['name' => 'Credit Card', 'slug' => 'credit_card', 'icon' => '💳', 'sort_order' => 2],
                ];
                break;
        }

        foreach ($methods as $method) {
            $method['gateway_slug'] = $gatewaySlug;
            DB::table('payment_methods')->insert($method);
        }
    }

    private function createAdPackages()
    {
        $packages = [
            [
                'name' => 'Featured Ad',
                'slug' => 'featured_ad',
                'description' => 'Get your ad featured on the top of search results for 7 days',
                'price' => 99.00,
                'currency' => 'INR',
                'duration_days' => 7,
                'free_trial_available' => false,
                'free_trial_days' => 0,
                'features' => json_encode([
                    'featured' => true,
                    'boosted' => false,
                    'bumped' => false,
                    'priority' => true
                ]),
                'is_active' => true,
                'sort_order' => 1,
            ],
            [
                'name' => 'Boost Ad',
                'slug' => 'boost_ad',
                'description' => 'Boost your ad visibility for 3 days',
                'price' => 49.00,
                'currency' => 'INR',
                'duration_days' => 3,
                'free_trial_available' => false,
                'free_trial_days' => 0,
                'features' => json_encode([
                    'featured' => false,
                    'boosted' => true,
                    'bumped' => false,
                    'priority' => false
                ]),
                'is_active' => true,
                'sort_order' => 2,
            ],
            [
                'name' => 'Bump Up Ad',
                'slug' => 'bump_up_ad',
                'description' => 'Bump your ad to the top of the list once',
                'price' => 19.00,
                'currency' => 'INR',
                'duration_days' => 1,
                'free_trial_available' => false,
                'free_trial_days' => 0,
                'features' => json_encode([
                    'featured' => false,
                    'boosted' => false,
                    'bumped' => true,
                    'priority' => false
                ]),
                'is_active' => true,
                'sort_order' => 3,
            ],
            [
                'name' => 'Premium Package',
                'slug' => 'premium_package',
                'description' => 'Get all premium features for 30 days',
                'price' => 299.00,
                'currency' => 'INR',
                'duration_days' => 30,
                'free_trial_available' => true,
                'free_trial_days' => 7,
                'features' => json_encode([
                    'featured' => true,
                    'boosted' => true,
                    'bumped' => true,
                    'priority' => true
                ]),
                'is_active' => true,
                'sort_order' => 4,
            ],
        ];

        foreach ($packages as $package) {
            DB::table('ad_packages')->insert($package);
        }
    }
}
