<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\GlobalField;
use App\Models\CategoryField;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

class GlobalFieldsSeeder extends Seeder
{
    public function run(): void
    {
        Schema::disableForeignKeyConstraints();
        
        $this->command->info('Cleaning up custom fields...');
        CategoryField::truncate();
        GlobalField::truncate();
        
        // SQLite sequence reset if needed
        if (DB::getDriverName() === 'sqlite') {
            DB::update("DELETE FROM sqlite_sequence WHERE name = 'category_fields'");
            DB::update("DELETE FROM sqlite_sequence WHERE name = 'global_fields'");
        }
        
        Schema::enableForeignKeyConstraints();

        $this->command->info('Creating Global Fields Library...');
        
        // --- 1. Mobile & Electronics Fields ---
        $brand = $this->createGlobal('Brand', 'brand', 'dropdown', ['Apple', 'Samsung', 'Xiaomi', 'Vivo', 'Oppo', 'OnePlus', 'Google', 'Realme', 'Sony', 'Other'], '🏷️');
        $ram = $this->createGlobal('RAM', 'ram', 'dropdown', ['1GB', '2GB', '3GB', '4GB', '6GB', '8GB', '12GB', '16GB+', 'Others'], '💾');
        $storage = $this->createGlobal('Storage', 'storage', 'dropdown', ['16GB', '32GB', '64GB', '128GB', '256GB', '512GB', '1TB+', 'Others'], '💿');
        
        // --- 2. Vehicle Fields ---
        $fuel = $this->createGlobal('Fuel Type', 'fuel_type', 'dropdown', ['Petrol', 'Diesel', 'CNG', 'Electric', 'Hybrid', 'LPG'], '⛽');
        $trans = $this->createGlobal('Transmission', 'transmission', 'dropdown', ['Manual', 'Automatic'], '⚙️');
        $km = $this->createGlobal('KM Driven', 'km_driven', 'number', null, 'speedometer');
        $year = $this->createGlobal('Year', 'year', 'number', null, 'calendar');
        $owners = $this->createGlobal('No. of Owners', 'owners', 'dropdown', ['1st', '2nd', '3rd', '4th+'], 'bust_in_silhouette');
        
        // --- 3. Property Fields ---
        $bedrooms = $this->createGlobal('Bedrooms', 'bedrooms', 'dropdown', ['1 RK', '1 BHK', '2 BHK', '3 BHK', '4 BHK', '5+ BHK'], '🛏️');
        $bathrooms = $this->createGlobal('Bathrooms', 'bathrooms', 'dropdown', ['1', '2', '3', '4+'], 'bath');
        $furnishing = $this->createGlobal('Furnishing', 'furnishing', 'dropdown', ['Furnished', 'Semi-Furnished', 'Unfurnished'], 'couch_and_lamp');
        $carpet = $this->createGlobal('Carpet Area (sqft)', 'carpet_area', 'number', null, 'straighten');
        $construction = $this->createGlobal('Construction Status', 'construction_status', 'dropdown', ['New Launch', 'Ready to Move', 'Under Construction'], 'building_construction');
        $listedBy = $this->createGlobal('Listed By', 'listed_by', 'dropdown', ['Owner', 'Dealer', 'Builder'], 'user_tie');
        
        // --- 4. Job Fields ---
        $salaryPeriod = $this->createGlobal('Salary Period', 'salary_period', 'dropdown', ['Hourly', 'Weekly', 'Monthly', 'Yearly'], 'calendar');
        $positionType = $this->createGlobal('Position Type', 'position_type', 'dropdown', ['Full-time', 'Part-time', 'Contract', 'Temporary', 'Internship'], 'briefcase');
        $salaryFrom = $this->createGlobal('Salary From', 'salary_from', 'number', null, 'money_bag');
        $salaryTo = $this->createGlobal('Salary To', 'salary_to', 'number', null, 'money_bag');


        $this->command->info('Assigning Fields to Subcategories...');

        // 1. Mobile Phones
        $this->assignToSub('mobile-phones', [
            [$brand, true, 1],
            [$ram, true, 2],
            [$storage, false, 3]
        ]);

        // 2. Cars
        $this->assignToSub('cars', [
            [$fuel, true, 1],
            [$trans, true, 2],
            [$km, false, 3],
            [$year, true, 4],
            [$owners, false, 5]
        ]);
        
        // 3. Properties (Houses, Apartments)
        // Find broad match or specific slugs
        $propSubs = ['for-sale-houses-apartments', 'for-rent-houses-apartments'];
        foreach($propSubs as $slug) {
            $this->assignToSub($slug, [
                [$bedrooms, true, 1],
                [$bathrooms, true, 2],
                [$furnishing, true, 3],
                [$carpet, true, 4],
                [$construction, true, 5],
                [$listedBy, true, 6]
            ]);
        }
        
        // 4. Jobs
        $this->assignToCat('jobs', [
            [$salaryPeriod, true, 1],
            [$positionType, true, 2],
            [$salaryFrom, false, 3],
            [$salaryTo, false, 4]
        ]);

        $this->command->info('Seed Complete!');
    }

    private function createGlobal($name, $slug, $type, $options = null, $icon = null)
    {
        return GlobalField::create([
            'name' => $name,
            'slug' => $slug,
            'type' => $type,
            'options' => $options,
            'icon' => $icon,
            'is_active' => true,
        ]);
    }

    private function assignToSub($slug, $fields)
    {
        $sub = Subcategory::where('slug', $slug)->first();
        if (!$sub) {
            $this->command->warn("Subcategory not found: $slug");
            return;
        }

        foreach ($fields as $f) {
            // $f is [GlobalField, is_required, sort_order]
            $gf = $f[0];
            CategoryField::create([
                'category_id' => $sub->category_id,
                'subcategory_id' => $sub->id,
                'global_field_id' => $gf->id,
                'name' => $gf->name, // Snapshot
                'slug' => $gf->slug,
                'type' => $gf->type,
                'options' => $gf->options,
                'icon' => $gf->icon,
                'is_required' => $f[1],
                'sort_order' => $f[2],
                'is_active' => true,
            ]);
        }
    }
    
    private function assignToCat($slug, $fields)
    {
        // For assigning to all subcategories of a main category (like Jobs)
        $cat = Category::where('slug', $slug)->first();
        if (!$cat) {
             $this->command->warn("Category not found: $slug");
             return;
        }
        
        // Assign to all active subcategories
        foreach ($cat->subcategories as $sub) {
            foreach ($fields as $f) {
                $gf = $f[0];
                CategoryField::create([
                    'category_id' => $cat->id,
                    'subcategory_id' => $sub->id,
                    'global_field_id' => $gf->id,
                    'name' => $gf->name,
                    'slug' => $gf->slug,
                    'type' => $gf->type,
                    'options' => $gf->options,
                    'icon' => $gf->icon,
                    'is_required' => $f[1],
                    'sort_order' => $f[2],
                    'is_active' => true,
                ]);
            }
        }
    }
}
