<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\User;
use App\Models\UserPlan;
use App\Models\Ad;
use App\Models\Category;
use App\Models\Subcategory;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class DemoAdsSeeder extends Seeder
{
    public function run(): void
    {
        // 1. Get Plans
        $standardPlan = UserPlan::where('plan_name', 'Standard')->first();
        $vipSellerPlan = UserPlan::where('plan_name', 'VIP Seller')->first();
        $vipProPlan = UserPlan::where('plan_name', 'VIP Pro')->first();

        // 2. Create Users
        $userStandard = User::firstOrCreate(
            ['email' => 'demo_standard@clxpro.com'],
            [
                'name' => 'John Doe (Standard)',
                'password' => Hash::make('password'),
                'current_plan_id' => $standardPlan?->id,
                'is_verified' => false,
                'city' => 'New York',
                'area' => 'Brooklyn',
                'mobile_number' => '1234567890',
            ]
        );

        $userVipSeller = User::firstOrCreate(
            ['email' => 'demo_vip_seller@clxpro.com'],
            [
                'name' => 'Sarah Smith (VIP Seller)',
                'password' => Hash::make('password'),
                'current_plan_id' => $vipSellerPlan?->id,
                'is_verified' => true,
                'city' => 'Los Angeles',
                'area' => 'Beverly Hills',
                'mobile_number' => '1234567891',
                // Add trust score related logic if needed via secondary update
            ]
        );

        $userVipPro = User::firstOrCreate(
            ['email' => 'demo_vip_pro@clxpro.com'],
            [
                'name' => 'Michael Luxury (VIP Pro)',
                'password' => Hash::make('password'),
                'current_plan_id' => $vipProPlan?->id,
                'is_verified' => true,
                'city' => 'Miami',
                'area' => 'South Beach',
                'mobile_number' => '1234567892',
            ]
        );

        // 3. Create Ads
        
        // --- VIP PRO ADS (High End, Featured, Boosted) ---
        $this->createAd(
            $userVipPro, 
            'Properties', 'Sale', // Hint: 'For Sale: Houses & Apartments'
            'Luxury Penthouse with Ocean View', 
            'Experience luxury living in this stunning penthouse suite featuring 3 bedrooms.',
            2500000.00, 'AED', 
            ['is_featured' => true, 'is_boosted' => true, 'ad_type' => 'selling']
        );

        $this->createAd(
            $userVipPro, 
            'Cars', 'Cars', // Category 'Cars' (or Vehicles), Sub 'Cars' (or verify)
            '2024 Sports Car - Limited Edition', 
            'Brand new condition, only 500 miles. Fully loaded.',
            120000.00, 'AED',
            ['is_featured' => true, 'is_boosted' => true, 'ad_type' => 'selling']
        );

        // --- VIP SELLER ADS (Mid-High End, Featured) ---
        $this->createAd(
            $userVipSeller, 
            'Electronics & Appliances', 'Computers', // Hint: 'Computers & Laptops'
            'MacBook Pro M3 Max - 16 inch', 
            'Latest model, 1TB SSD, 32GB RAM. Used for 1 month.',
            2800.00, 'AED',
            ['is_featured' => true, 'is_boosted' => false, 'ad_type' => 'selling']
        );

        $this->createAd(
            $userVipSeller, 
            'Furniture', 'Sofa', // Hint: 'Sofa & Dining'
            'Designer Italian Leather Sofa', 
            'Modern beige leather sofa, 3-seater.',
            1500.00, 'AED',
            ['is_featured' => true, 'is_boosted' => false, 'ad_type' => 'selling']
        );

        // --- STANDARD ADS (Everyday Items) ---
        $this->createAd(
            $userStandard, 
            'Mobiles', 'Mobile Phones', 
            'iPhone 13 - 128GB Blue', 
            'Good condition, minor scratches.',
            400.00, 'AED',
            ['is_featured' => false, 'is_boosted' => false, 'ad_type' => 'selling']
        );

        $this->createAd(
            $userStandard, 
            'Books', 'Books', // Might need fuzzy match
            'University Physics Textbook', 
            '14th Edition. Like new.',
            50.00, 'AED',
            ['is_featured' => false, 'is_boosted' => false, 'ad_type' => 'selling']
        );

        $this->command->info('✅ Demo Ads seeded successfully for VIP Pro, VIP Seller, and Standard users!');
    }

    private function createAd(User $user, string $catInfo, string $subInfo, string $title, string $desc, float $price, string $currency, array $options = [])
    {
        // Flexible Category Finding
        $category = Category::where('name', $catInfo)
            ->orWhere('name', 'like', "%$catInfo%")
            ->first();

        if (!$category) {
            // Fallback: Try to find by slug if name fails? Or just pick first to avoid crash
             $this->command->warn("Category '$catInfo' not found for: $title. Skipping.");
             return;
        }

        // Flexible Subcategory Finding within Category
        $subcategory = Subcategory::where('category_id', $category->id)
            ->where(function($q) use ($subInfo) {
                $q->where('name', $subInfo)
                  ->orWhere('name', 'like', "%$subInfo%");
            })->first();

        if (!$subcategory) {
             // Fallback: Pick first subcategory of this category
             $subcategory = $category->subcategories()->first();
             if (!$subcategory) {
                 $this->command->warn("No subcategories for '$catInfo' ($title). Skipping.");
                 return;
             }
             $this->command->info("Subcategory '$subInfo' not found, using '{$subcategory->name}' for $title");
        }

        $ad = Ad::create([
            'user_id' => $user->id,
            'category_id' => $category->id,
            'subcategory_id' => $subcategory->id,
            'title' => $title,
            'description' => $desc,
            'price' => $price,
            'currency' => $currency,
            'city' => $user->city,
            'state' => 'State', // Default
            'location_city' => $user->city,
            'location_state' => 'State',
            'location_country' => 'Country',
            'location_address' => $user->area . ', ' . $user->city,
            'location_latitude' => 25.2048, 
            'location_longitude' => 55.2708,
            'latitude' => 25.2048,
            'longitude' => 55.2708,
            'status' => 'Active',
            // 'slug' => Str::slug($title . '-' . Str::random(6)), // Removed as column missing
            // 'views' => rand(10, 500),
            // 'impressions' => rand(100, 2000),
            'is_featured' => $options['is_featured'] ?? false,
            'is_boosted' => $options['is_boosted'] ?? false,
            'ad_type' => $options['ad_type'] ?? 'selling',
            'created_at' => now(),
        ]);

        // Create Dummy Images
        // 1. Primary Image
        \App\Models\AdImage::create([
            'ad_id' => $ad->id,
            'image_url' => 'https://placehold.co/600x400/2563eb/ffffff/png?text=' . urlencode($title),
            'order' => 0,
            'is_primary' => true,
        ]);

        // 2. Secondary Image
        \App\Models\AdImage::create([
            'ad_id' => $ad->id,
            'image_url' => 'https://placehold.co/600x400/e11d48/ffffff/png?text=' . urlencode($title . ' Detail'),
            'order' => 1,
            'is_primary' => false,
        ]);
    }
}
