<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\CategoryField;

class CategoryFieldsSeeder extends Seeder
{
    public function run(): void
    {
        // Safety: If no subcategories match, the checks below (if $sub) will safely skip creating fields.
        \Illuminate\Support\Facades\Schema::disableForeignKeyConstraints();
        
        $this->command->info('Cleaning up old fields...');
        CategoryField::query()->delete();
        if (\Illuminate\Support\Facades\DB::getDriverName() === 'sqlite') {
            \Illuminate\Support\Facades\DB::update("DELETE FROM sqlite_sequence WHERE name = 'category_fields'");
        }
        
        \Illuminate\Support\Facades\Schema::enableForeignKeyConstraints();

        // 1. Mobile Phones
        $mobileSub = Subcategory::where('slug', 'mobile-phones')->first();
        if ($mobileSub) {
            $this->createField($mobileSub->id, 'Brand', 'brand', 'dropdown', ['Apple', 'Samsung', 'Xiaomi', 'Vivo', 'Oppo', 'OnePlus', 'Google', 'Other'], '🏷️', null, true); // Required
            $this->createField($mobileSub->id, 'RAM', 'ram', 'dropdown', ['1GB', '2GB', '3GB', '4GB', '6GB', '8GB', '12GB', '16GB+'], '💾', null, true); // Required
            $this->createField($mobileSub->id, 'Storage', 'storage', 'dropdown', ['16GB', '32GB', '64GB', '128GB', '256GB', '512GB', '1TB+'], '💿', null, false); // Optional
        }

        // 2. Cars
        $carsCat = Category::where('slug', 'cars')->first(); 
        if ($carsCat) {
             $this->createField(null, 'Fuel Type', 'fuel_type', 'dropdown', ['Petrol', 'Diesel', 'CNG', 'Hybrid', 'Electric', 'LPG'], '⛽', $carsCat->id, true); // Required
             $this->createField(null, 'Transmission', 'transmission', 'dropdown', ['Manual', 'Automatic'], '⚙️', $carsCat->id, true); // Required
             $this->createField(null, 'Owner', 'owner', 'dropdown', ['1st', '2nd', '3rd', '4th+'], 'bust_in_silhouette', $carsCat->id, false); // Optional
             $this->createField(null, 'KM Driven', 'km_driven', 'number', null, 'speedometer', $carsCat->id, false); // Optional
             $this->createField(null, 'Year', 'year', 'number', null, 'calendar', $carsCat->id, true); // Required
        }

        // 3. Properties - Using generic slug match for houses/apartments
        // Since we re-seeded, let's look for likely matches or just the main category if easier, but user wants specific.
        // Base seeder has 'For Sale: Houses & Apartments' -> slug 'for-sale-houses-apartments'
        $propSub = Subcategory::where('slug', 'for-sale-houses-apartments')->first();
        if ($propSub) {
            $this->createField($propSub->id, 'Bedrooms', 'bedrooms', 'dropdown', ['1 RK', '1 BHK', '2 BHK', '3 BHK', '4 BHK', '5+ BHK'], '🛏️', null, true); // Required
            $this->createField($propSub->id, 'Bathrooms', 'bathrooms', 'dropdown', ['1', '2', '3', '4+'], 'bath', null, true); // Required
            $this->createField($propSub->id, 'Furnishing', 'furnishing', 'dropdown', ['Furnished', 'Semi-Furnished', 'Unfurnished'], 'couch_and_lamp', null, true); // Required
            $this->createField($propSub->id, 'Carpet Area (sqft)', 'carpet_area', 'number', null, 'straighten', null, true); // Required - Type Input
            $this->createField($propSub->id, 'Construction Status', 'construction_status', 'dropdown', ['New Launch', 'Ready to Move', 'Under Construction'], 'building_construction', null, true); // Required
            $this->createField($propSub->id, 'Listed by', 'listed_by', 'dropdown', ['Owner', 'Dealer', 'Builder'], 'user_tie', null, true); // Required
        }

        // 4. Jobs
        $jobsCat = Category::where('slug', 'jobs')->first();
        if ($jobsCat) {
             $this->createField(null, 'Salary Period', 'salary_period', 'dropdown', ['Hourly', 'Weekly', 'Monthly', 'Yearly'], 'calendar', $jobsCat->id, true); // Required
             $this->createField(null, 'Position Type', 'position_type', 'dropdown', ['Full-time', 'Part-time', 'Contract', 'Temporary'], 'briefcase', $jobsCat->id, true); // Required
        }
    }

    private function createField($subId, $name, $slug, $type, $options = null, $icon = null, $catId = null, $required = false)
    {
        CategoryField::create([
            'category_id' => $catId,
            'subcategory_id' => $subId,
            'name' => $name,
            'slug' => $slug,
            'type' => $type,
            'options' => $options,
            'icon' => $icon,
            'is_active' => true,
            'is_required' => $required,
        ]);
    }
}
