<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Ad;
use App\Models\AdAnalytics;
use App\Models\AdAnalyticsRealtime;
use Carbon\Carbon;

class AnalyticsTestDataSeeder extends Seeder
{
    /**
     * Seed 30 days of analytics data for existing ads
     */
    public function run(): void
    {
        $ads = Ad::limit(20)->get(); // Get first 20 ads

        if ($ads->isEmpty()) {
            $this->command->warn('No ads found. Please seed ads first.');
            return;
        }

        $this->command->info('Seeding analytics for ' . $ads->count() . ' ads...');

        foreach ($ads as $ad) {
            $this->seedDailyAnalytics($ad);
            $this->seedRealtimeAnalytics($ad);
        }

        $this->command->info('Analytics data seeded successfully!');
    }

    /**
     * Seed 30 days of daily analytics
     */
    private function seedDailyAnalytics(Ad $ad): void
    {
        for ($i = 29; $i >=0; $i--) {
            $date = Carbon::now()->subDays($i)->startOfDay();
            
            // Generate realistic random data with some trends
            $baseImpressions = rand(100, 500);
            $dayOfWeek = $date->dayOfWeek;
            
            // Weekend boost
            if (in_array($dayOfWeek, [5, 6, 0])) { // Fri, Sat, Sun
                $baseImpressions = (int)($baseImpressions * 1.5);
            }
            
            $impressions = $baseImpressions + rand(-50, 100);
            $views = (int)($impressions * rand(15, 30) / 100); // 15-30% view rate
            $clicks = (int)($views * rand(5, 15) / 100); // 5-15% of views convert to clicks
            $ctr = $impressions > 0 ? round(($clicks / $impressions) * 100, 2) : 0;
            $uniqueViewers = (int)($views * rand(70, 90) / 100);
            $avgTimeSpent = rand(30, 180); // 30s to 3min
            $bounceRate = rand(20, 60); // 20-60%

            AdAnalytics::create([
                'ad_id' => $ad->id,
                'date' => $date,
                'impressions' => $impressions,
                'views' => $views,
                'clicks' => $clicks,
                'ctr' => $ctr,
                'unique_viewers' => $uniqueViewers,
                'avg_time_spent' => $avgTimeSpent,
                'bounce_rate' => $bounceRate,
            ]);
        }
    }

    /**
     * Seed realtime analytics for the last 6 hours
     */
    private function seedRealtimeAnalytics(Ad $ad): void
    {
        $now = Carbon::now();
        $sources = ['search', 'feed', 'related', 'direct'];

        // Generate events for last 6 hours
        for ($i = 0; $i < 360; $i++) { // Every minute
            $timestamp = $now->copy()->subMinutes($i);
            
            // Random number of events per minute
            $numEvents = rand(0, 5);
            
            for ($j = 0; $j < $numEvents; $j++) {
                $eventTypes = ['impression', 'impression', 'impression', 'view', 'click']; // Weighted
                $eventType = $eventTypes[array_rand($eventTypes)];
                
                AdAnalyticsRealtime::create([
                    'ad_id' => $ad->id,
                    'event_type' => $eventType,
                    'timestamp' => $timestamp->copy()->addSeconds(rand(0, 59)),
                    'user_id' => rand(0, 1) ? rand(1, 100) : null,
                    'session_id' => 'sess_' . uniqid(),
                    'source' => $sources[array_rand($sources)],
                    'ip_address' => $this->generateRandomIP(),
                    'user_agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64)',
                ]);
            }
        }
    }

    /**
     * Generate a random IP address
     */
    private function generateRandomIP(): string
    {
        return rand(1, 255) . '.' . rand(0, 255) . '.' . rand(0, 255) . '.' . rand(1, 255);
    }
}
