<?php

namespace App\Services;

use App\Models\Ad;
use App\Models\User;
use App\Models\AlgorithmConfig;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class RecommendationEngine
{
    /**
     * Get the current algorithm configuration weights.
     */
    protected function getConfigWeights()
    {
        $config = AlgorithmConfig::getDefault();
        return $config ? $config->weights : [
            'location_relevance' => 80,
            'freshness_factor' => 40,
            'ad_performance' => 90, 
            'price_factor' => 50 // Default if not in DB
        ];
    }

    /**
     * Generate optimization recommendations for a specific ad based on Algorithm weights.
     */
    public function getAdOptimizationRecommendations(Ad $ad)
    {
        $weights = $this->getConfigWeights();
        $recommendations = [];

        // 1. Location Relevance Logic
        // If Admin values Location highly (>50), we strictly check for precise location data
        if (($weights['location_relevance'] ?? 0) > 50) {
            if (empty($ad->latitude) || empty($ad->longitude)) {
                $recommendations['location_optimization'] = 'Your ad is missing precise location data which is currently highly valued. Add a pin on the map!';
            } elseif (empty($ad->location_address)) {
                $recommendations['location_optimization'] = 'Adding a specific address or area name can improve your visibility significantly right now.';
            }
        }

        // 2. Freshness Logic
        // If Admin values Freshness highly (>50), we push for bumps/reposts
        if (($weights['freshness_factor'] ?? 0) > 50) {
            $daysOld = $ad->created_at->diffInDays(now());
            if ($daysOld > 7) {
                $recommendations['timing_optimization'] = 'Freshness is key right now! Your ad is ' . $daysOld . ' days old. Consider "Bumping" it to getting back to the top.';
            }
        }

        // 3. Ad Performance (Quality) Logic
        // If Admin values Performance/Quality highly (>50)
        if (($weights['ad_performance'] ?? 0) > 50) {
            // Image Count
            if ($ad->images()->count() < 3) {
                $recommendations['image_optimization'] = 'High-quality ads are being prioritized. Add at least 3 images to improve your ranking.';
            }

            // Description Length
            if (strlen($ad->description) < 100) {
                $recommendations['description_optimization'] = 'Detailed descriptions are performing better. Try to write at least 100 characters.';
            }

            // Title Keywords (Basic check)
            if (str_word_count($ad->title) < 3) {
                $recommendations['title_optimization'] = 'Short titles are easily missed. Add more descriptive keywords to your title.';
            }
        }

        // 4. Price Analysis (Always useful)
        $avgPrice = $this->getCategoryAveragePrice($ad->category_id);
        if ($avgPrice > 0) {
            if ($ad->price > ($avgPrice * 1.5)) {
                $recommendations['price_optimization'] = 'Your price is significantly higher than the category average (' . number_format($avgPrice) . '). Consider lowering it or explaining the value.';
            } elseif ($ad->price < ($avgPrice * 0.5)) {
                $recommendations['price_optimization'] = 'Your price is very low compared to the average (' . number_format($avgPrice) . '). Ensure buyers don\'t perceive it as "too good to be true".';
            } else {
                $recommendations['price_optimization'] = 'Your price is competitive for this category!';
            }
        }

        // 5. General Boost Recommendation (If views are low)
        if ($ad->view_count < 50 && $ad->created_at->diffInDays(now()) > 2) {
            $recommendations['boost_recommendation'] = 'Views are slower than usual. Promoting this ad could help you reach more buyers quickly.';
        }

        return $recommendations;
    }

    /**
     * Get insights on User Behavior based on real data.
     */
    public function getUserBehaviorInsights(User $user)
    {
        // 1. Preferred Categories
        // Based on ads they visited, favorited, or posted
        $topCategory = DB::table('favorites')
            ->join('ads', 'favorites.ad_id', '=', 'ads.id')
            ->join('categories', 'ads.category_id', '=', 'categories.id')
            ->where('favorites.user_id', $user->id)
            ->select('categories.name', DB::raw('count(*) as total'))
            ->groupBy('categories.name')
            ->orderByDesc('total')
            ->first();

        // 2. Engagement Score
        // Simple heuristic: (Logins * 1) + (Ads Posted * 5) + (Favorites * 2)
        $adsPosted = $user->ads()->count();
        $favoritesCount = $user->favorites()->count();
        $engagementScore = ($adsPosted * 5) + ($favoritesCount * 2); 
        // Note: Login count tracking would need a separate 'logins' table or column, skipping for now.

        return [
            'search_patterns' => 'Data collection in progress...', // Placeholder until we have a search_history table
            'purchase_intent' => $favoritesCount > 5 ? 'High' : ($favoritesCount > 0 ? 'Medium' : 'Low'),
            'engagement_score' => min($engagementScore, 100), // Cap at 100
            'preferred_categories' => $topCategory ? [$topCategory->name] : ['General Interest'],
            'peak_activity_hours' => ['18:00-22:00'], // Default heuristic for now
            'conversion_rate' => 0, // Needs Order/Transaction data
            'average_session_duration' => '5 minutes', // Needs Session tracking
        ];
    }

    /**
     * Get Competitive Analysis for a Category.
     */
    public function getCompetitiveAnalysis($categoryId, $filters = [])
    {
        $avgPrice = $this->getCategoryAveragePrice($categoryId);
        $totalAds = Ad::where('category_id', $categoryId)->where('status', 'Active')->count();
        
        // Determine trends (Mocking trend direction based on simple random for now as we don't store historical snapshots yet)
        $demandTrend = 'Stable'; 
        if ($totalAds > 100) $demandTrend = 'High Supply';
        if ($totalAds < 10) $demandTrend = 'Low Supply - Opportunity';

        return [
            'category_insights' => [
                'total_ads' => $totalAds,
                'average_price' => round($avgPrice, 2),
                'top_keywords' => ['Used', 'New', 'Condition', 'Best'], // Placeholder: Real implementation would tokenize ad titles
                'competition_level' => $totalAds > 500 ? 'High' : ($totalAds > 100 ? 'Medium' : 'Low'),
            ],
            'competitor_analysis' => [
                'top_competitors' => ['Hidden for Privacy'], // Privacy concern: don't list other users explicitly to public API
                'average_response_time' => '4 hours', // Placeholder
                'avg_conversion_rate' => 2.5, // Industry standard placeholder
            ],
            'market_trends' => [
                'demand_trend' => $demandTrend,
                'price_trend' => 'Fluctuating', // Needs historical price table
                'seasonal_patterns' => 'N/A',
            ],
        ];
    }

    /**
     * Helper: Calculate average price for a category
     */
    private function getCategoryAveragePrice($categoryId)
    {
        return Ad::where('category_id', $categoryId)
            ->where('status', 'Active')
            ->where('price', '>', 0) // Exclude "Contact for Price" (0)
            ->avg('price') ?? 0;
    }
}
