<?php

namespace App\Services;

use App\Models\PaymentGateway;
use App\Models\Transaction;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;

class PhonePeGateway
{
    protected $gateway;
    protected $baseUrl;
    protected $merchantId;
    protected $saltKey;
    protected $saltIndex;

    public function __construct(PaymentGateway $gateway)
    {
        $this->gateway = $gateway;
        
        if ($gateway->isTestMode()) {
            $this->baseUrl = 'https://api-preprod.phonepe.com/apis/pg-sandbox';
        } else {
            $this->baseUrl = 'https://api.phonepe.com/apis/pg';
        }
        
        $this->merchantId = $gateway->getApiKey();
        $this->saltKey = $gateway->getApiSecret();
        $this->saltIndex = 1; // Default salt index
    }

    public function createOrder($orderId, $amount, $currency, User $user, $description = null)
    {
        try {
            $payload = [
                'merchantId' => $this->merchantId,
                'merchantTransactionId' => $orderId,
                'merchantUserId' => 'USER' . $user->id,
                'amount' => $amount * 100, // Convert to paise
                'redirectUrl' => route('payment.phonepe.callback', ['order_id' => $orderId]),
                'redirectMode' => 'POST',
                'callbackUrl' => route('payment.phonepe.webhook'),
                'paymentInstrument' => [
                    'type' => 'PAY_PAGE'
                ]
            ];

            if ($user->email) {
                $payload['email'] = $user->email;
            }
            
            if ($user->phone_number) {
                $payload['mobileNumber'] = $user->phone_number;
            }

            if ($description) {
                $payload['description'] = $description;
            }

            // Encode payload
            $encodedPayload = base64_encode(json_encode($payload));
            $payloadHash = hash('sha256', $encodedPayload . '/pg/v1/pay' . $this->saltKey);
            $finalXHeader = $payloadHash . '###' . $this->saltIndex;

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'X-VERIFY' => $finalXHeader,
                'X-MERCHANT-ID' => $this->merchantId,
            ])->post($this->baseUrl . '/pg/v1/pay', [
                'request' => $encodedPayload
            ]);

            if ($response->successful()) {
                $responseData = $response->json();
                
                if (isset($responseData['success']) && $responseData['success']) {
                    return [
                        'success' => true,
                        'payment_url' => $responseData['data']['instrumentResponse']['redirectInfo']['url'],
                        'transaction_id' => $responseData['data']['merchantTransactionId'],
                        'order_id' => $orderId,
                    ];
                } else {
                    return [
                        'success' => false,
                        'error' => $responseData['message'] ?? 'Payment initiation failed',
                    ];
                }
            } else {
                return [
                    'success' => false,
                    'error' => 'PhonePe API request failed',
                ];
            }

        } catch (\Exception $e) {
            Log::error('PhonePe order creation failed', [
                'order_id' => $orderId,
                'amount' => $amount,
                'currency' => $currency,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function verifyPayment(Transaction $transaction, $paymentData)
    {
        try {
            $transactionId = $paymentData['transaction_id'] ?? null;
            $responseCode = $paymentData['response_code'] ?? null;
            $responseMessage = $paymentData['response_message'] ?? null;

            if (!$transactionId) {
                return [
                    'success' => false,
                    'error' => 'Missing transaction ID',
                ];
            }

            // Verify the transaction status
            $statusPayload = [
                'merchantId' => $this->merchantId,
                'merchantTransactionId' => $transactionId,
            ];

            $encodedStatusPayload = base64_encode(json_encode($statusPayload));
            $statusPayloadHash = hash('sha256', $encodedStatusPayload . '/pg/v1/status/' . $transactionId . $this->saltKey);
            $statusXHeader = $statusPayloadHash . '###' . $this->saltIndex;

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'X-VERIFY' => $statusXHeader,
                'X-MERCHANT-ID' => $this->merchantId,
            ])->get($this->baseUrl . '/pg/v1/status/' . $transactionId);

            if ($response->successful()) {
                $statusData = $response->json();
                
                if (isset($statusData['success']) && $statusData['success']) {
                    if ($statusData['data']['code'] === 'PAYMENT_SUCCESS') {
                        return [
                            'success' => true,
                            'transaction_id' => $transactionId,
                            'amount' => $statusData['data']['amount'] / 100, // Convert back to rupees
                            'currency' => $statusData['data']['currency'],
                            'payment_method' => $statusData['data']['paymentInstrument']['type'] ?? 'unknown',
                        ];
                    } else {
                        return [
                            'success' => false,
                            'error' => 'Payment failed: ' . $statusData['data']['code'],
                        ];
                    }
                } else {
                    return [
                        'success' => false,
                        'error' => $statusData['message'] ?? 'Payment verification failed',
                    ];
                }
            } else {
                return [
                    'success' => false,
                    'error' => 'PhonePe status API request failed',
                ];
            }

        } catch (\Exception $e) {
            Log::error('PhonePe payment verification failed', [
                'transaction_id' => $transaction->transaction_id,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function createSubscription(User $user, $plan)
    {
        // PhonePe doesn't have native subscription support like Razorpay
        // This would need to be implemented using their recurring payment APIs
        // For now, return not supported
        
        return [
            'success' => false,
            'error' => 'PhonePe subscriptions not implemented yet',
        ];
    }

    public function refundPayment(Transaction $transaction, $amount, $reason = null)
    {
        try {
            $transactionId = $transaction->gateway_transaction_id;
            
            if (!$transactionId) {
                return [
                    'success' => false,
                    'error' => 'No transaction ID found for refund',
                ];
            }

            $refundPayload = [
                'merchantId' => $this->merchantId,
                'merchantTransactionId' => $transactionId . '_REFUND_' . time(),
                'originalTransactionId' => $transactionId,
                'amount' => $amount * 100, // Convert to paise
            ];

            if ($reason) {
                $refundPayload['reason'] = $reason;
            }

            $encodedRefundPayload = base64_encode(json_encode($refundPayload));
            $refundPayloadHash = hash('sha256', $encodedRefundPayload . '/pg/v1/refund' . $this->saltKey);
            $refundXHeader = $refundPayloadHash . '###' . $this->saltIndex;

            $response = Http::withHeaders([
                'Content-Type' => 'application/json',
                'X-VERIFY' => $refundXHeader,
                'X-MERCHANT-ID' => $this->merchantId,
            ])->post($this->baseUrl . '/pg/v1/refund', [
                'request' => $encodedRefundPayload
            ]);

            if ($response->successful()) {
                $refundData = $response->json();
                
                if (isset($refundData['success']) && $refundData['success']) {
                    return [
                        'success' => true,
                        'refund_id' => $refundData['data']['refundId'],
                        'refunded_amount' => $refundData['data']['amount'] / 100,
                        'status' => $refundData['data']['status'],
                    ];
                } else {
                    return [
                        'success' => false,
                        'error' => $refundData['message'] ?? 'Refund failed',
                    ];
                }
            } else {
                return [
                    'success' => false,
                    'error' => 'PhonePe refund API request failed',
                ];
            }

        } catch (\Exception $e) {
            Log::error('PhonePe refund failed', [
                'transaction_id' => $transaction->transaction_id,
                'amount' => $amount,
                'error' => $e->getMessage(),
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
            ];
        }
    }

    public function verifyWebhookSignature($payload, $signature)
    {
        // PhonePe webhook verification logic
        // This would need to be implemented based on PhonePe's webhook signature format
        return true; // Placeholder
    }

    public function processWebhook($webhookData)
    {
        try {
            // Process PhonePe webhook
            $transactionId = $webhookData['transactionId'] ?? null;
            $status = $webhookData['status'] ?? null;
            $amount = $webhookData['amount'] ?? null;

            if ($transactionId && $status === 'SUCCESS') {
                // Find transaction and update status
                $transaction = Transaction::where('gateway_order_id', $transactionId)->first();
                
                if ($transaction) {
                    $transaction->update([
                        'status' => 'success',
                        'payment_status' => 'captured',
                        'gateway_transaction_id' => $transactionId,
                        'gateway_response' => $webhookData,
                        'paid_at' => now(),
                        'captured_at' => now(),
                    ]);

                    // Process the entity payment
                    $this->processEntityPayment($transaction);

                    return ['success' => true];
                }
            }

            return ['success' => false, 'error' => 'Transaction not found or not successful'];

        } catch (\Exception $e) {
            Log::error('PhonePe webhook processing failed', [
                'webhook_data' => $webhookData,
                'error' => $e->getMessage(),
            ]);

            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
}
