<?php

namespace App\Services;

use PayPalCheckoutSdk\Core\PayPalHttpClient;
use PayPalCheckoutSdk\Core\SandboxEnvironment;
use PayPalCheckoutSdk\Core\ProductionEnvironment;
use PayPalCheckoutSdk\Orders\OrdersCreateRequest;
use PayPalCheckoutSdk\Orders\OrdersCaptureRequest;
use PayPalCheckoutSdk\Orders\OrdersGetRequest;
use Exception;

class PayPalService
{
    protected $client;

    public function __construct()
    {
        $clientId = config('services.paypal.client_id');
        $clientSecret = config('services.paypal.client_secret');
        $mode = config('services.paypal.mode', 'sandbox');

        $environment = $mode === 'production'
            ? new ProductionEnvironment($clientId, $clientSecret)
            : new SandboxEnvironment($clientId, $clientSecret);

        $this->client = new PayPalHttpClient($environment);
    }

    /**
     * Create an order
     */
    public function createOrder($amount, $currency = 'USD', $description = 'Purchase')
    {
        try {
            $request = new OrdersCreateRequest();
            $request->prefer('return=representation');
            $request->body = [
                'intent' => 'CAPTURE',
                'purchase_units' => [[
                    'amount' => [
                        'currency_code' => $currency,
                        'value' => number_format($amount, 2, '.', '')
                    ],
                    'description' => $description
                ]],
                'application_context' => [
                    'return_url' => config('app.url') . '/payment/success',
                    'cancel_url' => config('app.url') . '/payment/cancel',
                ]
            ];

            $response = $this->client->execute($request);

            return [
                'id' => $response->result->id,
                'status' => $response->result->status,
                'approval_url' => $this->getApprovalUrl($response->result->links),
            ];
        } catch (Exception $e) {
            throw new Exception('PayPal order creation failed: ' . $e->getMessage());
        }
    }

    /**
     * Capture an order
     */
    public function captureOrder($orderId)
    {
        try {
            $request = new OrdersCaptureRequest($orderId);
            $response = $this->client->execute($request);

            return [
                'id' => $response->result->id,
                'status' => $response->result->status,
                'payer' => $response->result->payer,
                'amount' => $response->result->purchase_units[0]->payments->captures[0]->amount->value,
            ];
        } catch (Exception $e) {
            throw new Exception('PayPal order capture failed: ' . $e->getMessage());
        }
    }

    /**
     * Get order details
     */
    public function getOrder($orderId)
    {
        try {
            $request = new OrdersGetRequest($orderId);
            $response = $this->client->execute($request);

            return [
                'id' => $response->result->id,
                'status' => $response->result->status,
                'amount' => $response->result->purchase_units[0]->amount->value,
                'currency' => $response->result->purchase_units[0]->amount->currency_code,
            ];
        } catch (Exception $e) {
            throw new Exception('Failed to fetch PayPal order: ' . $e->getMessage());
        }
    }

    /**
     * Get approval URL from links
     */
    private function getApprovalUrl($links)
    {
        foreach ($links as $link) {
            if ($link->rel === 'approve') {
                return $link->href;
            }
        }
        return null;
    }
}
