<?php

namespace App\Services;

use App\Models\User;
use App\Models\Notification;
use App\Models\NotificationDelivery;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Http;
use Google\Auth\Credentials\ServiceAccountCredentials;
use Exception;

class FirebaseService
{
    protected $projectId;
    protected $credentialsFile;
    
    public function __construct()
    {
        $this->credentialsFile = storage_path('app/firebase_credentials.json');
        
        // Extract Project ID from JSON file if possible, or env
        if (file_exists($this->credentialsFile)) {
            $json = json_decode(file_get_contents($this->credentialsFile), true);
            $this->projectId = $json['project_id'] ?? env('FIREBASE_PROJECT_ID');
        } else {
            $this->projectId = env('FIREBASE_PROJECT_ID');
        }
    }

    /**
     * Get OAuth2 Access Token
     */
    protected function getAccessToken()
    {
        if (!file_exists($this->credentialsFile)) {
            Log::error("Firebase credentials file not found at: " . $this->credentialsFile);
            throw new Exception("Firebase credentials not configured.");
        }

        $scopes = ['https://www.googleapis.com/auth/firebase.messaging'];
        $credentials = new ServiceAccountCredentials($scopes, $this->credentialsFile);
        $token = $credentials->fetchAuthToken();

        return $token['access_token'];
    }

    /**
     * Send push notification using HTTP v1 API
     */
    public function sendPushNotification(User $user, Notification $notification, NotificationDelivery $delivery): bool
    {
        if (!$this->validateConfiguration()) {
            Log::error('Firebase configuration is invalid');
            return false;
        }

        $deviceToken = $user->device_token;
        if (!$deviceToken) {
            Log::warning("No device token found for user {$user->id}");
            return false;
        }

        try {
            $accessToken = $this->getAccessToken();
            $url = "https://fcm.googleapis.com/v1/projects/{$this->projectId}/messages:send";

            $payload = $this->buildV1Payload($user->device_token, $notification);

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $accessToken,
                'Content-Type' => 'application/json'
            ])->post($url, $payload);

            if ($response->successful()) {
                Log::info("Notification sent successfully to user {$user->id}");
                return true;
            } else {
                $this->handleFirebaseError($response, $user, $notification);
                return false;
            }
        } catch (\Exception $e) {
            Log::error("Failed to send notification: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Build HTTP v1 Payload
     */
    protected function buildV1Payload(string $token, Notification $notification): array
    {
        return [
            'message' => [
                'token' => $token,
                'notification' => [
                    'title' => $notification->title,
                    'body' => $notification->message,
                    'image' => $notification->image ?? null,
                ],
                'data' => [
                    'notification_id' => (string)$notification->id,
                    'click_action' => 'FLUTTER_NOTIFICATION_CLICK',
                    'deep_link' => $notification->deep_link_url ?? '',
                    'priority' => $notification->priority,
                    'category' => $notification->category,
                    'type' => 'general' 
                ],
                'android' => [
                    'priority' => $notification->priority === 'high' ? 'HIGH' : 'NORMAL',
                    'notification' => [
                        'channel_id' => 'high_importance_channel',
                        'default_sound' => true,
                        // 'color' => '#7c3aed' 
                    ]
                ],
                // Add APNS config if needed
            ]
        ];
    }

    /**
     * Handle Firebase Error
     */
    protected function handleFirebaseError($response, User $user, Notification $notification): void
    {
        $error = $response->json()['error'] ?? [];
        $message = $error['message'] ?? 'Unknown error';
        $status = $error['status'] ?? 'UNKNOWN';

        Log::error("Firebase V1 Error: {$status} - {$message}");

        if ($status === 'UNREGISTERED' || $message === 'Requested entity was not found.') {
            // Token is invalid
             // $user->update(['device_token' => null]);
             Log::info("Invalid token for user {$user->id}, considering removing.");
        }
    }

    /**
     * Validate Config
     */
    public function validateConfiguration(): bool
    {
        return file_exists($this->credentialsFile) && !empty($this->projectId);
    }

    /**
     * Get Configuration Status
     */
    public function getConfigurationStatus(): array
    {
        $status = [
            'configured' => $this->validateConfiguration(),
            'credentials_file_exists' => file_exists($this->credentialsFile),
            'project_id' => $this->projectId ?? 'Not set',
            'credentials_path' => $this->credentialsFile,
        ];

        if ($status['configured']) {
            try {
                // Try to get access token to verify credentials are valid
                $this->getAccessToken();
                $status['credentials_valid'] = true;
                $status['message'] = 'Firebase is fully configured and operational';
            } catch (\Exception $e) {
                $status['credentials_valid'] = false;
                $status['message'] = 'Firebase credentials file exists but is invalid: ' . $e->getMessage();
            }
        } else {
            $status['credentials_valid'] = false;
            $status['message'] = 'Firebase is not configured. Missing credentials file or project ID.';
        }

        return $status;
    }
}
class HttpHandlerFactory {
    public static function build() {
        return new \GuzzleHttp\Client();
    }
}
