<?php

namespace App\Services;

use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Exception;

class FirebaseConfigurationService
{
    /**
     * Validate Firebase configuration
     */
    public function validateConfiguration()
    {
        $required = [
            'FIREBASE_PROJECT_ID',
            'FIREBASE_CLIENT_EMAIL', 
            'FIREBASE_PRIVATE_KEY',
            'FIREBASE_SERVER_KEY',
            'FIREBASE_API_KEY'
        ];

        $missing = [];
        foreach ($required as $key) {
            if (empty(env($key))) {
                $missing[] = $key;
            }
        }

        if (!empty($missing)) {
            return [
                'valid' => false,
                'missing' => $missing,
                'message' => 'Firebase configuration is incomplete. Missing: ' . implode(', ', $missing)
            ];
        }

        return [
            'valid' => true,
            'message' => 'Firebase configuration is complete and valid'
        ];
    }

    /**
     * Get Firebase configuration status
     */
    public function getConfigurationStatus()
    {
        $status = [
            'configured' => false,
            'project_id' => env('FIREBASE_PROJECT_ID'),
            'client_email' => env('FIREBASE_CLIENT_EMAIL'),
            'has_private_key' => !empty(env('FIREBASE_PRIVATE_KEY')),
            'has_server_key' => !empty(env('FIREBASE_SERVER_KEY')),
            'has_api_key' => !empty(env('FIREBASE_API_KEY')),
            'database_url' => env('FIREBASE_DATABASE_URL'),
            'storage_bucket' => env('FIREBASE_STORAGE_BUCKET'),
            'sender_id' => env('FIREBASE_SENDER_ID'),
            'app_id' => env('FIREBASE_APP_ID'),
            'measurement_id' => env('FIREBASE_MEASUREMENT_ID'),
            'errors' => []
        ];

        // Check each required field
        $required = [
            'FIREBASE_PROJECT_ID' => 'Project ID',
            'FIREBASE_CLIENT_EMAIL' => 'Client Email',
            'FIREBASE_PRIVATE_KEY' => 'Private Key',
            'FIREBASE_SERVER_KEY' => 'Server Key',
            'FIREBASE_API_KEY' => 'API Key'
        ];

        foreach ($required as $key => $name) {
            if (empty(env($key))) {
                $status['errors'][] = "$name is missing";
            }
        }

        $status['configured'] = empty($status['errors']);

        return $status;
    }

    /**
     * Get Firebase credentials array for SDK
     */
    public function getCredentials()
    {
        return [
            'projectId' => env('FIREBASE_PROJECT_ID'),
            'clientEmail' => env('FIREBASE_CLIENT_EMAIL'),
            'privateKey' => $this->formatPrivateKey(env('FIREBASE_PRIVATE_KEY')),
            'databaseURL' => env('FIREBASE_DATABASE_URL'),
            'storageBucket' => env('FIREBASE_STORAGE_BUCKET'),
            'messagingSenderId' => env('FIREBASE_SENDER_ID'),
            'appId' => env('FIREBASE_APP_ID'),
            'measurementId' => env('FIREBASE_MEASUREMENT_ID'),
        ];
    }

    /**
     * Format private key for Firebase SDK
     */
    protected function formatPrivateKey($key)
    {
        if (empty($key)) {
            return null;
        }

        // Remove any quotes and decode newlines
        $key = str_replace('"', '', $key);
        $key = str_replace('\n', "\n", $key);
        
        return $key;
    }

    /**
     * Get Firebase Cloud Messaging configuration
     */
    public function getMessagingConfig()
    {
        return [
            'server_key' => env('FIREBASE_SERVER_KEY'),
            'api_key' => env('FIREBASE_API_KEY'),
            'project_id' => env('FIREBASE_PROJECT_ID'),
            'sender_id' => env('FIREBASE_SENDER_ID'),
        ];
    }

    /**
     * Check if Firebase is ready for notifications
     */
    public function isReadyForNotifications()
    {
        $config = $this->getConfigurationStatus();
        
        if (!$config['configured']) {
            return [
                'ready' => false,
                'message' => 'Firebase configuration is incomplete',
                'errors' => $config['errors']
            ];
        }

        // Additional checks for notification readiness
        $messagingConfig = $this->getMessagingConfig();
        
        if (empty($messagingConfig['server_key'])) {
            return [
                'ready' => false,
                'message' => 'Firebase Server Key is required for notifications',
                'errors' => ['Missing Firebase Server Key']
            ];
        }

        return [
            'ready' => true,
            'message' => 'Firebase is ready for notifications'
        ];
    }

    /**
     * Generate Firebase setup instructions
     */
    public function getSetupInstructions()
    {
        return [
            'step1' => [
                'title' => 'Create Firebase Project',
                'description' => 'Go to Firebase Console (console.firebase.google.com) and create a new project named "CLX Pro Notifications"',
                'url' => 'https://console.firebase.google.com/'
            ],
            'step2' => [
                'title' => 'Enable Firebase Cloud Messaging',
                'description' => 'In your Firebase project, go to Project Settings > Cloud Messaging and note down the Server Key and Sender ID',
                'url' => 'https://console.firebase.google.com/project/_/settings/cloudmessaging/'
            ],
            'step3' => [
                'title' => 'Create Service Account',
                'description' => 'Go to Project Settings > Service Accounts, click "Generate new private key", and download the JSON file',
                'url' => 'https://console.firebase.google.com/project/_/settings/serviceaccounts/adminsdk'
            ],
            'step4' => [
                'title' => 'Configure Environment',
                'description' => 'Copy the values from your Firebase project and service account into your .env file',
                'env_example' => [
                    'FIREBASE_PROJECT_ID' => 'your-firebase-project-id',
                    'FIREBASE_CLIENT_EMAIL' => 'your-service-account@your-project-id.iam.gserviceaccount.com',
                    'FIREBASE_PRIVATE_KEY' => '-----BEGIN PRIVATE KEY-----\nYOUR_PRIVATE_KEY\n-----END PRIVATE KEY-----\n',
                    'FIREBASE_SERVER_KEY' => 'your-server-key-from-firebase-console',
                    'FIREBASE_API_KEY' => 'your-api-key-from-firebase-console'
                ]
            ],
            'step5' => [
                'title' => 'Test Configuration',
                'description' => 'Run the Firebase configuration test to verify all settings are correct',
                'command' => 'php artisan firebase:test'
            ]
        ];
    }

    /**
     * Test Firebase connection
     */
    public function testConnection()
    {
        try {
            // Test if we can create Firebase service
            $firebaseService = app(FirebaseService::class);
            
            // Test basic configuration
            $config = $this->getConfigurationStatus();
            
            if (!$config['configured']) {
                return [
                    'success' => false,
                    'message' => 'Firebase configuration test failed',
                    'errors' => $config['errors']
                ];
            }

            // Test Firebase SDK initialization
            $credentials = $this->getCredentials();
            
            if (empty($credentials['projectId']) || empty($credentials['privateKey'])) {
                return [
                    'success' => false,
                    'message' => 'Firebase SDK initialization failed - missing required credentials'
                ];
            }

            return [
                'success' => true,
                'message' => 'Firebase configuration test passed successfully',
                'project_id' => $credentials['projectId']
            ];

        } catch (Exception $e) {
            Log::error('Firebase connection test failed: ' . $e->getMessage());
            
            return [
                'success' => false,
                'message' => 'Firebase connection test failed',
                'error' => $e->getMessage()
            ];
        }
    }
}
