<?php

namespace App\Services;

use App\Models\ChatMessage;
use App\Models\User;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Exception;

class FirebaseChatService
{
    protected $fcmService;
    protected $firebaseUrl;
    protected $firebaseSecret;

    public function __construct(FCMService $fcmService)
    {
        $this->fcmService = $fcmService;
        $this->firebaseUrl = env('FIREBASE_DATABASE_URL', 'https://your-project.firebaseio.com');
        $this->firebaseSecret = env('FIREBASE_DATABASE_SECRET');
    }

    /**
     * Initialize or get a chat room
     */
    public function initializeChat($userId1, $userId2, $adId = null)
    {
        try {
            // Create a consistent chat ID regardless of user order
            $chatId = $this->generateChatId($userId1, $userId2, $adId);
            
            // Check if chat exists in Firestore
            $chatData = [
                'chatId' => $chatId,
                'participants' => [$userId1, $userId2],
                'adId' => $adId,
                'createdAt' => now()->toIso8601String(),
                'lastMessage' => null,
                'lastMessageTime' => null,
                'unreadCount' => [
                    $userId1 => 0,
                    $userId2 => 0,
                ],
            ];

            return [
                'chatId' => $chatId,
                'chatData' => $chatData,
            ];
        } catch (Exception $e) {
            Log::error("Firebase chat initialization error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Send a message to Firebase and notify recipient
     */
    public function sendMessage($chatId, $senderId, $receiverId, $message, $type = 'text', $imageUrl = null)
    {
        try {
            // Save to local database first
            $chatMessage = ChatMessage::create([
                'sender_id' => $senderId,
                'receiver_id' => $receiverId,
                'message' => $message,
                'message_type' => $type,
                'attachment_url' => $imageUrl,
            ]);

            // Prepare message data for Firebase
            $messageData = [
                'id' => $chatMessage->id,
                'senderId' => $senderId,
                'text' => $message,
                'type' => $type,
                'timestamp' => now()->toIso8601String(),
                'readBy' => [$senderId],
                'deleted' => false,
            ];

            if ($imageUrl) {
                $messageData['imageUrl'] = $imageUrl;
            }

            // Get sender and receiver info
            $sender = User::find($senderId);
            $receiver = User::find($receiverId);

            // Send push notification to receiver
            if ($receiver && $receiver->fcm_token) {
                $this->fcmService->sendToDevice(
                    $receiver->fcm_token,
                    $sender->name ?? 'New Message',
                    $message,
                    [
                        'type' => 'chat_message',
                        'chat_id' => $chatId,
                        'sender_id' => $senderId,
                        'message_id' => $chatMessage->id,
                    ]
                );
            }

            return [
                'success' => true,
                'message' => $messageData,
                'localId' => $chatMessage->id,
            ];
        } catch (Exception $e) {
            Log::error("Firebase send message error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Mark messages as read
     */
    public function markAsRead($chatId, $userId, $messageIds)
    {
        try {
            // Update local database
            ChatMessage::whereIn('id', $messageIds)
                ->where('receiver_id', $userId)
                ->whereNull('read_at')
                ->update(['read_at' => now()]);

            return ['success' => true];
        } catch (Exception $e) {
            Log::error("Mark as read error: " . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update typing status
     */
    public function updateTypingStatus($chatId, $userId, $isTyping)
    {
        try {
            // This would update Firestore typing indicator
            // For now, just return success
            return [
                'success' => true,
                'chatId' => $chatId,
                'userId' => $userId,
                'isTyping' => $isTyping,
            ];
        } catch (Exception $e) {
            Log::error("Update typing status error: " . $e->getMessage());
            return ['success' => false];
        }
    }

    /**
     * Get chat messages (fallback if Firebase fails)
     */
    public function getChatMessages($userId1, $userId2, $limit = 50)
    {
        try {
            $messages = ChatMessage::where(function($query) use ($userId1, $userId2) {
                $query->where('sender_id', $userId1)
                      ->where('receiver_id', $userId2);
            })->orWhere(function($query) use ($userId1, $userId2) {
                $query->where('sender_id', $userId2)
                      ->where('receiver_id', $userId1);
            })
            ->with(['sender', 'receiver'])
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();

            return $messages->map(function($msg) {
                return [
                    'id' => $msg->id,
                    'senderId' => $msg->sender_id,
                    'receiverId' => $msg->receiver_id,
                    'text' => $msg->message,
                    'type' => $msg->message_type,
                    'imageUrl' => $msg->attachment_url,
                    'timestamp' => $msg->created_at->toIso8601String(),
                    'readBy' => $msg->read_at ? [$msg->sender_id, $msg->receiver_id] : [$msg->sender_id],
                    'sender' => [
                        'id' => $msg->sender->id,
                        'name' => $msg->sender->name,
                        'profileImage' => $msg->sender->profile_image,
                    ],
                ];
            });
        } catch (Exception $e) {
            Log::error("Get chat messages error: " . $e->getMessage());
            return collect([]);
        }
    }

    /**
     * Generate a consistent chat ID
     */
    protected function generateChatId($userId1, $userId2, $adId = null)
    {
        // Sort user IDs to ensure consistency
        $users = [$userId1, $userId2];
        sort($users);
        
        $baseId = implode('_', $users);
        
        if ($adId) {
            return "chat_{$baseId}_ad_{$adId}";
        }
        
        return "chat_{$baseId}";
    }

    /**
     * Delete a message
     */
    public function deleteMessage($messageId, $userId)
    {
        try {
            $message = ChatMessage::find($messageId);
            
            if (!$message) {
                throw new Exception("Message not found");
            }

            // Soft delete based on user role
            if ($message->sender_id == $userId) {
                $message->update(['deleted_by_sender' => true]);
            } elseif ($message->receiver_id == $userId) {
                $message->update(['deleted_by_receiver' => true]);
            }

            return ['success' => true];
        } catch (Exception $e) {
            Log::error("Delete message error: " . $e->getMessage());
            throw $e;
        }
    }
}
