<?php

namespace App\Services;

use App\Models\Ad;
use App\Models\AnalyticsEvent;
use App\Models\ChatMessage;
use App\Models\Notification as NotificationModel;
use App\Models\Transaction;
use App\Models\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class AdPostAnalyticsService
{
    /**
     * Get comprehensive ad post performance dashboard
     */
    public function getAdPostPerformanceDashboard($startDate = null, $endDate = null)
    {
        return [
            'ad_performance_overview' => $this->getAdPerformanceOverview($startDate, $endDate),
            'ad_engagement_metrics' => $this->getAdEngagementMetrics($startDate, $endDate),
            'ad_monetization_analytics' => $this->getAdMonetizationAnalytics($startDate, $endDate),
            'ad_conversion_funnels' => $this->getAdConversionFunnels($startDate, $endDate),
            'ad_timing_optimization' => $this->getAdTimingOptimization($startDate, $endDate),
            'ad_category_performance' => $this->getAdCategoryPerformance($startDate, $endDate),
            'ad_user_behavior' => $this->getAdUserBehavior($startDate, $endDate),
            'ad_competitive_analysis' => $this->getAdCompetitiveAnalysis($startDate, $endDate),
        ];
    }

    /**
     * Get Ad Performance Overview
     */
    private function getAdPerformanceOverview($startDate, $endDate)
    {
        $totalAds = Ad::when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
            return $query->whereBetween('created_at', [$startDate, $endDate]);
        })->count();

        $activeAds = Ad::where('status', 'active')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->count();

        $expiredAds = Ad::where('status', 'expired')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->count();

        $avgAdLifetime = $this->getAverageAdLifetime($startDate, $endDate);
        $adViews = $this->getAdViews($startDate, $endDate);
        $adClicks = $this->getAdClicks($startDate, $endDate);
        $adCTR = $adViews > 0 ? ($adClicks / $adViews) * 100 : 0;

        return [
            'total_ads' => $totalAds,
            'active_ads' => $activeAds,
            'expired_ads' => $expiredAds,
            'avg_ad_lifetime' => $avgAdLifetime,
            'total_views' => $adViews,
            'total_clicks' => $adClicks,
            'ctr' => $adCTR,
            'conversion_rate' => $this->getAdConversionRate($startDate, $endDate),
            'revenue_per_ad' => $this->getRevenuePerAd($startDate, $endDate),
        ];
    }

    /**
     * Get Ad Engagement Metrics
     */
    private function getAdEngagementMetrics($startDate, $endDate)
    {
        return [
            'message_engagement' => $this->getMessageEngagement($startDate, $endDate),
            'call_to_action_performance' => $this->getCallToActionPerformance($startDate, $endDate),
            'ad_interaction_patterns' => $this->getAdInteractionPatterns($startDate, $endDate),
            'user_engagement_depth' => $this->getUserEngagementDepth($startDate, $endDate),
            'ad_sharing_metrics' => $this->getAdSharingMetrics($startDate, $endDate),
        ];
    }

    /**
     * Get Ad Monetization Analytics
     */
    private function getAdMonetizationAnalytics($startDate, $endDate)
    {
        return [
            'ad_package_performance' => $this->getAdPackagePerformance($startDate, $endDate),
            'boost_feature_revenue' => $this->getBoostFeatureRevenue($startDate, $endDate),
            'premium_ad_performance' => $this->getPremiumAdPerformance($startDate, $endDate),
            'monetization_trends' => $this->getMonetizationTrends($startDate, $endDate),
            'roi_analysis' => $this->getROIAnalysis($startDate, $endDate),
        ];
    }

    /**
     * Get Ad Conversion Funnels
     */
    private function getAdConversionFunnels($startDate, $endDate)
    {
        return [
            'post_to_view_funnel' => $this->getPostToViewFunnel($startDate, $endDate),
            'view_to_engagement_funnel' => $this->getViewToEngagementFunnel($startDate, $endDate),
            'engagement_to_conversion_funnel' => $this->getEngagementToConversionFunnel($startDate, $endDate),
            'conversion_optimization' => $this->getConversionOptimization($startDate, $endDate),
        ];
    }

    /**
     * Get Ad Timing Optimization
     */
    private function getAdTimingOptimization($startDate, $endDate)
    {
        return [
            'optimal_posting_times' => $this->getOptimalPostingTimes($startDate, $endDate),
            'ad_lifetime_optimization' => $this->getAdLifetimeOptimization($startDate, $endDate),
            'seasonal_trends' => $this->getSeasonalTrends($startDate, $endDate),
            'peak_engagement_periods' => $this->getPeakEngagementPeriods($startDate, $endDate),
        ];
    }

    /**
     * Get Ad Category Performance
     */
    private function getAdCategoryPerformance($startDate, $endDate)
    {
        return [
            'category_popularity' => $this->getCategoryPopularity($startDate, $endDate),
            'category_conversion_rates' => $this->getCategoryConversionRates($startDate, $endDate),
            'category_revenue_analysis' => $this->getCategoryRevenueAnalysis($startDate, $endDate),
            'category_competition' => $this->getCategoryCompetition($startDate, $endDate),
        ];
    }

    /**
     * Get Ad User Behavior
     */
    private function getAdUserBehavior($startDate, $endDate)
    {
        return [
            'buyer_behavior_patterns' => $this->getBuyerBehaviorPatterns($startDate, $endDate),
            'seller_performance_metrics' => $this->getSellerPerformanceMetrics($startDate, $endDate),
            'user_retention_by_ad_interaction' => $this->getUserRetentionByAdInteraction($startDate, $endDate),
            'user_lifecycle_analysis' => $this->getUserLifecycleAnalysis($startDate, $endDate),
        ];
    }

    /**
     * Get Ad Competitive Analysis
     */
    private function getAdCompetitiveAnalysis($startDate, $endDate)
    {
        return [
            'market_positioning' => $this->getMarketPositioning($startDate, $endDate),
            'price_competition_analysis' => $this->getPriceCompetitionAnalysis($startDate, $endDate),
            'feature_adoption_rates' => $this->getFeatureAdoptionRates($startDate, $endDate),
            'competitive_intelligence' => $this->getCompetitiveIntelligence($startDate, $endDate),
        ];
    }

    // Helper Methods

    private function getAverageAdLifetime($startDate, $endDate)
    {
        // Calculate average time from ad creation to expiration
        $ads = Ad::when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
            return $query->whereBetween('created_at', [$startDate, $endDate]);
        })
        ->whereNotNull('expires_at')
        ->get();

        if ($ads->isEmpty()) return 0;

        $totalDays = $ads->sum(function ($ad) {
            return $ad->created_at->diffInDays($ad->expires_at);
        });

        return round($totalDays / $ads->count(), 1);
    }

    private function getAdViews($startDate, $endDate)
    {
        return AnalyticsEvent::where('event_type', 'ad_viewed')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->count();
    }

    private function getAdClicks($startDate, $endDate)
    {
        return AnalyticsEvent::where('event_type', 'ad_clicked')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->count();
    }

    private function getAdConversionRate($startDate, $endDate)
    {
        $adViews = $this->getAdViews($startDate, $endDate);
        $messagesSent = ChatMessage::when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
            return $query->whereBetween('created_at', [$startDate, $endDate]);
        })->count();

        return $adViews > 0 ? ($messagesSent / $adViews) * 100 : 0;
    }

    private function getRevenuePerAd($startDate, $endDate)
    {
        $totalRevenue = Transaction::where('status', 'completed')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->sum('amount') ?? 0;

        $totalAds = Ad::when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
            return $query->whereBetween('created_at', [$startDate, $endDate]);
        })->count();

        return $totalAds > 0 ? ($totalRevenue / $totalAds) : 0;
    }

    private function getMessageEngagement($startDate, $endDate)
    {
        $messages = ChatMessage::when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
            return $query->whereBetween('created_at', [$startDate, $endDate]);
        })->get();

        return [
            'total_messages' => $messages->count(),
            'messages_per_ad' => $this->getMessagesPerAd($messages),
            'response_rate' => $this->getResponseRate($messages),
            'avg_response_time' => $this->getAverageResponseTime($messages),
        ];
    }

    private function getMessagesPerAd($messages)
    {
        $uniqueAds = $messages->pluck('ad_id')->unique()->count();
        return $uniqueAds > 0 ? round($messages->count() / $uniqueAds, 2) : 0;
    }

    private function getResponseRate($messages)
    {
        // Calculate percentage of ads that received responses
        $adsWithResponses = $messages->groupBy('ad_id')->filter(function ($messages) {
            return $messages->count() > 1; // More than one message indicates conversation
        })->count();

        $totalAds = $messages->pluck('ad_id')->unique()->count();
        return $totalAds > 0 ? ($adsWithResponses / $totalAds) * 100 : 0;
    }

    private function getAverageResponseTime($messages)
    {
        // Calculate average time between first and second message
        $responseTimes = [];
        
        $messages->groupBy('ad_id')->each(function ($adMessages) use (&$responseTimes) {
            if ($adMessages->count() > 1) {
                $firstMessage = $adMessages->sortBy('created_at')->first();
                $secondMessage = $adMessages->sortBy('created_at')->skip(1)->first();
                
                if ($firstMessage && $secondMessage) {
                    $responseTimes[] = $firstMessage->created_at->diffInMinutes($secondMessage->created_at);
                }
            }
        });

        return count($responseTimes) > 0 ? round(array_sum($responseTimes) / count($responseTimes), 1) : 0;
    }

    private function getCallToActionPerformance($startDate, $endDate)
    {
        return [
            'call_button_clicks' => AnalyticsEvent::where('event_type', 'call_button_clicked')
                ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                    return $query->whereBetween('created_at', [$startDate, $endDate]);
                })
                ->count(),
            'whatsapp_button_clicks' => AnalyticsEvent::where('event_type', 'whatsapp_button_clicked')
                ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                    return $query->whereBetween('created_at', [$startDate, $endDate]);
                })
                ->count(),
            'website_button_clicks' => AnalyticsEvent::where('event_type', 'website_button_clicked')
                ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                    return $query->whereBetween('created_at', [$startDate, $endDate]);
                })
                ->count(),
        ];
    }

    private function getAdInteractionPatterns($startDate, $endDate)
    {
        return [
            'view_to_message_rate' => $this->getViewToMessageRate($startDate, $endDate),
            'time_to_engagement' => $this->getTimeToEngagement($startDate, $endDate),
            'repeat_visitor_rate' => $this->getRepeatVisitorRate($startDate, $endDate),
            'ad_sharing_rate' => $this->getAdSharingRate($startDate, $endDate),
        ];
    }

    private function getViewToMessageRate($startDate, $endDate)
    {
        $views = $this->getAdViews($startDate, $endDate);
        $messages = ChatMessage::when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
            return $query->whereBetween('created_at', [$startDate, $endDate]);
        })->count();

        return $views > 0 ? ($messages / $views) * 100 : 0;
    }

    private function getTimeToEngagement($startDate, $endDate)
    {
        // Calculate average time from ad view to first message
        $engagementTimes = [];

        // This would require more complex query to match views with subsequent messages
        // For now, return placeholder
        return 0;
    }

    private function getRepeatVisitorRate($startDate, $endDate)
    {
        // Calculate percentage of users who view the same ad multiple times
        $repeatVisitors = AnalyticsEvent::select('user_id', 'data->ad_id as ad_id')
            ->where('event_type', 'ad_viewed')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->whereNotNull('user_id')
            ->whereNotNull('data->ad_id')
            ->groupBy('user_id', 'data->ad_id')
            ->havingRaw('COUNT(*) > 1')
            ->count();

        $totalViews = $this->getAdViews($startDate, $endDate);
        
        return $totalViews > 0 ? ($repeatVisitors / $totalViews) * 100 : 0;
    }

    private function getAdSharingRate($startDate, $endDate)
    {
        return AnalyticsEvent::where('event_type', 'ad_shared')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->count();
    }

    private function getAdPackagePerformance($startDate, $endDate)
    {
        return Transaction::select('transaction_type', DB::raw('COUNT(*) as count, SUM(amount) as revenue'))
            ->where('status', 'completed')
            ->where('transaction_type', 'like', '%ad_%')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->groupBy('transaction_type')
            ->orderBy('revenue', 'desc')
            ->get();
    }

    private function getBoostFeatureRevenue($startDate, $endDate)
    {
        return Transaction::where('transaction_type', 'ad_boost')
            ->where('status', 'completed')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->sum('amount') ?? 0;
    }

    private function getPremiumAdPerformance($startDate, $endDate)
    {
        return [
            'featured_ads_revenue' => Transaction::where('transaction_type', 'ad_featured')
                ->where('status', 'completed')
                ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                    return $query->whereBetween('created_at', [$startDate, $endDate]);
                })
                ->sum('amount') ?? 0,
            'bumped_ads_revenue' => Transaction::where('transaction_type', 'ad_bumped')
                ->where('status', 'completed')
                ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                    return $query->whereBetween('created_at', [$startDate, $endDate]);
                })
                ->sum('amount') ?? 0,
            'premium_ad_conversion' => $this->getPremiumAdConversionRate($startDate, $endDate),
        ];
    }

    private function getPremiumAdConversionRate($startDate, $endDate)
    {
        // Calculate conversion rate for premium ads vs regular ads
        $premiumAds = Ad::where('is_featured', true)
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->count();

        $regularAds = Ad::where('is_featured', false)
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->count();

        return [
            'premium_ads_count' => $premiumAds,
            'regular_ads_count' => $regularAds,
            'premium_conversion_boost' => $premiumAds > 0 ? 1.5 : 1.0, // Placeholder
        ];
    }

    private function getOptimalPostingTimes($startDate, $endDate)
    {
        return AnalyticsEvent::selectRaw('HOUR(created_at) as hour, COUNT(*) as views')
            ->where('event_type', 'ad_viewed')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->groupBy('hour')
            ->orderBy('views', 'desc')
            ->limit(5)
            ->get();
    }

    private function getCategoryPopularity($startDate, $endDate)
    {
        return Ad::select('category_id', DB::raw('COUNT(*) as ad_count'))
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->groupBy('category_id')
            ->orderBy('ad_count', 'desc')
            ->limit(10)
            ->get();
    }

    private function getBuyerBehaviorPatterns($startDate, $endDate)
    {
        return [
            'avg_search_sessions' => $this->getAverageSearchSessions($startDate, $endDate),
            'avg_time_per_session' => $this->getAverageTimePerSession($startDate, $endDate),
            'favorite_categories' => $this->getFavoriteCategories($startDate, $endDate),
            'purchase_timing' => $this->getPurchaseTiming($startDate, $endDate),
        ];
    }

    private function getAverageSearchSessions($startDate, $endDate)
    {
        return AnalyticsEvent::select('user_id', DB::raw('COUNT(*) as sessions'))
            ->where('event_type', 'search_performed')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->whereNotNull('user_id')
            ->groupBy('user_id')
            ->avg('sessions') ?? 0;
    }

    private function getAverageTimePerSession($startDate, $endDate)
    {
        // Calculate average session duration for ad browsing
        return 5.2; // Placeholder in minutes
    }

    private function getFavoriteCategories($startDate, $endDate)
    {
        return AnalyticsEvent::select('data->category_id as category_id', DB::raw('COUNT(*) as searches'))
            ->where('event_type', 'search_performed')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->whereNotNull('data->category_id')
            ->groupBy('data->category_id')
            ->orderBy('searches', 'desc')
            ->limit(5)
            ->get();
    }

    private function getPurchaseTiming($startDate, $endDate)
    {
        return AnalyticsEvent::selectRaw('DAYOFWEEK(created_at) as day, COUNT(*) as purchases')
            ->where('event_type', 'purchase_completed')
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->groupBy('DAYOFWEEK(created_at)')
            ->orderBy('purchases', 'desc')
            ->get();
    }
}
