<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class UserTrustScore extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'score',
        'tier',
        'breakdown',
        'penalty_points',
        'last_calculated',
        'is_suspended',
        'suspension_until',
        'suspension_reason'
    ];

    protected $casts = [
        'breakdown' => 'array',
        'last_calculated' => 'datetime',
        'suspension_until' => 'datetime',
        'is_suspended' => 'boolean',
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    // Accessors
    public function getTierAttribute($value)
    {
        return $this->calculateTier($this->score);
    }

    // Methods
    public function calculateTier($score)
    {
        if ($score >= 85) return 'legend';
        if ($score >= 70) return 'elite';
        if ($score >= 50) return 'highly_trusted';
        if ($score >= 30) return 'trusted';
        return 'new';
    }

    public function isSuspended()
    {
        return $this->is_suspended && (!$this->suspension_until || now()->lt($this->suspension_until));
    }

    public function getTierColor()
    {
        switch ($this->tier) {
            case 'legend': return '#FFD700'; // Gold
            case 'elite': return '#4F46E5'; // Indigo
            case 'highly_trusted': return '#10B981'; // Emerald
            case 'trusted': return '#3B82F6'; // Blue
            case 'new': return '#6B7280'; // Gray
            default: return '#6B7280';
        }
    }

    public function getTierBadge()
    {
        switch ($this->tier) {
            case 'legend': return '🏆 Legend';
            case 'elite': return '👑 Elite';
            case 'highly_trusted': return '⭐ Highly Trusted';
            case 'trusted': return '✅ Trusted';
            case 'new': return '🆕 New';
            default: return '❓';
        }
    }

    // Scopes
    public function scopeSuspended($query)
    {
        return $query->where('is_suspended', true)
                    ->where(function($q) {
                        $q->whereNull('suspension_until')
                          ->orWhere('suspension_until', '>', now());
                    });
    }

    public function scopeByTier($query, $tier)
    {
        return $query->where('tier', $tier);
    }

    public function scopeHighRisk($query)
    {
        return $query->where('score', '<', 30);
    }

    public function scopeLowRisk($query)
    {
        return $query->where('score', '>=', 70);
    }
}
