<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class UserPlan extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        // Basic Plan Information
        'plan_name',
        'price',
        'price_multiplier',
        'duration_days',
        'city_multiplier',
        'post_limit',
        'lead_limit',
        'contact_limit',
        'analytics_tier',
        'validity_days',
        'ad_free',
        'has_analytics',
        'verified_badge',
        'vip_badge',
        'badge_style',
        'theme',
        'halo_effect',
        'ad_border',
        'status',

        // Billing Cycle Support
        'monthly_price',
        'quarterly_price',
        'half_yearly_price',
        'yearly_price',
        'billing_cycle_months',

        // Discount Support
        'discount_percentage',
        'discount_fixed_amount',
        'discount_start_date',
        'discount_end_date',

        // Plan Features for Different Cycles
        'enable_monthly',
        'enable_quarterly',
        'enable_half_yearly',
        'enable_yearly',

        // Promotional Text
        'promotion_text',
        'savings_text',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        // Basic Fields
        'price' => 'decimal:2',
        'price_multiplier' => 'decimal:2',
        'city_multiplier' => 'decimal:2',
        'post_limit' => 'integer',
        'lead_limit' => 'integer',
        'validity_days' => 'integer',
        'ad_free' => 'boolean',
        'has_analytics' => 'boolean',
        'verified_badge' => 'boolean',
        'vip_badge' => 'boolean',
        'status' => 'string',

        // Billing Cycle Fields
        'monthly_price' => 'decimal:2',
        'quarterly_price' => 'decimal:2',
        'half_yearly_price' => 'decimal:2',
        'yearly_price' => 'decimal:2',
        'billing_cycle_months' => 'integer',

        // Discount Fields
        'discount_percentage' => 'decimal:2',
        'discount_fixed_amount' => 'decimal:2',
        'discount_start_date' => 'datetime',
        'discount_end_date' => 'datetime',

        // Feature Flags
        'enable_monthly' => 'boolean',
        'enable_quarterly' => 'boolean',
        'enable_half_yearly' => 'boolean',
        'enable_yearly' => 'boolean',
    ];

    /**
     * Get the users with this plan.
     */
    public function users()
    {
        return $this->hasMany(User::class, 'current_plan_id');
    }

    /**
     * Get the subscriptions for this plan.
     */
    public function subscriptions()
    {
        return $this->hasMany(Subscription::class);
    }

    /**
     * Check if the plan is active
     */
    public function isActive(): bool
    {
        return $this->status === 'Active';
    }

    /**
     * Get the plan features as an array
     */
    public function getFeatures(): array
    {
        return [
            'ad_free' => $this->ad_free,
            'analytics' => $this->has_analytics,
            'verified_badge' => $this->verified_badge,
            'analytics_tier' => $this->analytics_tier,
            'post_limit' => $this->post_limit,
            'lead_limit' => $this->lead_limit,
        ];
    }

    /**
     * Get the plan UI configuration
     */
    public function getUiConfig(): array
    {
        return [
            'theme' => $this->theme,
            'halo_effect' => $this->halo_effect,
            'ad_border' => $this->ad_border,
            'badge_style' => $this->badge_style,
        ];
    }

    /**
     * Get pricing for a specific billing cycle
     */
    public function getPriceForCycle(string $cycle): float
    {
        switch ($cycle) {
            case 'monthly':
                return $this->monthly_price ?? $this->price;
            case 'quarterly':
                return $this->quarterly_price ?? $this->price;
            case 'half_yearly':
                return $this->half_yearly_price ?? $this->price;
            case 'yearly':
                return $this->yearly_price ?? $this->price;
            default:
                return $this->price;
        }
    }

    /**
     * Get all available billing cycles
     */
    public function getAvailableCycles(): array
    {
        $cycles = [];
        
        if ($this->enable_monthly) {
            $cycles['monthly'] = [
                'name' => 'Monthly',
                'price' => $this->getPriceForCycle('monthly'),
                'months' => 1,
                'label' => 'per month'
            ];
        }
        
        if ($this->enable_quarterly) {
            $cycles['quarterly'] = [
                'name' => 'Quarterly',
                'price' => $this->getPriceForCycle('quarterly'),
                'months' => 3,
                'label' => 'per 3 months'
            ];
        }
        
        if ($this->enable_half_yearly) {
            $cycles['half_yearly'] = [
                'name' => 'Half Yearly',
                'price' => $this->getPriceForCycle('half_yearly'),
                'months' => 6,
                'label' => 'per 6 months'
            ];
        }
        
        if ($this->enable_yearly) {
            $cycles['yearly'] = [
                'name' => 'Yearly',
                'price' => $this->getPriceForCycle('yearly'),
                'months' => 12,
                'label' => 'per year'
            ];
        }

        return $cycles;
    }

    /**
     * Calculate savings for a billing cycle compared to monthly
     */
    public function calculateSavings(string $cycle): array
    {
        $monthlyPrice = $this->getPriceForCycle('monthly');
        $cyclePrice = $this->getPriceForCycle($cycle);
        $cycleInfo = $this->getAvailableCycles()[$cycle] ?? null;

        if (!$cycleInfo) {
            return ['savings' => 0, 'percentage' => 0, 'total_cost' => 0];
        }

        $totalMonthlyCost = $monthlyPrice * $cycleInfo['months'];
        $totalCycleCost = $cyclePrice * $cycleInfo['months'];
        $savings = $totalMonthlyCost - $totalCycleCost;
        $percentage = $totalMonthlyCost > 0 ? ($savings / $totalMonthlyCost) * 100 : 0;

        return [
            'savings' => round($savings, 2),
            'percentage' => round($percentage, 2),
            'total_cost' => round($totalCycleCost, 2),
            'monthly_equivalent' => round($totalMonthlyCost, 2)
        ];
    }

    /**
     * Get the best value billing cycle
     */
    public function getBestValueCycle(): ?array
    {
        $cycles = $this->getAvailableCycles();
        $bestValue = null;
        $highestSavings = 0;

        foreach ($cycles as $cycleKey => $cycle) {
            if ($cycleKey === 'monthly') continue;

            $savings = $this->calculateSavings($cycleKey);
            if ($savings['percentage'] > $highestSavings) {
                $highestSavings = $savings['percentage'];
                $bestValue = [
                    'cycle' => $cycleKey,
                    'cycle_info' => $cycle,
                    'savings' => $savings
                ];
            }
        }

        return $bestValue;
    }

    /**
     * Check if discount is currently active
     */
    public function isDiscountActive(): bool
    {
        if ($this->discount_percentage <= 0 && $this->discount_fixed_amount <= 0) {
            return false;
        }

        if ($this->discount_start_date && $this->discount_end_date) {
            $now = now();
            return $now->between($this->discount_start_date, $this->discount_end_date);
        }

        return true;
    }

    /**
     * Get discount information
     */
    public function getDiscountInfo(): array
    {
        return [
            'is_active' => $this->isDiscountActive(),
            'percentage' => $this->discount_percentage,
            'fixed_amount' => $this->discount_fixed_amount,
            'start_date' => $this->discount_start_date,
            'end_date' => $this->discount_end_date,
            'promotion_text' => $this->promotion_text,
            'savings_text' => $this->savings_text
        ];
    }

    /**
     * Get plan comparison data for frontend display
     */
    public function getComparisonData(): array
    {
        $cycles = $this->getAvailableCycles();
        $comparison = [];

        foreach ($cycles as $cycleKey => $cycle) {
            $savings = $this->calculateSavings($cycleKey);
            
            $comparison[$cycleKey] = [
                'name' => $cycle['name'],
                'price' => $cycle['price'],
                'months' => $cycle['months'],
                'label' => $cycle['label'],
                'savings' => $savings,
                'is_best_value' => $cycleKey === ($this->getBestValueCycle()['cycle'] ?? null)
            ];
        }

        return [
            'plan_name' => $this->plan_name,
            'base_price' => $this->price,
            'cycles' => $comparison,
            'best_value' => $this->getBestValueCycle(),
            'discount' => $this->getDiscountInfo(),
            'features' => $this->getFeatures()
        ];
    }

    /**
     * Scope to get active plans only
     */
    public function scopeActive($query)
    {
        return $query->where('status', 'Active');
    }

    /**
     * Scope to get plans with billing cycles enabled
     */
    public function scopeWithBillingCycles($query)
    {
        return $query->where(function($q) {
            $q->where('enable_monthly', true)
              ->orWhere('enable_quarterly', true)
              ->orWhere('enable_half_yearly', true)
              ->orWhere('enable_yearly', true);
        });
    }
}
