<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class UserNotificationPreference extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'enabled',
        'plan_notifications',
        'category_notifications',
        'location_notifications',
        'deep_link_enabled',
        'blocked_categories',
        'blocked_locations',
        'quiet_hours_start',
        'quiet_hours_end'
    ];

    protected $casts = [
        'enabled' => 'boolean',
        'plan_notifications' => 'boolean',
        'category_notifications' => 'boolean',
        'location_notifications' => 'boolean',
        'deep_link_enabled' => 'boolean',
        'blocked_categories' => 'array',
        'blocked_locations' => 'array',
        'quiet_hours_start' => 'datetime:H:i',
        'quiet_hours_end' => 'datetime:H:i',
    ];

    /**
     * Get the user that owns the notification preference.
     */
    public function user(): BelongsTo
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Check if user has notifications enabled.
     */
    public function isEnabled(): bool
    {
        return $this->enabled;
    }

    /**
     * Check if user allows plan notifications.
     */
    public function allowsPlanNotifications(): bool
    {
        return $this->plan_notifications;
    }

    /**
     * Check if user allows category notifications.
     */
    public function allowsCategoryNotifications(): bool
    {
        return $this->category_notifications;
    }

    /**
     * Check if user allows location notifications.
     */
    public function allowsLocationNotifications(): bool
    {
        return $this->location_notifications;
    }

    /**
     * Check if user has deep linking enabled.
     */
    public function hasDeepLinkingEnabled(): bool
    {
        return $this->deep_link_enabled;
    }

    /**
     * Check if user has blocked a specific category.
     */
    public function isCategoryBlocked(string $categoryId): bool
    {
        return $this->blocked_categories && in_array($categoryId, $this->blocked_categories);
    }

    /**
     * Check if user has blocked a specific location.
     */
    public function isLocationBlocked(string $location): bool
    {
        return $this->blocked_locations && in_array($location, $this->blocked_locations);
    }

    /**
     * Check if current time is within quiet hours.
     */
    public function isQuietHours(): bool
    {
        if (!$this->quiet_hours_start || !$this->quiet_hours_end) {
            return false;
        }

        $currentTime = now()->format('H:i');
        $startTime = $this->quiet_hours_start->format('H:i');
        $endTime = $this->quiet_hours_end->format('H:i');

        if ($startTime < $endTime) {
            return $currentTime >= $startTime && $currentTime <= $endTime;
        } else {
            // Overnight quiet hours (e.g., 22:00 to 08:00)
            return $currentTime >= $startTime || $currentTime <= $endTime;
        }
    }

    /**
     * Check if user should receive a notification based on preferences.
     */
    public function shouldReceiveNotification(string $notificationType, ?string $categoryId = null, ?string $location = null): bool
    {
        if (!$this->isEnabled()) {
            return false;
        }

        if ($this->isQuietHours()) {
            return false;
        }

        // Check specific notification type preferences
        switch ($notificationType) {
            case 'plan':
                return $this->allowsPlanNotifications();
            case 'category':
                return $this->allowsCategoryNotifications() && !$this->isCategoryBlocked($categoryId ?? '');
            case 'location':
                return $this->allowsLocationNotifications() && !$this->isLocationBlocked($location ?? '');
            default:
                return true;
        }
    }
}
