<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class SafetyAction extends Model
{
    use HasFactory;

    protected $fillable = [
        'user_id',
        'ad_id',
        'report_id',
        'action_type',
        'reason',
        'auto_generated',
        'admin_id',
        'action_data',
        'expires_at',
        'is_active'
    ];

    protected $casts = [
        'action_data' => 'array',
        'expires_at' => 'datetime',
        'auto_generated' => 'boolean',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function ad()
    {
        return $this->belongsTo(Ad::class);
    }

    public function report()
    {
        return $this->belongsTo(SafetyReport::class);
    }

    public function admin()
    {
        return $this->belongsTo(User::class, 'admin_id');
    }

    // Methods
    public function isActive()
    {
        if (!$this->is_active) return false;
        if ($this->expires_at && now()->gt($this->expires_at)) {
            $this->is_active = false;
            $this->save();
            return false;
        }
        return true;
    }

    public function getActionTypeLabel()
    {
        $labels = [
            'auto_suspend' => 'Auto Suspension',
            'manual_suspend' => 'Manual Suspension',
            'auto_delete' => 'Auto Delete',
            'warning' => 'Warning',
            'restriction' => 'Feature Restriction',
            'manual_review' => 'Manual Review Required',
            'appeal_approved' => 'Appeal Approved',
            'appeal_rejected' => 'Appeal Rejected'
        ];
        
        return $labels[$this->action_type] ?? ucfirst(str_replace('_', ' ', $this->action_type));
    }

    public function getActionColor()
    {
        $colors = [
            'auto_suspend' => 'danger',
            'manual_suspend' => 'danger',
            'auto_delete' => 'danger',
            'warning' => 'warning',
            'restriction' => 'info',
            'manual_review' => 'primary',
            'appeal_approved' => 'success',
            'appeal_rejected' => 'danger'
        ];
        
        return $colors[$this->action_type] ?? 'secondary';
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true)
                    ->where(function($q) {
                        $q->whereNull('expires_at')
                          ->orWhere('expires_at', '>', now());
                    });
    }

    public function scopeAutoGenerated($query)
    {
        return $query->where('auto_generated', true);
    }

    public function scopeManual($query)
    {
        return $query->where('auto_generated', false);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('action_type', $type);
    }

    public function scopeRecent($query, $days = 30)
    {
        return $query->where('created_at', '>=', now()->subDays($days));
    }
}
