<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class PaymentGateway extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'name',
        'slug',
        'api_key',
        'api_secret',
        'webhook_secret',
        'is_active',
        'is_test_mode',
        'supported_currencies',
        'supported_methods',
        'transaction_fee_percent',
        'transaction_fee_fixed',
        'description',
        'sort_order',
    ];

    protected $casts = [
        'supported_currencies' => 'array',
        'supported_methods' => 'array',
        'is_active' => 'boolean',
        'is_test_mode' => 'boolean',
        'transaction_fee_percent' => 'decimal:2',
        'transaction_fee_fixed' => 'decimal:2',
    ];

    public function paymentMethods()
    {
        return $this->hasMany(PaymentMethod::class, 'gateway_slug', 'slug');
    }

    public function transactions()
    {
        return $this->hasMany(Transaction::class, 'gateway_slug', 'slug');
    }

    public function isActive()
    {
        return $this->is_active && !$this->trashed();
    }

    public function isTestMode()
    {
        return $this->is_test_mode;
    }

    public function getApiKey()
    {
        return $this->isTestMode() ? $this->test_api_key : $this->api_key;
    }

    public function getApiSecret()
    {
        return $this->isTestMode() ? $this->test_api_secret : $this->api_secret;
    }

    public function getSupportedCurrencies()
    {
        return $this->supported_currencies ?: ['INR'];
    }

    public function getSupportedMethods()
    {
        return $this->supported_methods ?: ['credit_card', 'debit_card', 'upi', 'net_banking'];
    }

    public function calculateFee($amount)
    {
        $percentageFee = ($amount * $this->transaction_fee_percent) / 100;
        return $percentageFee + $this->transaction_fee_fixed;
    }
}
