<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

use Illuminate\Database\Eloquent\SoftDeletes;

class Banner extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'user_id',
        'package_id',
        'title',
        'image_url',
        'redirect_type',
        'redirect_value',
        'billing_model',
        'target_location_type',
        'target_countries',
        'target_states',
        'target_cities',
        'target_latitude',
        'target_longitude',
        'target_radius',
        'placement_screens', // Add this
        'total_budget',
        'cost_per_impression',
        'cost_per_click',
        'spent_amount',
        'total_impressions',
        'total_clicks',
        'ctr',
        'goal_type',
        'goal_impressions',
        'goal_clicks',
        'goal_budget',
        'start_date',
        'end_date',
        'status',
        'admin_notes',
        'is_admin_created',
        'priority'
    ];

    protected $casts = [
        'target_countries' => 'array',
        'target_states' => 'array',
        'target_cities' => 'array',
        'placement_screens' => 'array', // Add this
        'target_latitude' => 'float',
        'target_longitude' => 'float',
        'target_radius' => 'float',
        'total_budget' => 'decimal:2',
        'cost_per_impression' => 'decimal:6',
        'cost_per_click' => 'decimal:2',
        'spent_amount' => 'decimal:2',
        'ctr' => 'decimal:2',
        'goal_budget' => 'decimal:2',
        'start_date' => 'datetime',
        'end_date' => 'datetime',
        'is_admin_created' => 'boolean'
    ];

    // Relationships
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function package()
    {
        return $this->belongsTo(BannerPackage::class, 'package_id');
    }

    public function impressions()
    {
        return $this->hasMany(BannerImpression::class);
    }

    public function clicks()
    {
        return $this->hasMany(BannerClick::class);
    }

    public function transactions()
    {
        return $this->hasMany(BannerTransaction::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeForLocation($query, $country, $state = null, $city = null)
    {
        return $query->where(function ($q) use ($country, $state, $city) {
            $q->where('target_location_type', 'all')
              ->orWhere(function ($q2) use ($country) {
                  $q2->where('target_location_type', 'country')
                     ->whereJsonContains('target_countries', $country);
              })
              ->orWhere(function ($q2) use ($state) {
                  if ($state) {
                      $q2->where('target_location_type', 'state')
                         ->whereJsonContains('target_states', $state);
                  }
              })
              ->orWhere(function ($q2) use ($city) {
                  if ($city) {
                      $q2->where('target_location_type', 'city')
                         ->whereJsonContains('target_cities', $city);
                  }
              });
        });
    }

    public function scopeByPriority($query)
    {
        return $query->orderBy('priority', 'desc');
    }

    /**
     * Get the full image URL.
     */
    public function getImageUrlAttribute($value)
    {
        if ($value && !str_starts_with($value, 'http')) {
            return url($value);
        }
        return $value;
    }
}
