<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class AdPostStatistics extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'ad_id',
        'total_views',
        'unique_views',
        'total_impressions',
        'unique_impressions',
        'total_clicks',
        'unique_clicks',
        'conversion_rate',
        'bounce_rate',
        'avg_time_on_page',
        'last_viewed_at',
        'last_clicked_at',
        'view_count_today',
        'view_count_week',
        'view_count_month',
        'click_count_today',
        'click_count_week',
        'click_count_month',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'total_views' => 'integer',
        'unique_views' => 'integer',
        'total_impressions' => 'integer',
        'unique_impressions' => 'integer',
        'total_clicks' => 'integer',
        'unique_clicks' => 'integer',
        'conversion_rate' => 'decimal:4',
        'bounce_rate' => 'decimal:4',
        'avg_time_on_page' => 'decimal:2',
        'view_count_today' => 'integer',
        'view_count_week' => 'integer',
        'view_count_month' => 'integer',
        'click_count_today' => 'integer',
        'click_count_week' => 'integer',
        'click_count_month' => 'integer',
        'last_viewed_at' => 'datetime',
        'last_clicked_at' => 'datetime',
    ];

    /**
     * Get the ad that owns the statistics.
     */
    public function ad()
    {
        return $this->belongsTo(Ad::class);
    }

    /**
     * Increment view counts
     */
    public function incrementViews($isUnique = false)
    {
        $this->increment('total_views');
        if ($isUnique) {
            $this->increment('unique_views');
        }
        
        // Increment daily/weekly/monthly counters
        $this->increment('view_count_today');
        $this->increment('view_count_week');
        $this->increment('view_count_month');
        
        $this->update(['last_viewed_at' => now()]);
    }

    /**
     * Increment impression counts
     */
    public function incrementImpressions($isUnique = false)
    {
        $this->increment('total_impressions');
        if ($isUnique) {
            $this->increment('unique_impressions');
        }
    }

    /**
     * Increment click counts
     */
    public function incrementClicks($isUnique = false)
    {
        $this->increment('total_clicks');
        if ($isUnique) {
            $this->increment('unique_clicks');
        }
        
        // Increment daily/weekly/monthly counters
        $this->increment('click_count_today');
        $this->increment('click_count_week');
        $this->increment('click_count_month');
        
        $this->update(['last_clicked_at' => now()]);
    }

    /**
     * Reset daily counters (called by scheduled task)
     */
    public function resetDailyCounters()
    {
        $this->update([
            'view_count_today' => 0,
            'click_count_today' => 0,
        ]);
    }

    /**
     * Reset weekly counters (called by scheduled task)
     */
    public function resetWeeklyCounters()
    {
        $this->update([
            'view_count_week' => 0,
            'click_count_week' => 0,
        ]);
    }

    /**
     * Reset monthly counters (called by scheduled task)
     */
    public function resetMonthlyCounters()
    {
        $this->update([
            'view_count_month' => 0,
            'click_count_month' => 0,
        ]);
    }

    /**
     * Calculate conversion rate
     */
    public function calculateConversionRate()
    {
        if ($this->total_impressions > 0) {
            $this->conversion_rate = ($this->total_clicks / $this->total_impressions) * 100;
            $this->save();
        }
    }

    /**
     * Calculate bounce rate (simplified calculation)
     */
    public function calculateBounceRate()
    {
        // Simplified bounce rate calculation
        // In a real implementation, this would need more sophisticated tracking
        if ($this->total_views > 0) {
            $this->bounce_rate = 0; // Placeholder calculation
            $this->save();
        }
    }
}
