<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class AdPackage extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'slug',
        'type',
        'description',
        'features',
        'algorithm_params',
        'is_active',
        'sort_order',
        'terms',
        'icon',
    ];

    protected $appends = ['icon_url'];

    protected $casts = [
        'features' => 'array',
        'algorithm_params' => 'array',
        'is_active' => 'boolean',
    ];

    // Relationships
    public function pricingTiers()
    {
        return $this->hasMany(PackagePricingTier::class, 'package_id');
    }

    public function activePricingTiers()
    {
        return $this->hasMany(PackagePricingTier::class, 'package_id')->where('is_active', true);
    }

    public function purchases()
    {
        return $this->hasMany(AdPackagePurchase::class, 'package_id');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('is_active', true);
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('sort_order')->orderBy('name');
    }

    // Helper methods
    public function isActive()
    {
        return $this->is_active;
    }

    public function getFeature($key, $default = null)
    {
        return $this->features[$key] ?? $default;
    }

    public function getAlgorithmParam($key, $default = null)
    {
        return $this->algorithm_params[$key] ?? $default;
    }

    // Package type checks
    public function isMultipleAds()
    {
        return $this->type === 'multiple_ads';
    }

    public function isFeatured()
    {
        return $this->type === 'featured';
    }

    public function isBoost()
    {
        return $this->type === 'boost';
    }

    public function isCombo()
    {
        return $this->type === 'combo';
    }

    public function getIconUrlAttribute()
    {
        if ($this->icon) {
            return asset('storage/' . $this->icon);
        }
        return null;
    }
}
