<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Ad extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'user_id',
        'buyer_id',
        'category_id',
        'subcategory_id',
        'title',
        'description',
        'price',
        'currency',
        'city', // Added to support Seeder
        'state', // Added to support Seeder
        'location_city',
        'location_state',
        'location_country',
        'location_address',
        'location_latitude',
        'location_longitude',
        'latitude',  // Legacy column support
        'longitude', // Legacy column support
        'status',
        'ad_type', // Replaces is_featured/boosted logic partially or works with it
        'is_featured',
        'is_boosted',
        'last_bumped_at',
        'views',
        'impressions',
        'clicks',
        'expires_at',
        'approved_at',
        'approved_by',
        'rejected_at',
        'rejected_by',
        'rejection_reason',
        'deactivated_at',
        'deactivated_by',
        'created_at', // Allow manual create date for testing/import
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'price' => 'decimal:2',
        'location_latitude' => 'decimal:7',
        'location_longitude' => 'decimal:7',
        'views' => 'integer',
        'impressions' => 'integer',
        'clicks' => 'integer',
        'is_featured' => 'boolean',
        'is_boosted' => 'boolean',
        'last_bumped_at' => 'datetime',
    ];

    /**
     * Get the user that owns the ad.
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function buyer()
    {
        return $this->belongsTo(User::class, 'buyer_id');
    }

    /**
     * Get the packages associated with the ad.
     */
    public function packages()
    {
        return $this->belongsToMany(AdPackage::class, 'ad_package_purchases', 'ad_id', 'ad_package_id')
                    ->withPivot(['purchase_id', 'status', 'start_date', 'end_date', 'features_applied'])
                    ->withTimestamps();
    }

    /**
     * Get the custom attribute values for the ad.
     */
    public function attributes()
    {
        return $this->hasMany(AdAttributeValue::class);
    }

    /**
     * Get the category that owns the ad.
     */
    public function category()
    {
        return $this->belongsTo(Category::class);
    }

    /**
     * Get the subcategory that owns the ad.
     */
    public function subcategory()
    {
        return $this->belongsTo(Subcategory::class);
    }

    /**
     * Get the images for the ad.
     */
    public function images()
    {
        return $this->hasMany(AdImage::class);
    }

    /**
     * Get the analytics for the ad.
     */
    public function analytics()
    {
        return $this->hasMany(AdAnalytics::class);
    }

    /**
     * Get the post statistics for the ad.
     */
    public function postStatistics()
    {
        return $this->hasOne(AdPostStatistics::class);
    }

    /**
     * Get the reports for the ad.
     */
    public function reports()
    {
        return $this->hasMany(AdReport::class);
    }

    /**
     * Get the users who favorited this ad.
     */
    public function favoritedBy()
    {
        return $this->belongsToMany(User::class, 'favorites', 'ad_id', 'user_id')->withTimestamps();
    }

    /**
     * Scope a query to only include active ads.
     */
    public function scopeActive($query)
    {
        return $query->where('ads.status', 'Active');
    }

    /**
     * Scope a query to only include featured ads.
     */
    public function scopeFeatured($query)
    {
        return $query->where('ads.is_featured', true);
    }

    /**
     * Scope a query to only include boosted ads.
     */
    public function scopeBoosted($query)
    {
        return $query->where('ads.is_boosted', true);
    }

    /**
     * Scope a query to filter by category.
     */
    public function scopeByCategory($query, $categoryId)
    {
        return $query->where('ads.category_id', $categoryId);
    }

    /**
     * Scope a query to filter by subcategory.
     */
    public function scopeBySubcategory($query, $subcategoryId)
    {
        return $query->where('ads.subcategory_id', $subcategoryId);
    }

    /**
     * Scope a query to filter by price range.
     */
    public function scopeByPriceRange($query, $minPrice, $maxPrice)
    {
        return $query->whereBetween('price', [$minPrice, $maxPrice]);
    }

    /**
     * Scope a query to filter by location radius.
     */
    public function scopeNearLocation($query, $latitude, $longitude, $radiusKm = 50)
    {
        // Haversine formula for distance calculation using COALESCE to support both legacy and new coordinate columns
        $haversine = "(6371 * acos(cos(radians($latitude)) * cos(radians(COALESCE(ads.location_latitude, ads.latitude, 0))) * cos(radians(COALESCE(ads.location_longitude, ads.longitude, 0)) - radians($longitude)) + sin(radians($latitude)) * sin(radians(COALESCE(ads.location_latitude, ads.latitude, 0)))))";
        
        return $query->whereRaw("$haversine <= ?", [$radiusKm])
                     ->whereNotNull([
                        'ads.latitude', 
                        'ads.longitude',
                        'ads.location_latitude',
                        'ads.location_longitude'
                     ], 'or'); // Matches if either set is present
    }

    /**
     * Scope a query to order by user interest and location.
     */
    /**
     * Scope a query to order by user interest and location using Weighted Algorithm.
     */
    public function scopeOrderByInterestAndLocation($query, $userLatitude = null, $userLongitude = null)
    {
        // 1. Get Algorithm Configuration
        $config = \App\Models\AlgorithmConfig::getDefault();
        $weights = $config ? $config->weights : [
            'location_relevance' => 80,
            'freshness_factor' => 40,
            'ad_performance' => 90,
        ];
        $vipSettings = ($config && isset($config->vip_settings['vip_tiers'])) ? $config->vip_settings['vip_tiers'] : [
            'platinum_boost' => 100,
            'gold_boost' => 75,
            'silver_boost' => 50,
            'free_boost' => 0,
        ];

        // Normalize weights
        $wLocation = ($weights['location_relevance'] ?? 50);
        $wFreshness = ($weights['freshness_factor'] ?? 50);
        $wPerformance = ($weights['ad_performance'] ?? 50);

        // VIP Boost Weights
        $bPlatinum = $vipSettings['platinum_boost'] ?? 100;
        $bGold = $vipSettings['gold_boost'] ?? 75;
        $bSilver = $vipSettings['silver_boost'] ?? 50;
        $bFree = $vipSettings['free_boost'] ?? 0;

        // 2. Joins for VIP Logic
        // We join users and plans to get the plan_name
        $query->leftJoin('users', 'ads.user_id', '=', 'users.id')
              ->leftJoin('user_plans', 'users.current_plan_id', '=', 'user_plans.id');

        // 3. Build SQL for Ranking Score
        $scoreComponents = [];
        $bindings = [];

        // A. Performance Component (Featured/Boosted)
        // Note: specifying table `ads` for columns to prevent ambiguity
        $scoreComponents[] = "(CASE WHEN ads.ad_type = 'featured' THEN ? ELSE 0 END + CASE WHEN ads.ad_type = 'premium' THEN ? ELSE 0 END)";
        $bindings[] = $wPerformance * 10; 
        $bindings[] = $wPerformance * 5;

        // B. Freshness Component
        $scoreComponents[] = "((UNIX_TIMESTAMP(ads.created_at) / 86400) * ?)"; 
        $bindings[] = $wFreshness / 5; 

        // C. Location Component
        if ($userLatitude !== null && $userLongitude !== null) {
            $haversine = "(6371 * acos(cos(radians(?)) * cos(radians(COALESCE(ads.location_latitude, ads.latitude, 0))) * cos(radians(COALESCE(ads.location_longitude, ads.longitude, 0)) - radians(?)) + sin(radians(?)) * sin(radians(COALESCE(ads.location_latitude, ads.latitude, 0)))))";
             
            $scoreComponents[] = "COALESCE((-1 * $haversine * ?), 0)";
            $bindings[] = $userLatitude;
            $bindings[] = $userLongitude;
            $bindings[] = $userLatitude;
            $bindings[] = $wLocation / 10;
        }

        // D. [NEW] Dynamic VIP Plan Priority
        $vipCaseStatements = [];
        $vipBindings = [];
        
        if (is_array($vipSettings)) {
            foreach ($vipSettings as $key => $value) {
                $planId = null;
                
                // 1. Check for dynamic key: plan_1_boost
                if (preg_match('/^plan_(\d+)_boost$/', $key, $matches)) {
                    $planId = $matches[1];
                } 
                // 2. Fallback: Map legacy keys to known IDs (Quick Fix for Default Config)
                else if ($key === 'platinum_boost') $planId = 9; // VIP Pro
                else if ($key === 'gold_boost') $planId = 8;     // VIP Seller
                else if ($key === 'silver_boost') $planId = 7;   // VIP Buyer
                else if ($key === 'free_boost') $planId = 6;     // Standard

                if ($planId) {
                    $vipCaseStatements[] = "WHEN users.current_plan_id = ? THEN ?";
                    $vipBindings[] = $planId;
                    $vipBindings[] = (float)$value * 10;
                }
            }
        }

        if (!empty($vipCaseStatements)) {
            $vipSql = implode(' ', $vipCaseStatements);
            $scoreComponents[] ="(CASE $vipSql ELSE 0 END)";
            // Merge bindings safely
            foreach ($vipBindings as $vb) {
                $bindings[] = $vb;
            }
        } else {
             $scoreComponents[] = "0";
        }

        // E. [NEW] Dynamic Ad Package Boosts
        // This sums up the 'boost' parameter from all active packages purchased for this ad
        // We use a subquery to avoid duplicate rows and handle stacking boosts
        $scoreComponents[] = "COALESCE((
            SELECT SUM(CAST(JSON_EXTRACT(p.algorithm_params, '$.boost') AS DECIMAL))
            FROM ad_package_purchases app
            JOIN ad_packages p ON app.package_id = p.id
            WHERE app.ad_id = ads.id 
            AND app.status = 'active' 
            AND app.expires_at > ?
        ), 0) * 10";
        $bindings[] = now()->toDateTimeString();

        // 4. Assemble Query
        $scoreSql = implode(' + ', $scoreComponents);
        
        return $query->select('ads.*') // Select ads.* to avoid pulling user/plan columns into model attributes
                     ->selectRaw("($scoreSql) as relevance_score", $bindings)
                     ->orderByDesc('relevance_score');
    }

    /**
     * Increment view count
     */
    public function incrementViews()
    {
        $this->increment('views');
        $this->postStatistics()->increment('total_views');
    }

    /**
     * Increment impression count
     */
    public function incrementImpressions()
    {
        $this->increment('impressions');
    }

    /**
     * Increment click count
     */
    public function incrementClicks()
    {
        $this->increment('clicks');
    }

    /**
     * Bump the ad (update last_bumped_at)
     */
    public function bump()
    {
        $this->update(['last_bumped_at' => now()]);
    }

    /**
     * Check if ad can be bumped again
     */
    public function canBump(): bool
    {
        // Implement bump cooldown logic if needed
        return true;
    }
}
