<?php

namespace App\Jobs;

use App\Models\User;
use App\Models\UserTrustScore;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class UpdateTrustScores implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Execute the job.
     */
    public function handle()
    {
        Log::info("Starting trust score update job");

        try {
            // Get all users with trust scores
            $users = User::with('trustScore')->get();
            $updatedCount = 0;

            foreach ($users as $user) {
                $this->updateUserTrustScore($user);
                $updatedCount++;
                
                // Log progress every 100 users
                if ($updatedCount % 100 === 0) {
                    Log::info("Updated trust scores for {$updatedCount} users");
                }
            }

            Log::info("Trust score update completed. Updated {$updatedCount} users");

        } catch (\Exception $e) {
            Log::error("Error updating trust scores: " . $e->getMessage());
        }
    }

    /**
     * Update individual user's trust score
     */
    protected function updateUserTrustScore(User $user)
    {
        // Load Settings
        $settings = \App\Models\Setting::where('key', 'like', 'trust_%')->pluck('value', 'key');
        
        // Base & Verification
        $baseScore = 0; // Starts from 0, climbs up
        
        // Max Points (Mapping UI Weights to Max Points)
        $maxSales = (int) ($settings['trust_score_max_sales'] ?? 20);
        $maxReputation = (int) ($settings['trust_score_max_reputation'] ?? 20);
        $maxMaturity = (int) ($settings['trust_score_max_maturity'] ?? 15);
        $maxContent = (int) ($settings['trust_score_max_content'] ?? 20);
        $maxBehavior = (int) ($settings['trust_score_max_behavior'] ?? 15);
        $maxSecurity = (int) ($settings['trust_score_max_security'] ?? 10);
        
        // Penalties
        $penaltyReport = (int) ($settings['trust_penalty_report'] ?? 10);
        $penaltyWarning = (int) ($settings['trust_penalty_warning'] ?? 5);
        $penaltySuspend = (int) ($settings['trust_penalty_suspension'] ?? 25);
        $penaltyMultiplier = (int) ($settings['trust_penalty_multiplier'] ?? 2);

        $trustScore = UserTrustScore::firstOrCreate(
            ['user_id' => $user->id],
            [
                'score' => $baseScore,
                'tier' => 'new',
                'breakdown' => $this->getDefaultBreakdown()
            ]
        );

        $score = $baseScore;
        $breakdown = [];

        // 1. Transaction History
        $completedSales = $user->ads()->where('status', 'sold')->count(); 
        $salesScore = min($completedSales * 2, $maxSales);
        
        $breakdown['transaction_history'] = $salesScore;
        $score += $salesScore;

        // 2. Reputation (Reviews)
        // Logic: Weighted mix of Average Rating and Review Count
        // 5 Star Average = High points. But needs volume.
        $avgRating = $user->averageRating ?? 0; // Assuming dynamic attribute or relationship
        $reviewCount = $user->reviewCount ?? 0;
        
        $reputationValues = 0;
        if ($reviewCount > 0) {
            // Base score from rating (0-5 scale mapped to percentage of maxReputation)
            // e.g. 4.5 stars / 5 = 0.9. 0.9 * maxReputation = points.
            // But we penalize low volume.
            $ratingFactor = ($avgRating / 5);
            
            // Volume factor: 1 review = 0.5 multiplier, 10+ reviews = 1.0 multiplier
            $volumeFactor = min(1.0, 0.5 + ($reviewCount * 0.05));
            
            $reputationValues = $maxReputation * $ratingFactor * $volumeFactor;
        }
        
        $breakdown['reputation'] = round($reputationValues);
        $score += $breakdown['reputation'];

        // 3. Account Maturity
        $accountAge = $user->created_at->diffInMonths(now());
        $maturityScore = min($accountAge * 1, $maxMaturity);
        $breakdown['account_maturity'] = $maturityScore;
        $score += $maturityScore;

        // 4. Content Quality
        $qualityScore = $this->calculateContentQuality($user);
        $normalizedQuality = ($qualityScore / 25) * $maxContent;
        
        $breakdown['content_quality'] = round($normalizedQuality);
        $score += $breakdown['content_quality'];

        // 5. Behavior Analysis
        $rawBehavior = $this->calculateBehaviorScore($user); 
        $normalizedBehavior = ($rawBehavior / 15) * $maxBehavior;
        
        $breakdown['behavior'] = round($normalizedBehavior);
        $score += $breakdown['behavior'];

        // 5. Security Features
        $securityScore = 0;
        $halfSecurity = $maxSecurity / 2;
        if ($user->phone_verified_at || $user->email_verified_at) $securityScore += $halfSecurity; // Simple check
        if ($user->is_verified) $securityScore += $halfSecurity; // ID Verified
        
        $breakdown['security'] = $securityScore;
        $score += $securityScore;

        // Apply Penalties
        $penaltyPoints = 0;
        
        // Reports
        $reports = $user->safetyReports()->where('status', 'action_taken')->count();
        $penaltyPoints += $reports * $penaltyReport;

        // Warnings
        $warnings = $user->safetyActions()->where('action_type', 'warning')->count();
        $penaltyPoints += $warnings * $penaltyWarning;

        // Suspensions
        $suspensions = $user->safetyActions()->where('action_type', 'auto_suspend')->count();
        $currentSuspension = $user->is_suspended; // Check simpler flag
        
        $suspensionCost = $suspensions * $penaltySuspend;
        if ($currentSuspension) {
            $suspensionCost *= $penaltyMultiplier;
        }
        $penaltyPoints += $suspensionCost;

        $score -= $penaltyPoints;

        // Final Bounds
        $score = max(0, min(100, $score));

        // Update DB
        $shouldUpdate = false;
        if ((int)$trustScore->score !== (int)$score) {
            $shouldUpdate = true;
        } elseif (!$trustScore->last_calculated || now()->diffInHours($trustScore->last_calculated) > 24) {
            $shouldUpdate = true;
        }

        if ($shouldUpdate) {
            $trustScore->update([
                'score' => $score,
                'tier' => $this->calculateTier($score),
                'breakdown' => $breakdown,
                'penalty_points' => $penaltyPoints,
                'last_calculated' => now()
            ]);
            
            // Sync with User table if needed for quick access
            if (\Schema::hasColumn('users', 'trust_score')) {
                $user->trust_score = $score;
                $user->saveQuietly();
            }

            Log::info("Updated trust score for user {$user->id}: {$trustScore->score} -> {$score}");
        }
    }

    /**
     * Calculate trust tier
     */
    protected function calculateTier($score)
    {
        $settings = \App\Models\Setting::where('key', 'like', 'trust_tier_%')->pluck('value', 'key');
        
        $legend = (int) ($settings['trust_tier_legend'] ?? 85);
        $elite = (int) ($settings['trust_tier_elite'] ?? 70);
        $highlyTrusted = (int) ($settings['trust_tier_highly_trusted'] ?? 50);
        $trusted = (int) ($settings['trust_tier_trusted'] ?? 30);

        if ($score >= $legend) return 'legend';
        if ($score >= $elite) return 'elite';
        if ($score >= $highlyTrusted) return 'highly_trusted';
        if ($score >= $trusted) return 'trusted';
        return 'new';
    }

    /**
     * Get default breakdown
     */
    /**
     * Get default breakdown
     */
    protected function getDefaultBreakdown()
    {
        return [
            'transaction_history' => 0,
            'reputation' => 0,
            'account_maturity' => 0,
            'content_quality' => 0,
            'behavior' => 0,
            'security' => 0
        ];
    }

    /**
     * Calculate content quality score (0-25 scale)
     */
    protected function calculateContentQuality($user)
    {
        $score = 15; // Base quality
        
        // Avg description length check
        $avgDescLen = $user->ads()->avg(\DB::raw('LENGTH(description)')) ?? 0;
        if ($avgDescLen > 100) $score += 5;
        if ($avgDescLen > 300) $score += 5;
        
        // Photos check
        // Assuming we can check photo count easily, pseudo:
        // $avgPhotos = $user->ads()->withCount('photos')->avg('photos_count');
        // if ($avgPhotos > 2) $score += 5;
        
        return min(25, $score);
    }

    /**
     * Calculate behavior score (0-15 scale)
     */
    protected function calculateBehaviorScore($user)
    {
        $score = 10;
        
        // Bonus for no reports ever?
        $totalReports = $user->safetyReports()->count();
        if ($totalReports === 0) $score += 5;
        
        return min(15, $score);
    }
}
