<?php

namespace App\Jobs;

use App\Models\SafetyReport;
use App\Models\SafetyAction;
use App\Models\UserTrustScore;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessSafetyReport implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    protected $reportId;

    /**
     * Create a new job instance.
     */
    public function __construct($reportId)
    {
        $this->reportId = $reportId;
    }

    /**
     * Execute the job.
     */
    public function handle()
    {
        $report = SafetyReport::find($this->reportId);
        
        if (!$report) {
            Log::warning("Safety report not found: {$this->reportId}");
            return;
        }

        try {
            // Calculate scam score
            $scamScore = $this->calculateScamScore($report);
            
            Log::info("Processing safety report {$report->id} for user {$report->reported_user_id}, scam score: {$scamScore}");

            // Determine action based on score
            if ($scamScore >= 80) {
                // High confidence scam - immediate suspension
                $this->takeAction($report, 'auto_suspend', 'High scam probability detected', 14);
            } elseif ($scamScore >= 60) {
                // Medium confidence - temporary suspension + manual review
                $this->takeAction($report, 'auto_suspend', 'Medium scam probability detected', 7);
                $this->takeAction($report, 'manual_review', 'Requires manual review due to medium scam probability', null);
            } elseif ($scamScore >= 40) {
                // Low-medium confidence - warning + monitoring
                $this->takeAction($report, 'warning', 'Potential scam behavior detected', null);
                $this->takeAction($report, 'manual_review', 'Requires monitoring due to potential scam behavior', null);
            } else {
                // Low confidence - just flag for review
                $this->takeAction($report, 'manual_review', 'Low confidence scam detection - manual review recommended', null);
            }

            // Update report status
            $report->status = 'action_taken';
            $report->save();

            // Update user's trust score
            $this->updateTrustScore($report->reported_user_id, -$scamScore / 10);

        } catch (\Exception $e) {
            Log::error("Error processing safety report {$this->reportId}: " . $e->getMessage());
            
            // Mark report as failed
            $report->status = 'dismissed';
            $report->admin_notes = 'Processing failed: ' . $e->getMessage();
            $report->save();
        }
    }

    /**
     * Calculate scam score for a report
     */
    protected function calculateScamScore(SafetyReport $report)
    {
        $score = 0;

        // Base score from report type
        $typeScores = [
            'scam' => 40,
            'fake' => 35,
            'spam' => 25,
            'inappropriate' => 20,
            'other' => 10
        ];
        
        $score += $typeScores[$report->report_type] ?? 10;

        // Analyze description for scam keywords
        $description = strtolower($report->description ?? '');
        $scamKeywords = [
            'urgent', 'too good', 'send money', 'wire transfer', 'advance fee',
            'lottery', 'inheritance', 'prize', 'winner', 'congratulations',
            'act now', 'limited time', 'immediate payment', 'bank transfer',
            'western union', 'gift card', 'paypal', 'bitcoin', 'crypto'
        ];

        foreach ($scamKeywords as $keyword) {
            if (strpos($description, $keyword) !== false) {
                $score += 5;
            }
        }

        // Check user's history
        $userReports = SafetyReport::where('reported_user_id', $report->reported_user_id)
            ->where('status', 'action_taken')
            ->count();
        
        $score += $userReports * 10;

        // Check if user has multiple reports recently
        $recentReports = SafetyReport::where('reported_user_id', $report->reported_user_id)
            ->where('created_at', '>=', now()->subDays(7))
            ->count();
        
        if ($recentReports >= 3) {
            $score += 15;
        }

        // Check if reporter is trusted (if not anonymous)
        if (!$report->is_anonymous && $report->reporter_id) {
            $reporterTrust = UserTrustScore::where('user_id', $report->reporter_id)->first();
            if ($reporterTrust && $reporterTrust->score >= 70) {
                $score += 5; // Trusted reporter increases confidence
            }
        }

        return min($score, 100);
    }

    /**
     * Take automated safety action
     */
    protected function takeAction(SafetyReport $report, $actionType, $reason, $durationDays)
    {
        // Check if similar action already exists
        $existingAction = SafetyAction::where('user_id', $report->reported_user_id)
            ->where('action_type', $actionType)
            ->where('is_active', true)
            ->where('expires_at', '>', now())
            ->first();

        if ($existingAction) {
            // Extend existing action if needed
            if ($durationDays && $existingAction->expires_at) {
                $newExpiry = now()->addDays($durationDays);
                if ($newExpiry > $existingAction->expires_at) {
                    $existingAction->expires_at = $newExpiry;
                    $existingAction->save();
                }
            }
            return;
        }

        // Create new action
        $action = SafetyAction::create([
            'user_id' => $report->reported_user_id,
            'report_id' => $report->id,
            'action_type' => $actionType,
            'reason' => $reason,
            'auto_generated' => true,
            'admin_id' => null,
            'expires_at' => $durationDays ? now()->addDays($durationDays) : null,
            'is_active' => true
        ]);

        // Apply the action
        $this->applyAction($action);
    }

    /**
     * Apply safety action to user
     */
    protected function applyAction(SafetyAction $action)
    {
        switch ($action->action_type) {
            case 'auto_suspend':
                $this->suspendUser($action->user, $action->expires_at, $action->reason);
                break;
            
            case 'warning':
                $this->sendWarningNotification($action->user, $action->reason);
                break;
            
            case 'manual_review':
                // Just log for manual review
                Log::info("User {$action->user_id} flagged for manual review: {$action->reason}");
                break;
        }
    }

    /**
     * Suspend user account
     */
    protected function suspendUser($user, $until, $reason)
    {
        $user->update([
            'is_suspended' => true,
            'suspension_until' => $until,
            'suspension_reason' => $reason
        ]);

        // Suspend all their ads
        $user->ads()->update(['status' => 'suspended']);

        Log::info("User {$user->id} suspended until {$until}: {$reason}");
    }

    /**
     * Send warning notification
     */
    protected function sendWarningNotification($user, $reason)
    {
        // This would integrate with your notification system
        Log::info("Warning sent to user {$user->id}: {$reason}");
        
        // You could add email notification here
        // Notification::send($user, new SafetyWarningNotification($reason));
    }

    /**
     * Update user's trust score
     */
    protected function updateTrustScore($userId, $points)
    {
        $trustScore = UserTrustScore::firstOrCreate(
            ['user_id' => $userId],
            ['score' => 50, 'tier' => 'trusted', 'breakdown' => []]
        );

        $newScore = max(0, min(100, $trustScore->score + $points));
        
        $trustScore->update([
            'score' => $newScore,
            'tier' => $this->calculateTier($newScore),
            'penalty_points' => $trustScore->penalty_points + abs($points),
            'last_calculated' => now()
        ]);

        Log::info("Trust score updated for user {$userId}: {$trustScore->score} -> {$newScore}");
    }

    /**
     * Calculate trust tier
     */
    protected function calculateTier($score)
    {
        if ($score >= 85) return 'legend';
        if ($score >= 70) return 'elite';
        if ($score >= 50) return 'highly_trusted';
        if ($score >= 30) return 'trusted';
        return 'new';
    }
}
