<?php

namespace App\Jobs;

use App\Models\Ad;
use App\Models\Notification;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class ProcessAdExpiration implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct()
    {
        //
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        Log::info('Processing ad expiration job started');

        try {
            // Get ads that should expire today
            $today = now()->startOfDay();
            $adsToExpire = Ad::where('status', 'Active')
                ->whereDate('expires_at', '<=', $today)
                ->get();

            $expiredCount = 0;

            foreach ($adsToExpire as $ad) {
                // Update ad status to expired
                $ad->update([
                    'status' => 'Expired',
                    'expired_at' => now(),
                ]);

                // Send expiration notification to user
                Notification::create([
                    'user_id' => $ad->user_id,
                    'type' => 'ad_expired',
                    'title' => 'Your Ad Has Expired',
                    'message' => "Your ad '{$ad->title}' has expired and is no longer visible to users. You can renew it from your dashboard.",
                    'data' => [
                        'ad_id' => $ad->id,
                        'ad_title' => $ad->title,
                        'ad_url' => route('ads.show', $ad->id)
                    ]
                ]);

                $expiredCount++;
            }

            Log::info("Ad expiration job completed. Expired {$expiredCount} ads.");

        } catch (\Exception $e) {
            Log::error('Ad expiration job failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
