<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\UserPlan;
use App\Models\Subscription;
use App\Models\VerificationRequest;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class UserController extends Controller
{
    /**
     * Display the specified user.
     */
    public function show($id)
    {
        $user = User::with(['currentPlan', 'trustScore'])->findOrFail($id);
        
        // Append extra attributes for profile
        $user->rating = 4.8; // Mock rating for now
        $user->review_count = 12; // Mock count
        $user->badges = $this->getUserBadges($user);

        return response()->json([
            'success' => true,
            'data' => $user,
        ]);
    }

    private function getUserBadges($user)
    {
        $badges = [];
        
        // Plan Badges
        if ($user->currentPlan) {
            if ($user->currentPlan->verified_badge) {
                 $badges[] = ['type' => 'verified', 'label' => 'Verified', 'icon' => 'verified'];
            }
            if ($user->currentPlan->price > 0) {
                 $badges[] = ['type' => 'premium', 'label' => $user->currentPlan->plan_name, 'icon' => 'workspace_premium'];
            }
        }
        
        // Trust Badge
        if ($user->trustScore) {
             $badges[] = ['type' => 'trust', 'label' => $user->trustScore->getTierBadge(), 'icon' => 'shield'];
        }

        return $badges;
    }

    /**
     * Update the specified user.
     */
    public function update(Request $request, $id)
    {
        $user = User::findOrFail($id);

        // Only allow users to update their own profile
        if ($user->id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'sometimes|required|string|max:255',
            'email' => 'sometimes|required|string|email|max:255|unique:users,email,' . $user->id,
            'phone_number' => 'sometimes|nullable|string|max:20',
            'profile_image' => 'sometimes|nullable|string',
            'bio' => 'sometimes|nullable|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        $user->update($request->only(['name', 'email', 'phone_number', 'profile_image', 'bio']));

        return response()->json([
            'success' => true,
            'message' => 'User updated successfully',
            'data' => $user->fresh('currentPlan'),
        ]);
    }

    /**
     * Remove the specified user.
     */
    public function destroy($id)
    {
        $user = User::findOrFail($id);

        // Only allow users to delete their own account
        if ($user->id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $user->delete();

        return response()->json([
            'success' => true,
            'message' => 'User deleted successfully',
        ]);
    }

    /**
     * Get current user's plan.
     */
    public function getCurrentPlan($id)
    {
        $user = User::findOrFail($id);

        // Only allow users to view their own plan
        if ($user->id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $plan = $user->currentPlan;

        if (!$plan) {
            return response()->json([
                'success' => false,
                'message' => 'No active plan found',
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $plan,
        ]);
    }

    /**
     * Subscribe user to a plan.
     */
    public function subscribeToPlan(Request $request, $id)
    {
        $user = User::findOrFail($id);

        // Only allow users to subscribe themselves
        if ($user->id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'plan_id' => 'required|exists:user_plans,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        $plan = UserPlan::findOrFail($request->plan_id);

        if (!$plan->isActive()) {
            return response()->json([
                'success' => false,
                'message' => 'Plan is not available',
            ], 400);
        }

        // Create subscription
        $subscription = Subscription::create([
            'user_id' => $user->id,
            'plan_id' => $plan->id,
            'start_date' => now(),
            'end_date' => now()->addDays($plan->duration_days),
            'status' => 'Paid', // Assume payment is handled elsewhere
        ]);

        // Update user's current plan
        $user->current_plan_id = $plan->id;
        $user->save();

        return response()->json([
            'success' => true,
            'message' => 'Subscription created successfully',
            'data' => [
                'subscription' => $subscription,
                'plan' => $plan,
            ],
        ]);
    }

    /**
     * Request user verification.
     */
    public function verifyUser(Request $request, $id)
    {
        $user = User::findOrFail($id);

        // Only allow users to verify themselves
        if ($user->id !== Auth::id()) {
            return response()->json([
                'success' => false,
                'message' => 'Unauthorized',
            ], 403);
        }

        $validator = Validator::make($request->all(), [
            'document_url' => 'required|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors(),
            ], 422);
        }

        // Check if already verified
        if ($user->is_verified) {
            return response()->json([
                'success' => false,
                'message' => 'User is already verified',
            ], 400);
        }

        // Check if verification request already exists
        $existingRequest = VerificationRequest::where('user_id', $user->id)
            ->whereIn('status', ['Pending', 'Approved'])
            ->exists();

        if ($existingRequest) {
            return response()->json([
                'success' => false,
                'message' => 'Verification request already exists',
            ], 400);
        }

        $verificationRequest = VerificationRequest::create([
            'user_id' => $user->id,
            'document_url' => $request->document_url,
            'status' => 'Pending',
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Verification request submitted successfully',
            'data' => $verificationRequest,
        ]);
    }
}
