<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Schema\Blueprint;

class InstallController extends Controller
{
    /**
     * Show the installation welcome page.
     */
    public function showWelcome()
    {
        // Check if already installed
        if ($this->isInstalled()) {
            return redirect('/admin/login');
        }

        return view('install.welcome');
    }

    /**
     * Show the system requirements check page.
     */
    public function showRequirements()
    {
        if ($this->isInstalled()) {
            return redirect('/admin/login');
        }

        $requirements = $this->checkRequirements();
        
        return view('install.requirements', compact('requirements'));
    }

    /**
     * Show the database configuration page.
     */
    public function showDatabase()
    {
        if ($this->isInstalled()) {
            return redirect('/admin/login');
        }

        return view('install.database');
    }

    /**
     * Process database configuration and run migrations.
     */
    public function processDatabase(Request $request)
    {
        $request->validate([
            'db_host' => 'required|string',
            'db_port' => 'required|string',
            'db_name' => 'required|string',
            'db_username' => 'required|string',
            'db_password' => 'nullable|string',
            'admin_email' => 'required|email',
            'admin_password' => 'required|min:8|confirmed',
        ]);

        try {
            // Test database connection
            $this->testDatabaseConnection($request->all());

            // Update .env file
            $this->updateEnvFile($request->all());

            // Run migrations
            Artisan::call('migrate', ['--force' => true]);

            // Run seeders
            Artisan::call('db:seed', ['--force' => true]);

            // Create admin user
            $this->createAdminUser($request->admin_email, $request->admin_password);

            // Mark as installed
            $this->markAsInstalled();

            return redirect()->route('install.complete');

        } catch (\Exception $e) {
            return back()->withErrors(['error' => 'Database setup failed: ' . $e->getMessage()]);
        }
    }

    /**
     * Show the installation complete page.
     */
    public function showComplete()
    {
        if (!$this->isInstalled()) {
            return redirect()->route('install.welcome');
        }

        return view('install.complete');
    }

    /**
     * Check if the application is already installed.
     */
    private function isInstalled()
    {
        try {
            return Schema::hasTable('users') && 
                   DB::table('users')->where('email', 'admin@example.com')->exists();
        } catch (\Exception $e) {
            return false;
        }
    }

    /**
     * Check system requirements.
     */
    private function checkRequirements()
    {
        $requirements = [
            'php_version' => [
                'current' => phpversion(),
                'required' => '8.0',
                'status' => version_compare(phpversion(), '8.0', '>='),
            ],
            'extensions' => [
                'openssl' => extension_loaded('openssl'),
                'pdo' => extension_loaded('pdo'),
                'mbstring' => extension_loaded('mbstring'),
                'tokenizer' => extension_loaded('tokenizer'),
                'xml' => extension_loaded('xml'),
                'ctype' => extension_loaded('ctype'),
                'json' => extension_loaded('json'),
                'bcmath' => extension_loaded('bcmath'),
            ],
            'permissions' => [
                'storage' => is_writable(storage_path()),
                'bootstrap_cache' => is_writable(base_path('bootstrap/cache')),
                'public' => is_writable(public_path()),
            ]
        ];

        return $requirements;
    }

    /**
     * Test database connection.
     */
    private function testDatabaseConnection($data)
    {
        $tempConfig = [
            'driver' => 'mysql',
            'host' => $data['db_host'],
            'port' => $data['db_port'],
            'database' => $data['db_name'],
            'username' => $data['db_username'],
            'password' => $data['db_password'] ?? '',
            'charset' => 'utf8mb4',
            'collation' => 'utf8mb4_unicode_ci',
            'prefix' => '',
        ];

        try {
            DB::purge('mysql');
            config(['database.connections.mysql' => $tempConfig]);
            DB::reconnect('mysql');
            DB::connection()->getPdo();
        } catch (\Exception $e) {
            throw new \Exception('Cannot connect to database: ' . $e->getMessage());
        }
    }

    /**
     * Update .env file with database configuration.
     */
    private function updateEnvFile($data)
    {
        $envPath = base_path('.env');
        $envContent = file_get_contents($envPath);

        // Generate new APP_KEY
        $newAppKey = 'base64:' . base64_encode(random_bytes(32));

        $replacements = [
            'DB_HOST=' => 'DB_HOST=' . $data['db_host'],
            'DB_PORT=' => 'DB_PORT=' . $data['db_port'],
            'DB_DATABASE=' => 'DB_DATABASE=' . $data['db_name'],
            'DB_USERNAME=' => 'DB_USERNAME=' . $data['db_username'],
            'DB_PASSWORD=' => 'DB_PASSWORD=' . ($data['db_password'] ?? ''),
            'APP_KEY=' => 'APP_KEY=' . $newAppKey,
        ];

        foreach ($replacements as $search => $replace) {
            $envContent = preg_replace('/^' . preg_quote($search, '/') . '.*$/m', $replace, $envContent);
        }

        file_put_contents($envPath, $envContent);
        
        // Reload configuration
        Artisan::call('config:clear');
    }

    /**
     * Create admin user.
     */
    private function createAdminUser($email, $password)
    {
        \App\Models\User::create([
            'name' => 'Admin User',
            'email' => $email,
            'password' => Hash::make($password),
        ]);
    }

    /**
     * Mark installation as complete.
     */
    private function markAsInstalled()
    {
        // Create a simple marker file
        file_put_contents(storage_path('installed'), date('Y-m-d H:i:s'));
    }
}
