<?php

namespace App\Http\Controllers;

use App\Models\ChatMessage;
use App\Models\User;
use App\Models\Ad;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;

class ChatController extends Controller
{
    /**
     * Get conversation between two users
     */
    public function getConversation(Request $request, $userId)
    {
        $currentUser = Auth::user();
        
        // Validate that the user exists
        $targetUser = User::find($userId);
        if (!$targetUser) {
            return response()->json([
                'success' => false,
                'message' => 'User not found'
            ], 404);
        }

        // Get messages between current user and target user
        $messages = ChatMessage::whereBetweenUsers($currentUser->id, $userId)
            ->with(['sender', 'receiver', 'ad'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);

        // Mark messages as read
        ChatMessage::where('receiver_id', $currentUser->id)
            ->where('sender_id', $userId)
            ->whereNull('read_at')
            ->update(['read_at' => now()]);

        return response()->json([
            'success' => true,
            'messages' => $messages,
            'target_user' => $targetUser
        ]);
    }

    /**
     * Send a new message
     */
    public function sendMessage(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'receiver_id' => 'required|exists:users,id',
            'message' => 'required_without:attachment|string|max:1000',
            'attachment' => 'required_without:message|file|max:10240', // 10MB max
            'ad_id' => 'nullable|exists:ads,id',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $currentUser = Auth::user();
        $receiver = User::find($request->receiver_id);

        if (!$receiver) {
            return response()->json([
                'success' => false,
                'message' => 'Receiver not found'
            ], 404);
        }

        // Check if users can chat (basic check - both should be active)
        if ($currentUser->status !== 'active' || $receiver->status !== 'active') {
            return response()->json([
                'success' => false,
                'message' => 'Cannot send message to inactive user'
            ], 400);
        }

        $messageData = [
            'sender_id' => $currentUser->id,
            'receiver_id' => $request->receiver_id,
            'message' => $request->message ?? '',
            'message_type' => 'text',
        ];

        // Handle attachment
        if ($request->hasFile('attachment')) {
            $attachment = $request->file('attachment');
            $fileName = time() . '_' . $attachment->getClientOriginalName();
            $path = $attachment->storeAs('chat_attachments', $fileName, 'public');
            
            $messageData['attachment_url'] = Storage::url($path);
            $messageData['attachment_type'] = $this->getAttachmentType($attachment->getClientOriginalExtension());
            $messageData['message_type'] = $messageData['attachment_type'];
        }

        // Add ad reference if provided
        if ($request->ad_id) {
            $messageData['ad_id'] = $request->ad_id;
        }

        $message = ChatMessage::create($messageData);

        // Load relationships for response
        $message->load(['sender', 'receiver', 'ad']);

        return response()->json([
            'success' => true,
            'message' => $message
        ], 201);
    }

    /**
     * Get user's chat list (conversations)
     */
    public function getChatList(Request $request)
    {
        $currentUser = Auth::user();
        
        // Get distinct users that the current user has chatted with
        $conversations = ChatMessage::selectRaw('
                CASE 
                    WHEN sender_id = ? THEN receiver_id 
                    ELSE sender_id 
                END as other_user_id,
                MAX(created_at) as last_message_time,
                MAX(message) as last_message,
                MAX(message_type) as last_message_type,
                COUNT(CASE WHEN receiver_id = ? AND read_at IS NULL THEN 1 END) as unread_count
            ', [$currentUser->id, $currentUser->id])
            ->where(function($query) use ($currentUser) {
                $query->where('sender_id', $currentUser->id)
                      ->orWhere('receiver_id', $currentUser->id);
            })
            ->where(function($query) use ($currentUser) {
                $query->where('deleted_by_sender', false)
                      ->orWhereNull('deleted_by_sender');
            })
            ->groupBy('other_user_id')
            ->orderBy('last_message_time', 'desc')
            ->get();

        // Get user details for each conversation
        $chatList = $conversations->map(function($conversation) {
            $otherUser = User::find($conversation->other_user_id);
            return [
                'user' => $otherUser,
                'last_message_time' => $conversation->last_message_time,
                'last_message' => $conversation->last_message,
                'last_message_type' => $conversation->last_message_type,
                'unread_count' => $conversation->unread_count,
            ];
        });

        return response()->json([
            'success' => true,
            'conversations' => $chatList
        ]);
    }

    /**
     * Get unread message count
     */
    public function getUnreadCount(Request $request)
    {
        $currentUser = Auth::user();
        
        $unreadCount = ChatMessage::unread($currentUser->id)->count();

        return response()->json([
            'success' => true,
            'unread_count' => $unreadCount
        ]);
    }

    /**
     * Delete a message
     */
    public function deleteMessage(Request $request, $messageId)
    {
        $currentUser = Auth::user();
        
        $message = ChatMessage::where('id', $messageId)
            ->where(function($query) use ($currentUser) {
                $query->where('sender_id', $currentUser->id)
                      ->orWhere('receiver_id', $currentUser->id);
            })
            ->first();

        if (!$message) {
            return response()->json([
                'success' => false,
                'message' => 'Message not found'
            ], 404);
        }

        if (!$message->can_delete) {
            return response()->json([
                'success' => false,
                'message' => 'Cannot delete message after 24 hours'
            ], 400);
        }

        // Soft delete by marking as deleted for the current user
        if ($message->sender_id === $currentUser->id) {
            $message->deleted_by_sender = true;
        } else {
            $message->deleted_by_receiver = true;
        }

        $message->save();

        return response()->json([
            'success' => true,
            'message' => 'Message deleted successfully'
        ]);
    }

    /**
     * Search messages in a conversation
     */
    public function searchMessages(Request $request, $userId)
    {
        $currentUser = Auth::user();
        
        $validator = Validator::make($request->all(), [
            'query' => 'required|string|min:2',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors()
            ], 422);
        }

        $messages = ChatMessage::whereBetweenUsers($currentUser->id, $userId)
            ->where('message', 'like', '%' . $request->query . '%')
            ->with(['sender', 'receiver'])
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        return response()->json([
            'success' => true,
            'messages' => $messages,
            'search_query' => $request->query
        ]);
    }

    /**
     * Helper method to determine attachment type
     */
    private function getAttachmentType($extension)
    {
        $imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'webp'];
        $videoExtensions = ['mp4', 'avi', 'mov', 'wmv', 'flv'];
        $audioExtensions = ['mp3', 'wav', 'ogg', 'm4a'];

        $extension = strtolower($extension);

        if (in_array($extension, $imageExtensions)) {
            return 'image';
        } elseif (in_array($extension, $videoExtensions)) {
            return 'video';
        } elseif (in_array($extension, $audioExtensions)) {
            return 'audio';
        } else {
            return 'file';
        }
    }
}
