<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Banner;
use App\Models\BannerPackage;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\DB;

class UserBannerController extends Controller
{
    /**
     * Get user's banners
     */
    public function index(Request $request)
    {
        $banners = Banner::where('user_id', $request->user()->id)
            ->with('package')
            ->orderBy('created_at', 'desc')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $banners,
        ]);
    }

    /**
     * Create a new banner
     */
    /**
     * Get active banner packages
     */
    public function getPackages()
    {
        $packages = BannerPackage::where('is_active', true)
            ->orderBy('display_order')
            ->get();
            
        return response()->json([
            'success' => true,
            'data' => $packages
        ]);
    }

    /**
     * Create a new banner
     */
    public function store(Request $request)
    {
        $request->validate([
            'package_id' => 'required|exists:banner_packages,id',
            'title' => 'required|string|max:255',
            'image' => 'required|image|max:2048',
            'redirect_type' => 'required|in:url,ad,category',
            'redirect_value' => 'required|string',
            'target_location_type' => 'required|in:all,country,state,city,radius',
            'target_latitude' => 'required_if:target_location_type,radius|numeric|nullable',
            'target_longitude' => 'required_if:target_location_type,radius|numeric|nullable',
            'target_radius' => 'required_if:target_location_type,radius|numeric|min:1|nullable',
            'placement_screens' => 'nullable|array',
            'placement_screens.*' => 'string',
            // total_budget, cost_per_impression/click are simpler now, derived from package
        ]);

        $package = BannerPackage::findOrFail($request->package_id);

        // Upload image
        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $request->file('image')->store('banners', 'public');
        }
        
        // Calculate costs/limits based on package
        $costPerImpression = $package->base_price_per_impression ?? 0;
        $costPerClick = $package->base_price_per_click ?? 0;
        
        // For simplified flow, we might just set the total budget or use the package limits directly.
        // If package is fixed price (e.g. $50 for 1000 views), we handle that.
        // Assuming user acts as buying a 'slot' or 'quota'.
        
        // But the previous logic had 'total_budget'. Let's assume for now the user "buys" the package.
        // We'll set the budget to an arbitrary high number or 0 if it's package based, 
        // OR better: if the package has a price, we record that transaction separately (Payment flow).
        // Here we just set up the banner.
        
        $viewsLimit = $package->max_impressions_included;
        $clicksLimit = $package->max_clicks_included;

        // Create banner
        $banner = Banner::create([
            'user_id' => $request->user()->id,
            'package_id' => $request->package_id,
            'title' => $request->title,
            'image_url' => $imagePath ? Storage::url($imagePath) : null,
            'redirect_type' => $request->redirect_type,
            'redirect_value' => $request->redirect_value,
            'billing_model' => $costPerClick > 0 ? 'click' : 'impression', // Infer from package
            'target_location_type' => $request->target_location_type,
            'target_countries' => $request->target_country ? [$request->target_country] : ($request->target_countries ?? null),
            'target_states' => $request->target_state ? [$request->target_state] : ($request->target_states ?? null),
            'target_cities' => $request->target_city ? [$request->target_city] : ($request->target_cities ?? null),
            'target_latitude' => $request->target_latitude,
            'target_longitude' => $request->target_longitude,
            'target_radius' => $request->target_radius,
            'placement_screens' => $request->placement_screens,
            'total_budget' => 0, // Managed by package limits
            'spent_amount' => 0,
            'cost_per_impression' => $costPerImpression,
            'cost_per_click' => $costPerClick,
            'views_limit' => $viewsLimit,
            'clicks_limit' => $clicksLimit,
            'start_date' => $request->start_date ?? now(),
            'end_date' => null, // Or set based on package duration if it exists
            'status' => ($costPerImpression <= 0 && $costPerClick <= 0) ? 'active' : 'pending',
            'priority' => $package->id,
        ]);

        return response()->json([
            'success' => true,
            'message' => ($costPerImpression <= 0 && $costPerClick <= 0) 
                ? 'Banner activated successfully!' 
                : 'Banner created successfully! Awaiting admin approval.',
            'data' => $banner->load('package'),
        ], 201);
    }

    /**
     * Get banner analytics
     */
    public function analytics(Request $request, $bannerId)
    {
        $banner = Banner::where('user_id', $request->user()->id)
            ->findOrFail($bannerId);

        $timeRange = $request->input('time_range', '24h');
        $startTime = $this->getStartTime($timeRange);

        $impressions = $banner->impressions()
            ->where('created_at', '>=', $startTime)
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->get();

        $clicks = $banner->clicks()
            ->where('created_at', '>=', $startTime)
            ->selectRaw('DATE(created_at) as date, COUNT(*) as count')
            ->groupBy('date')
            ->get();

        $deviceBreakdown = $banner->impressions()
            ->where('created_at', '>=', $startTime)
            ->selectRaw('user_agent, COUNT(*) as count')
            ->groupBy('user_agent')
            ->get();

        $locationBreakdown = $banner->impressions()
            ->where('created_at', '>=', $startTime)
            ->selectRaw('city, COUNT(*) as count')
            ->groupBy('city')
            ->limit(10)
            ->get();

        return response()->json([
            'success' => true,
            'data' => [
                'total_impressions' => $banner->total_impressions,
                'total_clicks' => $banner->total_clicks,
                'ctr' => $banner->ctr,
                'total_spent' => $banner->spent_amount,
                'impressions_over_time' => $impressions,
                'clicks_over_time' => $clicks,
                'device_breakdown' => $deviceBreakdown,
                'location_breakdown' => $locationBreakdown,
            ],
        ]);
    }

    /**
     * Get real-time stats for all user banners with graph data
     */
    public function realTimeStats(Request $request)
    {
        $timeRange = $request->input('time_range', 'today');
        
        if ($timeRange === 'custom') {
            $startTime = $request->input('start_date') ? Carbon::parse($request->input('start_date'))->startOfDay() : null;
            $endTime = $request->input('end_date') ? Carbon::parse($request->input('end_date'))->endOfDay() : null;
        } else {
            [$startTime, $endTime] = $this->getDateRange($timeRange);
        }

        $banners = Banner::where('user_id', $request->user()->id)->get();
        $bannerIds = $banners->pluck('id');

        // Total Stats
        $impressionQuery = DB::table('banner_impressions')
            ->whereIn('banner_id', $bannerIds);
        
        $clickQuery = DB::table('banner_clicks')
            ->whereIn('banner_id', $bannerIds);

        if ($startTime) {
            $impressionQuery->where('created_at', '>=', $startTime);
            $clickQuery->where('created_at', '>=', $startTime);
        }
        if ($endTime) {
            $impressionQuery->where('created_at', '<=', $endTime);
            $clickQuery->where('created_at', '<=', $endTime);
        }

        $totalImpressions = $impressionQuery->count();
        $totalClicks = $clickQuery->count();
        $ctr = $totalImpressions > 0 ? ($totalClicks / $totalImpressions) * 100 : 0;

        // Graph Data Logic
        // Determine grouping based on duration
        $groupBy = 'DATE';
        if ($timeRange === '30min' || $timeRange === '1h' || $timeRange === '2h') {
            $groupBy = 'MINUTE'; // 5-min intervals ideally, but minute for now
        } elseif (in_array($timeRange, ['6h', '12h', '24h', '48h', 'today', 'yesterday'])) {
            $groupBy = 'HOUR';
        } else {
            // For custom, decide based on diff
            if ($startTime && $endTime) {
                $diffInHours = $startTime->diffInHours($endTime);
                if ($diffInHours <= 2) $groupBy = 'MINUTE';
                elseif ($diffInHours <= 48) $groupBy = 'HOUR';
                else $groupBy = 'DATE';
            }
        }
        
        $dateFormat = '%Y-%m-%d';
        
        // SQLite support adjustments
        $isSqlite = DB::connection()->getDriverName() === 'sqlite';
        
        if ($isSqlite) {
            if ($groupBy === 'MINUTE') {
                // Group by 5 minute intervals: strftime('%Y-%m-%d %H', created_at) || ':' || (CAST(strftime('%M', created_at) as INT) / 5 * 5)
                // Simplified to single minute for 'Live' feel
                $dateSelect = "strftime('%H:%M', created_at) as label"; 
            } elseif ($groupBy === 'HOUR') {
                $dateSelect = "strftime('%H:00', created_at) as label";
            } else {
                $dateSelect = "date(created_at) as label";
            }
        } else {
             // MySQL/MariaDB
            if ($groupBy === 'MINUTE') {
                 $dateSelect = "DATE_FORMAT(created_at, '%H:%i') as label";
            } elseif ($groupBy === 'HOUR') {
                 $dateSelect = "DATE_FORMAT(created_at, '%H:00') as label";
            } else {
                 $dateSelect = "DATE(created_at) as label";
            }
        }

        // Impressions Over Time
        $impGraphQuery = DB::table('banner_impressions')
            ->whereIn('banner_id', $bannerIds)
            ->selectRaw("$dateSelect, COUNT(*) as count");
            
        if ($startTime) $impGraphQuery->where('created_at', '>=', $startTime);
        if ($endTime) $impGraphQuery->where('created_at', '<=', $endTime);
        
        $impressionsOverTime = $impGraphQuery->groupBy('label')->get();

        // Clicks Over Time
        $clickGraphQuery = DB::table('banner_clicks')
             ->whereIn('banner_id', $bannerIds)
             ->selectRaw("$dateSelect, COUNT(*) as count");

        if ($startTime) $clickGraphQuery->where('created_at', '>=', $startTime);
        if ($endTime) $clickGraphQuery->where('created_at', '<=', $endTime);

        $clicksOverTime = $clickGraphQuery->groupBy('label')->get();


        return response()->json([
            'success' => true,
            'data' => [
                'total_impressions' => $totalImpressions,
                'total_clicks' => $totalClicks,
                'ctr' => round($ctr, 2),
                'total_spent' => $banners->sum('spent_amount'),
                'active_banners' => $banners->where('status', 'active')->count(),
                'impressions_over_time' => $impressionsOverTime,
                'clicks_over_time' => $clicksOverTime,
                'time_range' => $timeRange,
                'group_by' => $groupBy
            ],
        ]);
    }

    /**
     * Pause banner
     */
    public function pause(Request $request, $bannerId)
    {
        $banner = Banner::where('user_id', $request->user()->id)
            ->findOrFail($bannerId);

        $banner->update(['status' => 'paused']);

        return response()->json([
            'success' => true,
            'message' => 'Banner paused successfully',
        ]);
    }

    /**
     * Resume banner
     */
    public function resume(Request $request, $bannerId)
    {
        $banner = Banner::where('user_id', $request->user()->id)
            ->findOrFail($bannerId);

        $banner->update(['status' => 'active']);

        return response()->json([
            'success' => true,
            'message' => 'Banner resumed successfully',
        ]);
    }

    /**
     * Delete banner
     */
    public function destroy(Request $request, $bannerId)
    {
        $banner = Banner::where('user_id', $request->user()->id)
            ->findOrFail($bannerId);

        // Delete image
        if ($banner->image_url) {
            $path = str_replace('/storage/', '', parse_url($banner->image_url, PHP_URL_PATH));
            Storage::disk('public')->delete($path);
        }

        $banner->delete();

        return response()->json([
            'success' => true,
            'message' => 'Banner deleted successfully',
        ]);
    }

    /**
     * Get date range based on filter
     * Returns [startTime, endTime]
     */
    private function getDateRange($timeRange)
    {
        $now = now();
        
        return match($timeRange) {
            '30min' => [$now->clone()->subMinutes(30), $now->clone()],
            '1h' => [$now->clone()->subHour(), $now->clone()],
            '2h' => [$now->clone()->subHours(2), $now->clone()],
            '6h' => [$now->clone()->subHours(6), $now->clone()],
            '12h' => [$now->clone()->subHours(12), $now->clone()],
            '24h' => [$now->clone()->subDay(), $now->clone()],
            '48h' => [$now->clone()->subDays(2), $now->clone()],
            
            'today' => [$now->clone()->startOfDay(), $now->clone()->endOfDay()],
            'yesterday' => [$now->clone()->subDay()->startOfDay(), $now->clone()->subDay()->endOfDay()],
            '7d' => [$now->clone()->subDays(6)->startOfDay(), $now->clone()->endOfDay()],
            '30d' => [$now->clone()->subDays(29)->startOfDay(), $now->clone()->endOfDay()],
            '60d' => [$now->clone()->subDays(59)->startOfDay(), $now->clone()->endOfDay()],
            '90d' => [$now->clone()->subDays(89)->startOfDay(), $now->clone()->endOfDay()],
            'lifetime' => [null, null],
            default => [$now->clone()->startOfDay(), $now->clone()->endOfDay()], // Default to today
        };
    }
}
