<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\AdPackage;
use App\Models\Ad;
use App\Models\AdPackagePricingTier; // Assuming this exists or relationship usage
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use App\Models\AdPackagePurchase;

class PackageController extends Controller
{
    /**
     * Get list of active packages
     */
    public function index()
    {
        $packages = AdPackage::active()
            ->ordered()
            ->with(['activePricingTiers'])
            ->get();

        return response()->json([
            'success' => true,
            'data' => $packages
        ]);
    }

    /**
     * Get details of a specific package
     */
    public function show($id)
    {
        $package = AdPackage::with(['activePricingTiers'])
            ->where('is_active', true)
            ->find($id);

        if (!$package) {
            return response()->json([
                'success' => false,
                'message' => 'Package not found'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $package
        ]);
    }


    /**
     * Purchase/Apply a package to an ad
     */
    public function purchase(Request $request, $id)
    {
        $request->validate([
            'tier_id' => 'required|exists:ad_package_pricing_tiers,id',
            'ad_id' => 'required|exists:ads,id',
            'use_credit' => 'boolean', // Optional flag to use plan credits
        ]);

        $user = $request->user();
        $package = AdPackage::find($id);
        
        if (!$package) {
            return response()->json(['success' => false, 'message' => 'Package not found'], 404);
        }

        $tier = $package->pricingTiers()->find($request->tier_id);
        if (!$tier) {
            return response()->json(['success' => false, 'message' => 'Invalid pricing tier'], 422);
        }

        $ad = Ad::where('id', $request->ad_id)->where('user_id', $user->id)->first();
        if (!$ad) {
            return response()->json(['success' => false, 'message' => 'Ad not found or unauthorized'], 404);
        }

        // --- Logic: Use Plan Credits ---
        $usingCredit = $request->boolean('use_credit');
        $finalPrice = $tier->final_price ?? $tier->base_price;

        if ($usingCredit) {
            $hasCredit = false;
            if ($package->type === 'featured') {
                $hasCredit = $user->hasFeatureCredit();
            } elseif ($package->type === 'boost') {
                $hasCredit = $user->hasBoostCredit();
            }

            if (!$hasCredit) {
                 return response()->json(['success' => false, 'message' => 'Insufficient plan credits'], 400);
            }

            // Deduct credit
            if ($package->type === 'featured') {
                $user->useFeatureCredit();
            } elseif ($package->type === 'boost') {
                $user->useBoostCredit();
            }
            
            $finalPrice = 0; // Free if using credit
        }

        // --- Payment or Free Activation ---
        if ($finalPrice <= 0) {
            try {
                DB::beginTransaction();
                
                $days = $tier->validity_days;
                $expiry = Carbon::now()->addDays($days);

                // 1. Create Purchase Record (Vital for Algorithm Section E)
                $purchase = AdPackagePurchase::create([
                    'user_id' => $user->id,
                    'ad_id' => $ad->id,
                    'package_id' => $package->id,
                    'amount_paid' => 0,
                    'currency' => 'USD', // Default or from tier
                    'payment_method' => $usingCredit ? 'plan_credit' : 'free',
                    'transaction_id' => $usingCredit ? 'CREDIT-' . uniqid() : 'FREE-' . uniqid(),
                    'status' => 'active',
                    'start_date' => Carbon::now(),
                    'expires_at' => $expiry,
                    'features_applied' => json_encode(['type' => $package->type, 'days' => $days]),
                ]);

                // 2. Update Ad Flags (Vital for Algorithm Section A & Visuals)
                if ($package->type === 'featured') {
                    $ad->update(['is_featured' => true]);
                    // You might want a 'featured_until' column on ads table too, 
                    // but usually we check active purchases. 
                    // For simplicity in this codebase, we trust the flag for now, 
                    // or the 'scopeOrderByInterestAndLocation' check on 'ad_package_purchases' handles expiration.
                } elseif ($package->type === 'boost') {
                    $ad->update(['is_boosted' => true]);
                }
                
                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => 'Package activated successfully!',
                    'data' => [
                        'status' => 'active',
                        'expires_at' => $expiry->toDateTimeString(),
                        'purchase_id' => $purchase->id
                    ]
                ]);

            } catch (\Exception $e) {
                DB::rollBack();
                return response()->json(['success' => false, 'message' => 'Activation failed', 'error' => $e->getMessage()], 500);
            }
        }

        // --- Paid Flow (Mock Instructions) ---
        return response()->json([
            'success' => true,
            'message' => 'Proceed to payment',
            'data' => [
                'requires_payment' => true,
                'amount' => $finalPrice,
                'currency' => 'USD' 
            ]
        ]);
    }
    /**
     * Get user's active packages
     */
    public function getMyPackages(Request $request)
    {
        $user = $request->user();
        
        $purchases = \App\Models\AdPackagePurchase::where('user_id', $user->id)
            ->active()
            ->with(['package:id,name,type,icon']) // Select specific fields if needed
            ->orderBy('expires_at', 'asc')
            ->get()
            ->map(function ($purchase) {
                return [
                    'id' => $purchase->id,
                    'package_name' => $purchase->package->name,
                    'package_type' => $purchase->package->type,
                    'package_icon' => $purchase->package->icon,
                    'usage_count' => $purchase->usage_count,
                    'usage_limit' => $purchase->usage_limit,
                    'expires_at' => $purchase->expires_at->toIso8601String(),
                    'days_remaining' => $purchase->getDaysRemaining(),
                    'status' => $purchase->status,
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $purchases
        ]);
    }
}
