<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Ad;
use App\Models\User;
use App\Services\FirebaseChatService;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class LeadController extends Controller
{
    protected $firebaseChatService;

    public function __construct(FirebaseChatService $firebaseChatService)
    {
        $this->firebaseChatService = $firebaseChatService;
    }

    /**
     * Handle contact request (Call/WhatsApp click).
     * Checks lead limits for seller and contact limits for buyer.
     */
    public function contactSeller(Request $request, $id)
    {
        try {
            $buyer = $request->user();
            $ad = Ad::with('user.currentPlan')->findOrFail($id);
            $seller = $ad->user;

            if (!$seller) {
                return response()->json(['success' => false, 'message' => 'Seller not found'], 404);
            }

            if ($buyer->id === $seller->id) {
                return response()->json(['success' => false, 'message' => 'Cannot contact yourself'], 400);
            }

            // 1. Reset Cycles if needed
            $this->checkAndResetCycle($buyer);
            $this->checkAndResetCycle($seller);

            // 2. Check Seller's Lead Limit (Incoming)
            $sellerPlan = $seller->currentPlan;
            $sellerLimit = $sellerPlan ? $sellerPlan->lead_limit : 0; // Default 0 if no plan
            
            // If Limit is NULL (Unlimited) or usage < limit
            if (is_null($sellerLimit) || $seller->leads_received < $sellerLimit) {
                
                // Increment Seller's Leads Received
                $seller->increment('leads_received');

                // Send Automated Message
                $this->sendAutomatedMessage($buyer, $seller, $ad, 'standard');

                return response()->json([
                    'success' => true,
                    'type' => 'standard',
                    'message' => 'Lead consumed successfully'
                ]);
            }

            // 3. Seller Limit Reached -> Check Buyer's Contact Limit (Outgoing Bypass)
            $buyerPlan = $buyer->currentPlan;
            $buyerLimit = $buyerPlan ? $buyerPlan->contact_limit : 0;

            if ($buyerLimit > 0 && $buyer->contacts_initiated < $buyerLimit) {
                
                // Increment Buyer's Contacts Initiated
                $buyer->increment('contacts_initiated');

                // Send Automated Message (VIP Bypass)
                $this->sendAutomatedMessage($buyer, $seller, $ad, 'vip_bypass');

                return response()->json([
                    'success' => true,
                    'type' => 'vip_bypass',
                    'message' => 'VIP Contact used successfully'
                ]);
            }

            // 4. Both Limits Reached
            return response()->json([
                'success' => false,
                'reason' => 'limit_reached',
                'message' => 'Seller is busy and you have no VIP contacts remaining.'
            ], 403);

        } catch (\Exception $e) {
            Log::error('LeadController Error: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'An error occurred',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Check if monthly cycle has passed and reset counters if necessary.
     */
    private function checkAndResetCycle(User $user)
    {
        $now = Carbon::now();
        
        // If cycle_reset_at is null, set it to next month
        if (!$user->cycle_reset_at) {
            $user->cycle_reset_at = $now->copy()->addMonth();
            $user->save();
            return;
        }

        // If current time is past the reset date
        if ($now->greaterThan($user->cycle_reset_at)) {
            $user->leads_received = 0;
            $user->contacts_initiated = 0;
            $user->cycle_reset_at = $now->copy()->addMonth();
            $user->save();
        }
    }

    /**
     * Send automated chat message via FirebaseChatService.
     */
    private function sendAutomatedMessage(User $buyer, User $seller, Ad $ad, string $type)
    {
        try {
            $messageText = "Hi, I'm interested in your ad: " . $ad->title . "\n\n";
            $messageText .= "My Contact: " . ($buyer->mobile_number ?? $buyer->phone_number ?? 'N/A') . "\n";
            $messageText .= "Seller Contact: " . ($seller->mobile_number ?? $seller->phone_number ?? 'N/A');

            if ($type === 'vip_bypass') {
                $messageText = "Hi! I'm a VIP Buyer and I'm very interested in your ad: " . $ad->title . "\n\n";
                $messageText .= "VIP Contact Info:\n";
                $messageText .= "My Number: " . ($buyer->mobile_number ?? $buyer->phone_number ?? 'N/A') . "\n";
                $messageText .= "Seller Number: " . ($seller->mobile_number ?? $seller->phone_number ?? 'N/A');
            }

            // Initialize Chat
            $chatData = $this->firebaseChatService->initializeChat($buyer->id, $seller->id, $ad->id);

            // Send Message
            $this->firebaseChatService->sendMessage(
                $chatData['chatId'],
                $buyer->id,
                $seller->id,
                $messageText,
                'text'
            );

        } catch (\Exception $e) {
            Log::error('Automated Message Error: ' . $e->getMessage());
            // Don't fail the request if chat fails, just log it
        }
    }
}
