<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class CategoryController extends Controller
{
    /**
     * Get all categories with subcategories and counts.
     */
    public function index()
    {
        $categories = \App\Models\Category::withCount('ads')
            ->with(['subcategories' => function($query) {
                $query->select('id', 'category_id', 'name', 'slug', 'icon');
            }])
            ->get();

        return response()->json([
            'success' => true,
            'data' => $categories
        ]);
    }

    /**
     * Get single category
     */
    public function show($id)
    {
        $category = \App\Models\Category::withCount('ads')
            ->with(['subcategories' => function($query) {
                // Determine if we need to filter or sort subcategories
                $query->orderBy('name');
            }])
            ->find($id);

        if (!$category) {
            return response()->json(['message' => 'Category not found'], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $category
        ]);
    }

    /**
     * Get category attributes/fields
     */
    public function getAttributes(Request $request, $id)
    {
        $subcategoryId = $request->query('subcategory_id');

        $query = \App\Models\CategoryField::query()
            ->where('is_active', true)
            ->orderBy('sort_order');

        if ($subcategoryId) {
            // If subcategory is selected, get fields specific to it PLUS category-wide fields
            $query->where(function($q) use ($id, $subcategoryId) {
                $q->where('subcategory_id', $subcategoryId)
                  ->orWhere(function($subQ) use ($id) {
                      $subQ->where('category_id', $id)
                           ->whereNull('subcategory_id');
                  });
            });
        } else {
            // General category fields
            $query->where('category_id', $id)
                  ->whereNull('subcategory_id');
        }

        $fields = $query->get();

        return response()->json([
            'success' => true,
            'data' => $fields
        ]);
    }

    /**
     * Get subcategories for a specific category
     */
    public function getSubcategories($id)
    {
        $subcategories = \App\Models\Subcategory::where('category_id', $id)
            ->select('id', 'category_id', 'name', 'slug', 'icon')
            ->orderBy('name')
            ->get();

        return response()->json([
            'success' => true,
            'data' => $subcategories
        ]);
    }
}
