<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Blog;
use App\Models\BlogCategory;
use Illuminate\Http\Request;

class BlogController extends Controller
{
    /**
     * Get paginated blog list
     */
    public function index(Request $request)
    {
        $query = Blog::with(['category', 'author'])->published()->recent();

        // Filter by category
        if ($request->filled('category_id')) {
            $query->where('blog_category_id', $request->category_id);
        }

        $blogs = $query->paginate(10);

        return response()->json([
            'success' => true,
            'data' => $blogs
        ]);
    }

    /**
     * Get single blog by slug with ad units
     */
    public function show($slug)
    {
        $blog = Blog::with(['category', 'author', 'userPlan', 'package'])
            ->where('slug', $slug)
            ->published()
            ->first();

        if (!$blog) {
            return response()->json([
                'success' => false,
                'message' => 'Blog not found'
            ], 404);
        }

        // Get ad unit details
        $adUnits = [];
        if ($blog->ad_unit_top_banner_id) {
            $adUnits['top_banner'] = \DB::table('ad_units')->find($blog->ad_unit_top_banner_id);
        }
        if ($blog->ad_unit_in_content_native_id) {
            $adUnits['in_content_native'] = \DB::table('ad_units')->find($blog->ad_unit_in_content_native_id);
        }
        if ($blog->ad_unit_bottom_interstitial_id) {
            $adUnits['bottom_interstitial'] = \DB::table('ad_units')->find($blog->ad_unit_bottom_interstitial_id);
        }

        return response()->json([
            'success' => true,
            'data' => [
                'blog' => $blog,
                'ad_units' => $adUnits,
                'reading_time' => $blog->reading_time
            ]
        ]);
    }

    /**
     * Get active blog categories
     */
    public function categories()
    {
        $categories = BlogCategory::active()
            ->ordered()
            ->withCount(['blogs' => function ($query) {
                $query->published();
            }])
            ->get();

        return response()->json([
            'success' => true,
            'data' => $categories
        ]);
    }

    /**
     * Increment views count
     */
    public function incrementViews($id)
    {
        $blog = Blog::find($id);
        
        if ($blog) {
            $blog->incrementViews();
            return response()->json(['success' => true]);
        }

        return response()->json(['success' => false], 404);
    }
}
