<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use App\Models\Banner;
use App\Models\BannerImpression;
use App\Models\BannerClick;
use Illuminate\Support\Str;

class BannerDisplayController extends Controller
{
    /**
     * Get active banners for a location
     */
    public function getActiveBanners(Request $request)
    {
        \Illuminate\Support\Facades\Log::info('Banner API Hit', $request->all());
        $country = $request->input('country');
        $state = $request->input('state');
        $city = $request->input('city');

        $banners = Banner::active()
            ->forLocation($country, $state, $city)
            ->byPriority()
            ->where(function ($query) {
                $query->where('start_date', '<=', now())
                    ->orWhereNull('start_date');
            })
            ->where(function ($query) {
                $query->where('end_date', '>=', now())
                    ->orWhereNull('end_date');
            })
            ->where(function ($query) {
                $query->whereColumn('spent_amount', '<', 'total_budget')
                      ->orWhere('is_admin_created', true);
            })
            ->with(['package'])
            ->limit(5)
            ->get();

        return response()->json([
            'success' => true,
            'data' => $banners,
        ]);
    }

    /**
     * Record a banner impression
     */
    public function recordImpression(Request $request, $bannerId)
    {
        $banner = Banner::find($bannerId);
        
        if (!$banner || $banner->status !== 'active') {
            return response()->json(['success' => false, 'message' => 'Banner not found'], 404);
        }

        // Create impression record
        $impression = BannerImpression::create([
            'banner_id' => $bannerId,
            'user_id' => auth()->id(),
            'session_id' => $request->input('session_id', Str::uuid()),
            'ip_address' => $request->ip(),
            'user_agent' => $request->userAgent(),
            'country' => $request->input('country'),
            'state' => $request->input('state'),
            'city' => $request->input('city'),
        ]);

        // Update banner metrics if billing is impression-based and not admin created
        if ($banner->billing_model === 'impression' && $banner->cost_per_impression && !$banner->is_admin_created) {
            $cost = $banner->cost_per_impression;
            $banner->increment('total_impressions');
            $banner->increment('spent_amount', $cost);
            
            // Update CTR
            if ($banner->total_impressions > 0) {
                $banner->ctr = ($banner->total_clicks / $banner->total_impressions) * 100;
                $banner->save();
            }
        } else {
            // Just increment count without billing
            $banner->increment('total_impressions');
        }

        return response()->json([
            'success' => true,
            'impression_id' => $impression->id,
        ]);
    }

    /**
     * Record a banner click
     */
    public function recordClick(Request $request, $bannerId)
    {
        $banner = Banner::find($bannerId);
        
        if (!$banner || $banner->status !== 'active') {
            return response()->json(['success' => false, 'message' => 'Banner not found'], 404);
        }

        // Create click record
        $click = BannerClick::create([
            'banner_id' => $bannerId,
            'impression_id' => $request->input('impression_id'),
            'user_id' => auth()->id(),
            'session_id' => $request->input('session_id', Str::uuid()),
            'ip_address' => $request->ip(),
            'country' => $request->input('country'),
            'state' => $request->input('state'),
            'city' => $request->input('city'),
        ]);

        // Update banner metrics
        $banner->increment('total_clicks');
        
        // Bill for click if click-based billing (and not admin)
        if ($banner->billing_model === 'click' && $banner->cost_per_click && !$banner->is_admin_created) {
            $cost = $banner->cost_per_click;
            $banner->increment('spent_amount', $cost);
        }
        
        // Update CTR
        if ($banner->total_impressions > 0) {
            $banner->ctr = ($banner->total_clicks / $banner->total_impressions) * 100;
            $banner->save();
        }

        // Return redirect information
        return response()->json([
            'success' => true,
            'click_id' => $click->id,
            'redirect_type' => $banner->redirect_type,
            'redirect_value' => $banner->redirect_value,
        ]);
    }
}
