<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Ad;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class AdController extends Controller
{
    /**
     * Get all ads (public - accessible without authentication)
     */
    public function index(Request $request)
    {
        Log::info('AdController::index - Request Params:', $request->all());

        try {
            $query = Ad::with(['user.currentPlan', 'user.trustScore', 'category', 'images'])
                ->where('ads.status', 'Active')
                ->where('ads.is_active', true);

            // 1. Filter by Location Radius (Prioritized)
            $lat = request('latitude');
            $lng = request('longitude');
            $radius = request('radius') ?? setting('location_range_default', 50); // Default 50km

            Log::info("AdController::index - Filtering: Lat: $lat, Lng: $lng, Radius: $radius, City: " . request('city'));

            if ($lat && $lng) {
                Log::info("AdController::index - Applying Radius Filter");
                // Filter ads strictly within the radius
                $query->nearLocation($lat, $lng, $radius);
            } 
            // 2. Fallback: Filter by City/Area string matching if no coordinates
            else if (request()->has('city') && !empty(request('city'))) {
                 Log::info("AdController::index - Applying City Fallback Filter");
                 $city = request('city');
                 $query->where(function($q) use ($city) {
                     $q->where('location_city', 'like', "%{$city}%")
                       ->orWhere('city', 'like', "%{$city}%")
                       ->orWhere('location_address', 'like', "%{$city}%"); // Check address too
                 });
            } else {
                 Log::info("AdController::index - No Location Filter Applied");
            }

            
            // This scope handles boosting, freshness, membership priority, and location sorting
            $query->orderByInterestAndLocation($lat, $lng);
            
            // Paginate results (20 per page)
            $ads = $query->paginate(20);

            Log::info("AdController::index - Results Count: " . $ads->count());

            // FALLBACK LOGIC: If no ads found in range, show broader results
            $fallbackTriggered = false;
            // Only trigger fallback if location filter was applied but yielded no results OR if count is too low
            if ($ads->total() === 0 && ($lat || request()->has('city'))) {
                Log::info("AdController::index - Fallback Triggered (No ads in range)");
                // Reset query to base state
                $fallbackQuery = Ad::with(['user.currentPlan', 'user.trustScore', 'category', 'images'])
                    ->where('ads.status', 'Active')
                    ->where('ads.is_active', true);
                
                // Still apply interest sorting (global)
                $fallbackQuery->orderByInterestAndLocation(null, null);
                
                $ads = $fallbackQuery->paginate(20);
                $fallbackTriggered = true;
            }

            // Check for authenticated user to determine favorite status
            $user = request()->user('sanctum');
            $favoriteAdIds = $user ? \App\Models\User::find($user->id)->favorites()->pluck('ad_id')->toArray() : [];

            // Transform data using the paginator's collection
            $transformedCollection = $ads->getCollection()->map(function($ad) use ($favoriteAdIds) {
                $mainImage = null;
                if ($ad->relationLoaded('images') && $ad->images->isNotEmpty()) {
                    $path = $ad->images->first()->image_url;
                    $mainImage = str_starts_with($path, 'http') ? $path : url('storage/' . $path);
                }

                return [
                    'id' => $ad->id,
                    'title' => $ad->title,
                    'description' => $ad->description ?? '',
                    'price' => (string) ($ad->price ?? 0),
                    'currency' => $ad->currency ?? 'USD',
                    'city' => $ad->location_city ?? $ad->city ?? null,
                    'area' => $ad->location_address ?? null,
                    'state' => $ad->location_state ?? $ad->state ?? null,
                    'country' => $ad->location_country ?? $ad->country ?? null,
                    'main_image' => $mainImage,
                    'slug' => 'ad-' . $ad->id,
                    'created_at' => $ad->created_at ? $ad->created_at->toDateTimeString() : now()->toDateTimeString(),
                    'is_featured' => (bool) ($ad->is_featured ?? false),
                    'view_count' => $ad->view_count ?? 0,
                    'category' => $ad->category ? $ad->category->name : null,
                    'is_favorited' => in_array($ad->id, $favoriteAdIds),
                    'user' => [
                        'id' => optional($ad->user)->id,
                        'name' => optional($ad->user)->name ?? 'Unknown',
                        'email' => optional($ad->user)->email ?? null,
                        'avatar' => optional($ad->user)->profile_image ? url('storage/' . $ad->user->profile_image) : null,
                        'membership_type' => optional($ad->user->currentPlan)->plan_name ?? 'Standard',
                        'trust_score' => optional($ad->user->trustScore)->score ?? 100, // Default to 100 for legacy/new users if no score
                        'trust_tier' => optional($ad->user->trustScore)->tier ?? 'new',
                        'member_since' => optional($ad->user->created_at)->format('Y') ?? '2024',
                        'rating' => 4.8, // Dummy for now
                        'review_count' => 124, // Dummy for now
                        'phone_number' => optional($ad->user)->phone_number,
                        'mobile_number' => optional($ad->user)->mobile_number,
                    ],
                ];
            });
            
            // Replace the collection in the paginator with transformed data
            $ads->setCollection($transformedCollection);

            return response()->json([
                'success' => true,
                'data' => $ads->items(), // The transformed items
                'meta' => [
                    'current_page' => $ads->currentPage(),
                    'last_page' => $ads->lastPage(),
                    'per_page' => $ads->perPage(),
                    'total' => $ads->total(),
                    'is_fallback' => $fallbackTriggered,
                    'message' => $fallbackTriggered ? 'No ads found in your area. Showing fresh finds from everywhere.' : null
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch ads',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get public ads for a specific user (Seller Profile)
     */
    /**
     * Get public ads for a specific user (Seller Profile)
     */
    public function getSellerProfile($userId)
    {
        try {
            $user = \App\Models\User::with(['trustScore', 'currentPlan'])->findOrFail($userId);
            
            // Get stats
            $activeAdsCount = Ad::where('user_id', $userId)->where('status', 'Active')->count();
            $soldAdsCount = Ad::where('user_id', $userId)->where('status', 'Sold')->count();
            
            // Get Ads
            $ads = Ad::with(['category', 'images'])
                ->where('user_id', $userId)
                ->where('status', 'Active')
                ->where('is_active', true)
                ->orderBy('is_featured', 'desc') // Featured first
                ->orderBy('created_at', 'desc')
                ->paginate(20);

            $transformedAds = $ads->getCollection()->map(function($ad) use ($user) {
                // Image logic
                $mainImage = null;
                if ($ad->relationLoaded('images') && $ad->images->isNotEmpty()) {
                    $path = $ad->images->first()->image_url;
                    $mainImage = str_starts_with($path, 'http') ? $path : url('storage/' . $path);
                }

                return [
                    'id' => $ad->id,
                    'title' => $ad->title,
                    'price' => (string) ($ad->price ?? 0),
                    'currency' => $ad->currency ?? 'USD',
                    'city' => $ad->location_city ?? $ad->city ?? null,
                    'area' => $ad->location_address ?? null,
                    'main_image' => $mainImage,
                    'slug' => 'ad-' . $ad->id,
                    'created_at' => $ad->created_at ? $ad->created_at->toDiffForHumans() : 'Just now',
                    'is_featured' => (bool) ($ad->is_featured ?? false),
                    'is_boosted' => (bool) ($ad->is_boosted ?? false),
                    'views' => $ad->view_count ?? 0,
                    'category' => $ad->category ? $ad->category->name : null,
                    'user' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'avatar' => $user->profile_image ? url('storage/' . $user->profile_image) : null,
                        'phone_number' => $user->phone_number,
                        'mobile_number' => $user->mobile_number,
                    ],
                ];
            });

            $ads->setCollection($transformedAds);

            return response()->json([
                'success' => true,
                'data' => [
                    'profile' => [
                        'id' => $user->id,
                        'name' => $user->name,
                        'avatar' => $user->profile_image ? url('storage/' . $user->profile_image) : null,
                        'city' => $user->city ?? 'Unknown',
                        'country' => $user->country ?? 'India',
                        'member_since' => optional($user->created_at)->format('M Y') ?? 'Jan 2024',
                        'response_rate' => '98%', 
                        'response_time' => '1 hour', 
                        'rating' => 4.9, 
                        'rating_count' => 124, 
                        'bio' => $user->bio ?? 'No bio enabled.',
                        'followers_count' => $user->followers_count ?? 0,
                        'following_count' => $user->following_count ?? 0,
                        'trust_score' => optional($user->trustScore)->score ?? 100,
                        'trust_tier' => optional($user->trustScore)->tier ?? 'new',
                        'is_verified' => (bool) $user->hasPlanFeature('verified_badge'),
                        'plan_name' => optional($user->currentPlan)->plan_name ?? 'Standard',
                        'active_ads_count' => $activeAdsCount,
                        'sold_ads_count' => $soldAdsCount,
                    ],
                    'ads' => $ads->items(),
                    'meta' => [
                        'current_page' => $ads->currentPage(),
                        'last_page' => $ads->lastPage(),
                    ]
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch seller profile',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Get authenticated user's ads (requires authentication)
     */
    public function getMyAds(Request $request)
    {
        try {
            $user = $request->user();
            Log::info('MyAds Request - User ID: ' . ($user ? $user->id : 'null'));
            
            if (!$user) {
                return response()->json([
                    'success' => false,
                    'message' => 'Unauthorized',
                ], 401);
            }

            $ads = Ad::with(['user.currentPlan', 'user.trustScore', 'category', 'images'])
                ->where('user_id', $user->id)
                ->withCount('favoritedBy')
                ->orderBy('ads.created_at', 'desc')
                ->get();
            
            Log::info('MyAds Request - Ads Found Query Count: ' . $ads->count());
            Log::info('MyAds Request - First Ad ID: ' . ($ads->first() ? $ads->first()->id : 'None'));

            $transformedAds = $ads->map(function($ad) {
                // ... (image logic existing)
                $mainImage = null;
                if ($ad->relationLoaded('images') && $ad->images->isNotEmpty()) {
                    $path = $ad->images->first()->image_url;
                    $mainImage = str_starts_with($path, 'http') ? $path : url('storage/' . $path);
                }
                
                $adUser = $ad->user;

                return [
                    'id' => $ad->id,
                    'title' => $ad->title,
                    'description' => $ad->description ?? '',
                    'price' => (string) ($ad->price ?? 0),
                    'currency' => $ad->currency ?? 'USD',
                    'city' => $ad->location_city ?? $ad->city ?? null,
                    'area' => $ad->location_address ?? null,
                    'state' => $ad->location_state ?? $ad->state ?? null,
                    'country' => $ad->location_country ?? $ad->country ?? null,
                    'main_image' => $mainImage,
                    'slug' => 'ad-' . $ad->id,
                    'created_at' => $ad->created_at ? $ad->created_at->toDateTimeString() : now()->toDateTimeString(),
                    'is_featured' => (bool) ($ad->is_featured ?? false),
                    'view_count' => $ad->view_count ?? 0,
                    'favorites_count' => $ad->favorited_by_count ?? 0,
                    'category' => $ad->category ? $ad->category->name : null,
                    'status' => $ad->status ?? 'Active',
                    'user' => [
                        'id' => $adUser ? $adUser->id : null,
                        'name' => $adUser ? $adUser->name : 'Unknown',
                        'email' => $adUser ? $adUser->email : null,
                        'avatar' => ($adUser && $adUser->profile_image) ? url('storage/' . $adUser->profile_image) : null,
                        'membership_type' => optional($adUser->currentPlan)->plan_name ?? 'Standard',
                        'trust_score' => optional($adUser->trustScore)->score ?? 100,
                        'trust_tier' => optional($adUser->trustScore)->tier ?? 'new',
                        'member_since' => optional($adUser->created_at)->format('Y') ?? '2024',
                        'rating' => 4.8,
                        'review_count' => 124,
                        'phone_number' => $adUser ? $adUser->phone_number : null,
                        'mobile_number' => $adUser ? $adUser->mobile_number : null,
                    ],
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedAds,
                'count' => $transformedAds->count(),
            ]);
        } catch (\Exception $e) {
            Log::error('AdController::getMyAds error: ' . $e->getMessage() . ' Trace: ' . $e->getTraceAsString());
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch your ads',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Search ads (public)
     */
    public function search(Request $request)
    {
        try {
            $query = Ad::with(['user', 'category', 'images'])
                ->where('status', 'Active')
                ->where('is_active', true);

            if ($request->has('q')) {
                $searchTerm = $request->q;
                $query->where(function($q) use ($searchTerm) {
                    $q->where('title', 'like', "%{$searchTerm}%")
                      ->orWhere('description', 'like', "%{$searchTerm}%");
                });
            }

            if ($request->has('category_id')) {
                $query->where('category_id', $request->category_id);
            }

            $ads = $query->orderBy('created_at', 'desc')->limit(50)->get();

            // Check favorites
            $user = request()->user('sanctum');
            $favoriteAdIds = $user ? \App\Models\User::find($user->id)->favorites()->pluck('ad_id')->toArray() : [];

            $transformedAds = $ads->map(function($ad) use ($favoriteAdIds) {
                return [
                    'id' => $ad->id,
                    'title' => $ad->title,
                    'description' => $ad->description,
                    'price' => (string) $ad->price,
                    'currency' => $ad->currency,
                    'city' => $ad->location_city,
                    'area' => $ad->location_address ?? null,
                    'state' => $ad->location_state,
                    'main_image' => $ad->images->isNotEmpty() 
                        ? (str_starts_with($ad->images->first()->image_url, 'http') 
                            ? $ad->images->first()->image_url 
                            : url('storage/' . $ad->images->first()->image_url))
                        : null,
                    'slug' => $ad->slug ?? 'ad-' . $ad->id,
                    'created_at' => $ad->created_at->toDateTimeString(),
                    'is_featured' => $ad->is_featured,
                    'is_favorited' => in_array($ad->id, $favoriteAdIds),
                    'user' => [
                        'id' => $ad->user_id,
                        'name' => optional($ad->user)->name ?? 'Unknown',
                        'email' => optional($ad->user)->email ?? null,
                        'avatar' => optional($ad->user)->profile_image ? url('storage/' . $ad->user->profile_image) : null,
                        'membership_type' => optional($ad->user->currentPlan)->plan_name ?? 'Standard',
                        'trust_score' => optional($ad->user->trustScore)->score ?? 100,
                        'trust_tier' => optional($ad->user->trustScore)->tier ?? 'new',
                        'member_since' => optional($ad->user->created_at)->format('Y') ?? '2024',
                        'rating' => 4.8,
                        'review_count' => 124,
                        'phone_number' => optional($ad->user)->phone_number,
                        'mobile_number' => optional($ad->user)->mobile_number,
                    ],
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedAds,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Search failed',
            ], 500);
        }
    }

    /**
     * Store a new ad (requires authentication)
     */
    public function store(Request $request)
    {
        Log::info('AdController::store - Creating new ad');

        try {
            $user = $request->user();
            if (!$user) {
                return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
            }

            // Validation with correct image limits
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'description' => 'required|string',
                'price' => 'required|numeric|min:0',
                'currency' => 'required|string|size:3',
                'category_id' => 'required|exists:categories,id',
                'subcategory_id' => 'nullable|exists:subcategories,id',
                'city' => 'required|string',
                'latitude' => 'required|numeric',
                'longitude' => 'required|numeric',
                'images' => 'required|array|min:1|max:10', // Enforce Min 1, Max 10 rule
                'images.*' => 'image|mimes:jpeg,png,jpg,webp|max:5120', // Max 5MB per image
                'attributes' => 'nullable|array',
            ]);

            // Create Ad
            
            // --- AI Guardian Integration ---
            $guardianEnabled = \App\Models\Setting::where('key', 'guardian_enabled')->value('value');
            if ($guardianEnabled == '1') {
                $guardian = new \App\Services\AiGuardianService(); // Should inject, but direct class for speed
                $analysis = $guardian->analyzeAd($validated['title'], $validated['description'], $user);

                if ($analysis['action'] === 'AUTO_BAN') {
                   // $user->update(['status' => 'banned']); // Optional: Ban logic
                    return response()->json([
                        'success' => false, 
                        'message' => 'Your account has been flagged for posting prohibited content.',
                        'error_code' => 'ACCOUNT_FLAGGED'
                    ], 403);
                }

                if ($analysis['action'] === 'SUGGEST_CHANGES') {
                    return response()->json([
                        'success' => false,
                        'message' => 'Please improve your ad content to proceed.',
                        'suggestions' => $analysis['suggestions'],
                        'error_code' => 'CONTENT_IMPROVEMENT_NEEDED'
                    ], 422);
                }

                // If FLAG_FOR_REVIEW, allow creation but force status to Pending
                if ($analysis['action'] === 'FLAG_FOR_REVIEW') {
                    $initialStatus = 'Pending';
                } else {
                    $initialStatus = 'Active'; // APPROVE
                }
            } else {
                // Legacy Logic
                $initialStatus = (\App\Models\Setting::where('key', 'auto_approve_ads')->value('value') == '1') ? 'Active' : 'Pending';
            }
            // -------------------------------

            $ad = Ad::create([
                'user_id' => $user->id,
                'category_id' => $validated['category_id'],
                'subcategory_id' => $validated['subcategory_id'] ?? null,
                'title' => $validated['title'],
                'description' => $validated['description'],
                'price' => $validated['price'],
                'currency' => $validated['currency'],
                'location_city' => $validated['city'],
                'location_state' => $request->state, // Optional or derived
                'location_country' => $request->country ?? 'India',
                'location_address' => $request->area,
                'latitude' => $validated['latitude'],
                'longitude' => $validated['longitude'],
                'status' => $initialStatus,
                'is_active' => true,
            ]);

            // Handle Image Uploads
            if ($request->hasFile('images')) {
                foreach ($request->file('images') as $index => $image) {
                    $path = $image->store('ads', 'public');
                    
                    \App\Models\AdImage::create([
                        'ad_id' => $ad->id,
                        'image_url' => $path,
                        'is_primary' => ($index === 0),
                    ]);
                }
            }

            // Handle Custom Attributes
            if ($request->has('attributes') && is_array($request->attributes)) {
                foreach ($request->attributes as $fieldId => $value) {
                    if (!empty($value)) {
                         \App\Models\AdAttributeValue::create([
                            'ad_id' => $ad->id,
                            'field_id' => $fieldId,
                            'value' => $value,
                        ]);
                    }
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Ad posted successfully! It is pending approval.',
                'data' => [
                    'id' => $ad->id,
                    'slug' => 'ad-' . $ad->id
                ]
            ], 201);

        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Ad creation failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create ad',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show single ad details
     */
    public function show($id)
    {
        try {
        $ad = Ad::with(['user.currentPlan', 'user.trustScore', 'category', 'images', 'attributes.field'])->find($id);

        if (!$ad) {
            return response()->json([
                'success' => false,
                'message' => 'Ad not found',
            ], 404);
        }

        $user = request()->user('sanctum');
        $isFavorited = false;
        
        if ($user) {
            $isFavorited = \App\Models\User::find($user->id)->favorites()->where('ad_id', $ad->id)->exists();
        }

        // Transform attributes
        $attributes = [];
        if ($ad->attributes) {
            foreach ($ad->attributes as $attr) {
                if ($attr->field) {
                    $attributes[$attr->field->name] = $attr->value;
                }
            }
        }

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $ad->id,
                'title' => $ad->title,
                'description' => $ad->description,
                'price' => (string) $ad->price,
                'currency' => $ad->currency,
                'city' => $ad->location_city,
                'state' => $ad->location_state,
                'latitude' => $ad->latitude,
                'longitude' => $ad->longitude,
                'main_image' => $ad->images->isNotEmpty() 
                    ? (str_starts_with($ad->images->first()->image_url, 'http') 
                        ? $ad->images->first()->image_url 
                        : url('storage/' . $ad->images->first()->image_url))
                    : null,
                'images' => $ad->images->map(function($img) {
                    return [
                        'id' => $img->id,
                        'url' => str_starts_with($img->image_url, 'http') ? $img->image_url : url('storage/' . $img->image_url)
                    ];
                }),
                'slug' => $ad->slug ?? 'ad-' . $ad->id,
                'created_at' => $ad->created_at->toDateTimeString(),
                'is_featured' => $ad->is_featured,
                'is_favorited' => $isFavorited,
                'view_count' => $ad->view_count ?? 0,
                'category' => $ad->category ? $ad->category->name : null,
                'attributes' => $attributes,
                'user' => [
                    'id' => $ad->user_id,
                    'name' => $ad->user->name ?? 'Unknown',
                    'email' => $ad->user->email ?? null,
                    'avatar' => $ad->user->avatar_url ?? null,
                    'membership_type' => optional($ad->user->currentPlan)->plan_name ?? 'Standard',
                    'trust_score' => optional($ad->user->trustScore)->score ?? 100,
                    'trust_tier' => optional($ad->user->trustScore)->tier ?? 'new',
                    'member_since' => optional($ad->user->created_at)->format('Y') ?? '2024',
                    'rating' => 4.8,
                    'review_count' => 124,
                    // Expose contact info for VIP feature
                    'phone_number' => $ad->user->phone_number,
                    'mobile_number' => $ad->user->mobile_number,
                ],
            ],
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Failed to fetch ad details',
        ], 500);
    }
}
    /**
     * Toggle ad status (Active/Inactive)
     */
    public function toggleStatus(Request $request, $id)
    {
        try {
            $user = $request->user();
            $ad = Ad::where('id', $id)->where('user_id', $user->id)->first();

            if (!$ad) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ad not found or unauthorized',
                ], 404);
            }

            // Toggle logic
            $newStatus = ($ad->status === 'Active') ? 'Inactive' : 'Active';
            
            // If activating, check if user has limits (optional, for now just toggle)
            $ad->status = $newStatus;
            $ad->save();

            return response()->json([
                'success' => true,
                'message' => 'Ad status updated',
                'status' => $newStatus,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to update ad status',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Update ad (requires authentication)
     */
    /**
     * Update ad (requires authentication)
     */
    public function update(Request $request, $id)
    {
        Log::info('AdController::update - Updating ad ' . $id);

        try {
            $user = $request->user();
            $ad = Ad::where('id', $id)->where('user_id', $user->id)->first();

            if (!$ad) {
                return response()->json(['success' => false, 'message' => 'Ad not found or unauthorized'], 404);
            }

            // Validation (Allow partial updates via 'sometimes')
            $validated = $request->validate([
                'title' => 'sometimes|required|string|max:255',
                'description' => 'sometimes|required|string',
                'price' => 'sometimes|required|numeric|min:0',
                'currency' => 'sometimes|required|string|size:3',
                'category_id' => 'sometimes|required|exists:categories,id',
                'subcategory_id' => 'nullable|exists:subcategories,id',
                'city' => 'sometimes|required|string',
                'latitude' => 'sometimes|required|numeric',
                'longitude' => 'sometimes|required|numeric',
                'new_images' => 'nullable|array|max:10',
                'new_images.*' => 'image|mimes:jpeg,png,jpg,webp|max:5120',
                'deleted_image_ids' => 'nullable|array',
                'deleted_image_ids.*' => 'integer|exists:ad_images,id',
                'attributes' => 'nullable|array',
            ]);

            // --- AI Guardian Check (if title/desc changed) ---
            if ($request->has('title') || $request->has('description')) {
                $guardianEnabled = \App\Models\Setting::where('key', 'guardian_enabled')->value('value');
                if ($guardianEnabled == '1') {
                    $guardian = new \App\Services\AiGuardianService();
                    $analysis = $guardian->analyzeAd(
                        $request->has('title') ? $validated['title'] : $ad->title,
                        $request->has('description') ? $validated['description'] : $ad->description,
                        $user
                    );

                    if ($analysis['action'] === 'AUTO_BAN') {
                        return response()->json([
                            'success' => false,
                            'message' => 'Update rejected. Content flagged as prohibited.',
                            'error_code' => 'ACCOUNT_FLAGGED'
                        ], 403);
                    }
                    if ($analysis['action'] === 'SUGGEST_CHANGES') {
                        return response()->json([
                            'success' => false,
                            'message' => 'Please improve your ad content.',
                            'suggestions' => $analysis['suggestions'],
                            'error_code' => 'CONTENT_IMPROVEMENT_NEEDED'
                        ], 422);
                    }
                }
            }
            // ------------------------------------------------

            // Update basic fields
            $ad->fill($request->only([
                'title', 'description', 'price', 'currency', 
                'category_id', 'subcategory_id', 'latitude', 'longitude'
            ]));
            
            if ($request->has('city')) {
                $ad->location_city = $request->city;
            }
            if ($request->has('state')) {
                $ad->location_state = $request->state;
            }
            if ($request->has('area')) {
                $ad->location_address = $request->area;
            }
            // Reset status to Pending if significant changes (optional logic, keeping simple for now)
            // $ad->status = 'Pending'; 

            $ad->save();

            // Handle Deleted Images
            if ($request->has('deleted_image_ids')) {
                $imagesToDelete = \App\Models\AdImage::whereIn('id', $request->deleted_image_ids)
                                    ->where('ad_id', $ad->id)
                                    ->get();
                
                foreach ($imagesToDelete as $img) {
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($img->image_url);
                    $img->delete();
                }
            }

            // Handle New Images
            if ($request->hasFile('new_images')) {
                foreach ($request->file('new_images') as $image) {
                     $path = $image->store('ads', 'public');
                     \App\Models\AdImage::create([
                        'ad_id' => $ad->id,
                        'image_url' => $path,
                        'is_primary' => false, // TODO: Logic to set primary if needed
                    ]);
                }
            }

            // Ensure at least one primary image exists
            if (!$ad->images()->where('is_primary', true)->exists()) {
                $firstImage = $ad->images()->first();
                if ($firstImage) {
                    $firstImage->update(['is_primary' => true]);
                }
            }

            // Handle Attributes (Full Replace for simplicity or Merge)
            // Here assuming merge/overwrite logic key-by-key
            if ($request->has('attributes') && is_array($request->attributes)) {
                 foreach ($request->attributes as $fieldId => $value) {
                    if (!empty($value)) {
                        \App\Models\AdAttributeValue::updateOrCreate(
                            ['ad_id' => $ad->id, 'field_id' => $fieldId],
                            ['value' => $value]
                        );
                    } else {
                        // Remove if empty
                        \App\Models\AdAttributeValue::where('ad_id', $ad->id)
                            ->where('field_id', $fieldId)->delete();
                    }
                }
            }

            return response()->json([
                'success' => true,
                'message' => 'Ad updated successfully',
                'data' => [
                    'id' => $ad->id,
                    'slug' => 'ad-' . $ad->id
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
             return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            Log::error('Ad update failed: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update ad',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    /**
     * Delete ad (requires authentication)
     */
    public function destroy(Request $request, $id)
    {
        try {
            $user = $request->user();
            $ad = Ad::where('id', $id)->where('user_id', $user->id)->first();

            if (!$ad) {
                return response()->json([
                    'success' => false,
                    'message' => 'Ad not found or unauthorized',
                ], 404);
            }

            // Delete associated images
            if ($ad->images) {
                foreach ($ad->images as $image) {
                    \Illuminate\Support\Facades\Storage::disk('public')->delete($image->image_url);
                    $image->delete();
                }
            }

            $ad->delete();

            return response()->json([
                'success' => true,
                'message' => 'Ad deleted successfully',
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete ad',
                'error' => $e->getMessage(),
            ], 500);
        }
        }

    /**
     * Toggle favorite status
     */
    public function toggleFavorite(Request $request, $id)
    {
        try {
            $user = $request->user();
            if (!$user) {
                return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
            }
            
            // Check if ad exists
            $ad = Ad::find($id);
            if (!$ad) {
                 return response()->json(['success' => false, 'message' => 'Ad not found'], 404);
            }

            // Using the 'favorites' relationship on User model (assuming it exists, otherwise use DB or attach)
            $isFavorited = $user->favorites()->where('ad_id', $id)->exists();

            if ($isFavorited) {
                $user->favorites()->detach($id);
                $status = 'removed';
            } else {
                $user->favorites()->attach($id);
                $status = 'added';
            }

            return response()->json([
                'success' => true,
                'status' => $status,
                'is_favorited' => !$isFavorited
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Action failed',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get user's favorite ads
     */
    public function getFavorites(Request $request) 
    {
        try {
            $user = $request->user();
            if (!$user) return response()->json(['success' => false, 'message' => 'Unauthorized'], 401);
            
            $ads = $user->favorites()->with(['user.currentPlan', 'category', 'images'])
                        ->orderBy('pivot_created_at', 'desc') // Assuming pivot timestamp
                        ->get();
                        
            // Transform
             $transformedAds = $ads->map(function($ad) {
                $mainImage = null;
                if ($ad->relationLoaded('images') && $ad->images->isNotEmpty()) {
                    $path = $ad->images->first()->image_url;
                    $mainImage = str_starts_with($path, 'http') ? $path : url('storage/' . $path);
                }

                return [
                    'id' => $ad->id,
                    'title' => $ad->title,
                    'description' => $ad->description ?? '',
                    'price' => (string) ($ad->price ?? 0),
                    'currency' => $ad->currency ?? 'USD',
                    'city' => $ad->location_city ?? $ad->city ?? null,
                    'area' => $ad->location_address ?? null,
                    'state' => $ad->location_state ?? $ad->state ?? null,
                    'main_image' => $mainImage,
                    'slug' => 'ad-' . $ad->id,
                    'created_at' => $ad->created_at ? $ad->created_at->toDateTimeString() : now()->toDateTimeString(),
                    'is_featured' => (bool) ($ad->is_featured ?? false),
                    'is_favorited' => true,
                    'user' => [
                        'id' => $ad->user_id,
                        'name' => optional($ad->user)->name ?? 'Unknown',
                        'email' => optional($ad->user)->email ?? null,
                        'avatar' => optional($ad->user)->profile_image ? url('storage/' . $ad->user->profile_image) : null,
                        'membership_type' => optional($ad->user->currentPlan)->plan_name ?? 'Standard',
                        'trust_score' => optional($ad->user->trustScore)->score ?? 100,
                        'trust_tier' => optional($ad->user->trustScore)->tier ?? 'new',
                        'member_since' => optional($ad->user->created_at)->format('Y') ?? '2024',
                        'rating' => 4.8,
                        'review_count' => 124,
                        'phone_number' => optional($ad->user)->phone_number,
                        'mobile_number' => optional($ad->user)->mobile_number,
                    ],
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedAds,
            ]);
            
        } catch (\Exception $e) {
             return response()->json(['success' => false, 'message' => 'Failed to fetch favorites'], 500);
        }
    }

    /**
     * Get related ads algorithmically
     */
    public function getRelated(Request $request, $id)
    {
        try {
            $ad = Ad::find($id);
            if (!$ad) {
                return response()->json(['success' => false, 'message' => 'Ad not found'], 404);
            }

            $query = Ad::with(['user.currentPlan', 'category', 'images'])
                ->where('ads.id', '!=', $id)
                ->where('ads.status', 'Active')
                ->where('ads.is_active', true)
                ->where('category_id', $ad->category_id);

            // Use current ad coordinates for algorithmic ranking
            $query->orderByInterestAndLocation($ad->latitude, $ad->longitude);

            $ads = $query->limit(10)->get();

            // Check for authenticated user to determine favorite status
            $user = request()->user('sanctum');
            $favoriteAdIds = $user ? \App\Models\User::find($user->id)->favorites()->pluck('ad_id')->toArray() : [];

            $transformedAds = $ads->map(function($ad) use ($favoriteAdIds) {
                $mainImage = null;
                if ($ad->relationLoaded('images') && $ad->images->isNotEmpty()) {
                    $path = $ad->images->first()->image_url;
                    $mainImage = str_starts_with($path, 'http') ? $path : url('storage/' . $path);
                }

                return [
                    'id' => $ad->id,
                    'title' => $ad->title,
                    'description' => $ad->description ?? '',
                    'price' => (string) ($ad->price ?? 0),
                    'currency' => $ad->currency ?? 'USD',
                    'city' => $ad->location_city ?? $ad->city ?? null,
                    'area' => $ad->location_address ?? null,
                    'state' => $ad->location_state ?? $ad->state ?? null,
                    'main_image' => $mainImage,
                    'slug' => 'ad-' . $ad->id,
                    'created_at' => $ad->created_at ? $ad->created_at->toDateTimeString() : now()->toDateTimeString(),
                    'is_featured' => (bool) ($ad->is_featured ?? false),
                    'is_favorited' => in_array($ad->id, $favoriteAdIds),
                    'user' => [
                        'id' => $ad->user_id,
                        'name' => optional($ad->user)->name ?? 'Unknown',
                        'email' => optional($ad->user)->email ?? null,
                        'avatar' => optional($ad->user)->profile_image ? url('storage/' . $ad->user->profile_image) : null,
                        'membership_type' => optional($ad->user->currentPlan)->plan_name ?? 'Standard',
                        'trust_score' => optional($ad->user->trustScore)->score ?? 100,
                        'trust_tier' => optional($ad->user->trustScore)->tier ?? 'new',
                    ],
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $transformedAds
            ]);
        } catch (\Exception $e) {
            return response()->json(['success' => false, 'message' => 'Failed to fetch related ads'], 500);
        }
    }
    protected $recommendationEngine;
    protected $notificationService;

    public function __construct(
        \App\Services\RecommendationEngine $recommendationEngine,
        \App\Services\NotificationService $notificationService
    ) {
        $this->recommendationEngine = $recommendationEngine;
        $this->notificationService = $notificationService;
    }

    /**
     * Get algorithmic recommendations for an ad (requires authentication & ownership/admin)
     */
    public function getRecommendations(Request $request, $id)
    {
        try {
            $user = $request->user();
            $ad = Ad::find($id);

            if (!$ad) {
                return response()->json(['success' => false, 'message' => 'Ad not found'], 404);
            }

            // Authorization: Only owner or admin can see recommendations
            if ($ad->user_id !== $user->id && !$user->hasRole('admin')) { 
                return response()->json(['success' => false, 'message' => 'Unauthorized'], 403);
            }

            $recommendations = $this->recommendationEngine->getAdOptimizationRecommendations($ad);

            // Format for easy frontend consumption
            $formattedRecs = [];
            foreach ($recommendations as $key => $message) {
                $icon = 'lightbulb_outline'; // Default
                if (str_contains($key, 'price')) $icon = 'attach_money';
                if (str_contains($key, 'image')) $icon = 'image';
                if (str_contains($key, 'location')) $icon = 'location_on';
                if (str_contains($key, 'boost')) $icon = 'rocket_launch';
                if (str_contains($key, 'timing')) $icon = 'access_time';

                $formattedRecs[] = [
                    'type' => $key,
                    'message' => $message,
                    'icon' => $icon
                ];
            }

            return response()->json([
                'success' => true,
                'data' => $formattedRecs
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to fetch recommendations',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mark ad as sold and notify parties
     */
    public function markAsSold(Request $request, $id)
    {
        $request->validate([
            'buyer_id' => 'required|exists:users,id',
        ]);

        $user = $request->user();
        $ad = Ad::where('id', $id)->where('user_id', $user->id)->first();

        if (!$ad) {
            return response()->json(['success' => false, 'message' => 'Ad not found or unauthorized'], 404);
        }

        if ($ad->buyer_id == $request->buyer_id) {
             return response()->json(['success' => true, 'message' => 'Already marked as sold to this buyer']);
        }

        $ad->update([
            'status' => 'Sold',
            'buyer_id' => $request->buyer_id,
            'is_active' => false, 
        ]);

        // Notify Buyer
        $buyer = \App\Models\User::find($request->buyer_id);
        if ($buyer) {
            $this->notificationService->sendNotification(
                $this->notificationService->createNotification([
                    'title' => 'Purchase Confirmed!',
                    'message' => "You bought '{$ad->title}'. Please rate your experience with {$user->name}.",
                    'target_type' => 'single_user',
                    'target_value' => ['user_ids' => [$buyer->id]],
                    'deep_link_url' => "clx://review/submit?ad_id={$ad->id}&user_id={$user->id}&type=seller",
                    'category' => 'review_request'
                ])
            );
        }

        // Notify Seller
        $this->notificationService->sendNotification(
            $this->notificationService->createNotification([
                'title' => 'Item Sold!',
                'message' => "You sold '{$ad->title}' to {$buyer->name}. Don't forget to leave a review!",
                'target_type' => 'single_user',
                'target_value' => ['user_ids' => [$user->id]],
                'deep_link_url' => "clx://review/submit?ad_id={$ad->id}&user_id={$buyer->id}&type=buyer",
                'category' => 'review_request'
            ])
        );

        return response()->json([
            'success' => true,
            'message' => 'Ad marked as sold successfully',
            'data' => $ad
        ]);
    }

    /**
     * Boost ad using plan credit (Shortcut)
     */
    public function boostAd(Request $request, $id)
    {
        $user = $request->user();
        $ad = Ad::where('id', $id)->where('user_id', $user->id)->first();

        if (!$ad) {
            return response()->json(['success' => false, 'message' => 'Ad not found or unauthorized'], 404);
        }

        if (!$user->hasBoostCredit()) {
             return response()->json(['success' => false, 'message' => 'No boost credits available. Please purchase a package.'], 402);
        }

        // Use credit
        $user->useBoostCredit();

        // Apply Boost (Default 7 Days)
        $package = \App\Models\AdPackage::where('type', 'boost')->where('is_active', true)->first();
        
        $days = 7;
        $packageId = null;

        if ($package && $package->pricingTiers()->exists()) {
            $days = $package->pricingTiers()->first()->validity_days ?? 7;
            $packageId = $package->id;
        }

        $expiry = now()->addDays($days);

        // Record Purchase logic linked to Algorithm
        if ($packageId) {
             \App\Models\AdPackagePurchase::create([
                'user_id' => $user->id,
                'ad_id' => $ad->id,
                'package_id' => $packageId, // Must be valid package ID
                'amount_paid' => 0,
                'currency' => 'USD',
                'payment_method' => 'plan_credit_shortcut',
                'transaction_id' => 'SHORTCUT-' . uniqid(),
                'status' => 'active',
                'start_date' => now(),
                'expires_at' => $expiry,
                'features_applied' => json_encode(['type' => 'boost', 'days' => $days]),
            ]);
        }

        $ad->update([
            'is_boosted' => true,
            'boosted_until' => $expiry, 
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Ad bumped and boosted successfully using plan credit!',
            'data' => $ad
        ]);
    }
}

