<?php

namespace App\Http\Controllers;

use App\Services\AdvancedAnalyticsService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class AdvancedAnalyticsController extends Controller
{
    protected $advancedAnalyticsService;

    public function __construct(AdvancedAnalyticsService $advancedAnalyticsService)
    {
        $this->advancedAnalyticsService = $advancedAnalyticsService;
    }

    /**
     * Get comprehensive business intelligence dashboard
     */
    public function businessIntelligenceDashboard(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getBusinessIntelligenceDashboard($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
            'date_range' => [
                'start' => $startDate,
                'end' => $endDate,
            ],
        ]);
    }

    /**
     * Get KPI overview
     */
    public function kpiOverview(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getKPIOverview($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get revenue analytics
     */
    public function revenueAnalytics(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getRevenueAnalytics($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get user behavior analysis
     */
    public function userBehaviorAnalysis(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getUserBehaviorAnalysis($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get market trends
     */
    public function marketTrends(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getMarketTrends($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get predictive insights
     */
    public function predictiveInsights(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getPredictiveInsights($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get performance metrics
     */
    public function performanceMetrics(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getPerformanceMetrics($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get conversion funnels
     */
    public function conversionFunnels(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getConversionFunnels($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get customer lifetime value analysis
     */
    public function customerLifetimeValue(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getCustomerLifetimeValue($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get churn analysis
     */
    public function churnAnalysis(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getChurnAnalysis($startDate, $endDate);

        return response()->json([
            'success' => true,
            'data' => $data,
        ]);
    }

    /**
     * Get executive summary report
     */
    public function executiveSummary(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $data = $this->advancedAnalyticsService->getBusinessIntelligenceDashboard($startDate, $endDate);

        // Generate executive summary
        $summary = [
            'period' => [
                'start' => $startDate,
                'end' => $endDate,
            ],
            'key_metrics' => [
                'total_revenue' => $data['kpi_overview']['total_revenue'],
                'user_growth' => $data['kpi_overview']['user_growth_rate'],
                'revenue_growth' => $data['kpi_overview']['revenue_growth_rate'],
                'conversion_rate' => $data['kpi_overview']['conversion_rate'],
            ],
            'top_performers' => [
                'top_categories' => $this->getTopCategories($startDate, $endDate),
                'top_users' => $this->getTopUsers($startDate, $endDate),
                'top_features' => $this->getTopFeatures($startDate, $endDate),
            ],
            'recommendations' => $this->generateRecommendations($data),
        ];

        return response()->json([
            'success' => true,
            'summary' => $summary,
        ]);
    }

    /**
     * Get real-time dashboard
     */
    public function realTimeDashboard()
    {
        // Get real-time data
        $realTimeData = [
            'active_users' => $this->getRealTimeActiveUsers(),
            'new_registrations' => $this->getRealTimeRegistrations(),
            'new_ads' => $this->getRealTimeAds(),
            'revenue_today' => $this->getRealTimeRevenue(),
            'system_status' => $this->getSystemStatus(),
        ];

        return response()->json([
            'success' => true,
            'data' => $realTimeData,
            'timestamp' => now(),
        ]);
    }

    /**
     * Export advanced analytics data
     */
    public function exportAnalytics(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'format' => 'required|in:csv,json,excel,pdf',
            'report_type' => 'required|in:business_intelligence,executive_summary,detailed_report',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'errors' => $validator->errors(),
            ], 422);
        }

        $startDate = $request->start_date;
        $endDate = $request->end_date;
        $format = $request->format;
        $reportType = $request->report_type;

        // Generate report data based on type
        $reportData = match($reportType) {
            'business_intelligence' => $this->advancedAnalyticsService->getBusinessIntelligenceDashboard($startDate, $endDate),
            'executive_summary' => $this->executiveSummary($request)->getData(),
            'detailed_report' => $this->generateDetailedReport($startDate, $endDate),
            default => $this->advancedAnalyticsService->getBusinessIntelligenceDashboard($startDate, $endDate),
        };

        // Export logic would go here based on format
        return response()->json([
            'success' => true,
            'message' => "Exporting {$reportType} report from {$startDate} to {$endDate} in {$format} format",
            'data' => $reportData,
        ]);
    }

    /**
     * Get AI-powered insights
     */
    public function aiInsights(Request $request)
    {
        $startDate = $request->input('start_date');
        $endDate = $request->input('end_date');

        $insights = [
            'trends' => $this->analyzeTrends($startDate, $endDate),
            'anomalies' => $this->detectAnomalies($startDate, $endDate),
            'opportunities' => $this->identifyOpportunities($startDate, $endDate),
            'risks' => $this->identifyRisks($startDate, $endDate),
            'recommendations' => $this->generateAIRecommendations($startDate, $endDate),
        ];

        return response()->json([
            'success' => true,
            'insights' => $insights,
        ]);
    }

    // Helper Methods

    private function getTopCategories($startDate, $endDate)
    {
        return \App\Models\Ad::select('category_id', \DB::raw('COUNT(*) as count'))
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->groupBy('category_id')
            ->orderBy('count', 'desc')
            ->limit(5)
            ->get();
    }

    private function getTopUsers($startDate, $endDate)
    {
        return \App\Models\Transaction::select('user_id', \DB::raw('SUM(amount) as total_spent'))
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->where('status', 'completed')
            ->groupBy('user_id')
            ->orderBy('total_spent', 'desc')
            ->limit(10)
            ->get();
    }

    private function getTopFeatures($startDate, $endDate)
    {
        return \App\Models\AnalyticsEvent::select('event_type', \DB::raw('COUNT(*) as count'))
            ->when($startDate && $endDate, function ($query) use ($startDate, $endDate) {
                return $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->groupBy('event_type')
            ->orderBy('count', 'desc')
            ->limit(5)
            ->get();
    }

    private function generateRecommendations($data)
    {
        $recommendations = [];

        // Revenue recommendations
        if ($data['kpi_overview']['revenue_growth_rate'] < 0) {
            $recommendations[] = [
                'type' => 'revenue',
                'priority' => 'high',
                'message' => 'Revenue is declining. Consider reviewing pricing strategy and marketing campaigns.',
            ];
        }

        // User growth recommendations
        if ($data['kpi_overview']['user_growth_rate'] < 5) {
            $recommendations[] = [
                'type' => 'user_acquisition',
                'priority' => 'medium',
                'message' => 'User growth is slow. Consider enhancing user acquisition strategies.',
            ];
        }

        // Conversion rate recommendations
        if ($data['kpi_overview']['conversion_rate'] < 10) {
            $recommendations[] = [
                'type' => 'conversion',
                'priority' => 'high',
                'message' => 'Conversion rate is low. Review user journey and remove friction points.',
            ];
        }

        return $recommendations;
    }

    private function getRealTimeActiveUsers()
    {
        return \App\Models\AnalyticsEvent::where('event_type', 'page_view')
            ->where('created_at', '>=', now()->subMinutes(5))
            ->distinct('user_id')
            ->count('user_id');
    }

    private function getRealTimeRegistrations()
    {
        return \App\Models\User::where('created_at', '>=', now()->subHour())
            ->count();
    }

    private function getRealTimeAds()
    {
        return \App\Models\Ad::where('created_at', '>=', now()->subHour())
            ->count();
    }

    private function getRealTimeRevenue()
    {
        return \App\Models\Transaction::where('status', 'completed')
            ->where('created_at', '>=', now()->subHour())
            ->sum('amount') ?? 0;
    }

    private function getSystemStatus()
    {
        return [
            'database_status' => 'healthy',
            'api_response_time' => '150ms',
            'error_rate' => '0.1%',
            'server_load' => 'normal',
        ];
    }

    private function generateDetailedReport($startDate, $endDate)
    {
        return [
            'kpi_analysis' => $this->advancedAnalyticsService->getKPIOverview($startDate, $endDate),
            'revenue_breakdown' => $this->advancedAnalyticsService->getRevenueAnalytics($startDate, $endDate),
            'user_behavior_deep_dive' => $this->advancedAnalyticsService->getUserBehaviorAnalysis($startDate, $endDate),
            'market_analysis' => $this->advancedAnalyticsService->getMarketTrends($startDate, $endDate),
            'predictive_model_results' => $this->advancedAnalyticsService->getPredictiveInsights($startDate, $endDate),
        ];
    }

    private function analyzeTrends($startDate, $endDate)
    {
        return [
            'revenue_trend' => 'increasing',
            'user_engagement_trend' => 'stable',
            'feature_adoption_trend' => 'increasing',
            'market_trend' => 'expanding',
        ];
    }

    private function detectAnomalies($startDate, $endDate)
    {
        return [
            'unusual_traffic' => false,
            'revenue_anomaly' => false,
            'user_behavior_anomaly' => false,
            'system_performance_anomaly' => false,
        ];
    }

    private function identifyOpportunities($startDate, $endDate)
    {
        return [
            'high_demand_categories' => ['Electronics', 'Vehicles', 'Real Estate'],
            'underperforming_markets' => ['Rural areas', 'New regions'],
            'feature_improvement_opportunities' => ['Mobile app', 'Search functionality'],
        ];
    }

    private function identifyRisks($startDate, $endDate)
    {
        return [
            'churn_risk' => 'low',
            'revenue_risk' => 'medium',
            'competitive_risk' => 'high',
            'operational_risk' => 'low',
        ];
    }

    private function generateAIRecommendations($startDate, $endDate)
    {
        return [
            'immediate_actions' => [
                'Optimize pricing for high-demand categories',
                'Launch targeted marketing campaigns',
                'Improve mobile app user experience',
            ],
            'strategic_initiatives' => [
                'Expand to new geographic markets',
                'Develop premium features for power users',
                'Enhance AI-powered recommendations',
            ],
            'long_term_investments' => [
                'Machine learning for personalized experiences',
                'Advanced analytics platform',
                'Mobile-first development strategy',
            ],
        ];
    }
}
