<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\SafetyReport;
use App\Models\UserTrustScore;
use App\Models\UserAppeal;
use App\Models\UserPlan;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class UserManagementController extends Controller
{
    /**
     * Display a listing of the users.
     */
    public function index(Request $request)
    {
        $query = User::with(['currentPlan', 'trustScore'])->withCount('ads');

        // Search
        if ($request->has('search') && $request->search) {
            $search = $request->search;
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('mobile_number', 'like', "%{$search}%")
                  ->orWhere('phone_number', 'like', "%{$search}%");
            });
        }

        // Filter by Status
        if ($request->has('status') && $request->status) {
            $status = $request->status;
            if ($status === 'active') {
                $query->where('is_banned', false)->where('is_suspended', false);
            } elseif ($status === 'suspended') {
                $query->where('is_suspended', true);
            } elseif ($status === 'banned') {
                $query->where('is_banned', true);
            } elseif ($status === 'pending') {
                $query->where('is_verified', false);
            }
        }

        // Filter by Plan
        if ($request->has('plan') && $request->plan) {
            $plan = $request->plan;
            if ($plan === 'free') {
                 $query->whereDoesntHave('currentPlan');
            } elseif ($plan === 'premium') {
                 $query->whereHas('currentPlan', function($q) {
                     $q->where('price', '>', 0);
                 });
            } else {
                 $query->whereHas('currentPlan', function($q) use($plan) {
                     $q->where('plan_name', 'like', "%{$plan}%");
                 });
            }
        }

        $users = $query->latest()->paginate(10);
        
        // Stats
        $stats = [
            'total' => User::count(),
            'active' => User::where('is_banned', false)->where('is_suspended', false)->count(),
            'premium' => User::whereHas('currentPlan', function($q) { $q->where('price', '>', 0); })->count(),
            'pending' => User::where('is_verified', false)->count()
        ];

        $allPlans = UserPlan::active()->get();

        return view('admin.users.index', compact('users', 'stats', 'allPlans'));
    }

    /**
     * Get detailed user information for admin view.
     */
    public function getUserDetails($userId)
    {
        $user = User::with(['ads', 'trustScore', 'appeals'])
                   ->findOrFail($userId);

        // Get user statistics
        $userStats = [
            'total_ads' => $user->ads->count(),
            'active_ads' => $user->ads->where('status', 'active')->count(),
            'pending_ads' => $user->ads->where('status', 'pending')->count(),
            'sold_ads' => $user->ads->where('status', 'sold')->count(),
            'expired_ads' => $user->ads->where('status', 'expired')->count(),
            'rejected_ads' => $user->ads->where('status', 'rejected')->count(),
            'rejected_ads' => $user->ads->where('status', 'rejected')->count(),
            'total_views' => $user->adAnalytics()->sum('views'),
            'account_age_days' => now()->diffInDays($user->created_at),
            'last_login' => $user->last_login,
            'is_verified' => $user->is_verified,
            'trust_score' => $user->trustScore ? $user->trustScore->score : 0,
            'safety_reports' => SafetyReport::where('reported_user_id', $userId)->count(),
            'appeals_pending' => UserAppeal::where('user_id', $userId)
                                       ->where('status', 'pending')
                                       ->count()
        ];

        // Get recent safety reports
        $recentReports = SafetyReport::where('reported_user_id', $userId)
                                   ->with('reporter')
                                   ->latest()
                                   ->take(5)
                                   ->get();

        // Get user's other ads
        $userAds = $user->ads()
                       ->select('id', 'title', 'status', 'price', 'currency', 'created_at')
                       ->orderBy('created_at', 'desc')
                       ->take(10)
                       ->get();

        return response()->json([
            'user' => $user,
            'stats' => $userStats,
            'recent_reports' => $recentReports,
            'user_ads' => $userAds
        ]);
    }

    /**
     * Ban a user.
     */
    public function banUser(Request $request, $userId)
    {
        $user = User::findOrFail($userId);
        
        $request->validate([
            'reason' => 'required|string|max:500',
            'duration_days' => 'nullable|integer|min:1|max:365'
        ]);

        $user->is_banned = true;
        $user->ban_reason = $request->reason;
        $user->banned_until = $request->duration_days 
            ? now()->addDays($request->duration_days)
            : null;
        $user->save();

        // Log the action
        $this->logUserAction($userId, 'ban', $request->reason, auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'User has been banned successfully'
        ]);
    }

    /**
     * Suspend a user.
     */
    public function suspendUser(Request $request, $userId)
    {
        $user = User::findOrFail($userId);
        
        $request->validate([
            'reason' => 'required|string|max:500',
            'duration_days' => 'required|integer|min:1|max:365'
        ]);

        $user->is_suspended = true;
        $user->suspension_reason = $request->reason;
        $user->suspended_until = now()->addDays($request->duration_days);
        $user->save();

        // Log the action
        $this->logUserAction($userId, 'suspend', $request->reason, auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'User has been suspended successfully'
        ]);
    }

    /**
     * Verify a user.
     */
    public function verifyUser($userId)
    {
        $user = User::findOrFail($userId);
        
        $user->is_verified = true;
        $user->verified_at = now();
        $user->save();

        // Log the action
        $this->logUserAction($userId, 'verify', 'User verification', auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'User has been verified successfully'
        ]);
    }

    /**
     * Unban a user.
     */
    public function unbanUser($userId)
    {
        $user = User::findOrFail($userId);
        
        $user->is_banned = false;
        $user->ban_reason = null;
        $user->banned_until = null;
        $user->save();

        // Log the action
        $this->logUserAction($userId, 'unban', 'User unbanned', auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'User has been unbanned successfully'
        ]);
    }

    /**
     * Unsuspend a user.
     */
    public function unsuspendUser($userId)
    {
        $user = User::findOrFail($userId);
        
        $user->is_suspended = false;
        $user->suspension_reason = null;
        $user->suspended_until = null;
        $user->save();

        // Log the action
        $this->logUserAction($userId, 'unsuspend', 'User unsuspended', auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'User has been unsuspended successfully'
        ]);
    }

    /**
     * Get safety reports for an ad.
     */
    public function getAdSafetyReports($adId)
    {
        $reports = SafetyReport::where('ad_id', $adId)
                             ->with(['reporter', 'ad'])
                             ->latest()
                             ->get();

        return response()->json([
            'reports' => $reports,
            'total_reports' => $reports->count()
        ]);
    }

    /**
     * Handle user appeal.
     */
    public function handleAppeal(Request $request, $appealId)
    {
        $appeal = UserAppeal::findOrFail($appealId);
        
        $request->validate([
            'decision' => 'required|in:approved,rejected',
            'reason' => 'required|string|max:500'
        ]);

        $appeal->status = $request->decision === 'approved' ? 'approved' : 'rejected';
        $appeal->admin_reason = $request->reason;
        $appeal->admin_id = auth()->id();
        $appeal->resolved_at = now();
        $appeal->save();

        // If appeal is approved, take appropriate action
        if ($appeal->decision === 'approved') {
            $this->handleApprovedAppeal($appeal);
        }

        return response()->json([
            'success' => true,
            'message' => 'Appeal has been processed successfully'
        ]);
    }

    /**
     * Log user management actions.
     */
    private function logUserAction($userId, $action, $reason, $adminId)
    {
        DB::table('user_management_logs')->insert([
            'user_id' => $userId,
            'admin_id' => $adminId,
            'action' => $action,
            'reason' => $reason,
            'created_at' => now(),
            'updated_at' => now()
        ]);
    }

    /**
     * Update user details.
     */
    public function update(Request $request, $userId)
    {
        $user = User::findOrFail($userId);
        
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,'.$userId,
            'phone_number' => 'nullable|string|max:20',
            'is_verified' => 'boolean',
            // Add other editable fields as needed
        ]);

        $user->name = $request->name;
        $user->email = $request->email;
        $user->phone_number = $request->phone_number;
        
        if ($request->has('is_verified')) {
            $user->is_verified = $request->is_verified;
            if ($user->is_verified && !$user->verified_at) {
                $user->verified_at = now();
            }
        }

        $user->save();

        $this->logUserAction($userId, 'update', 'Admin updated user details', auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'User updated successfully'
        ]);
    }

    /**
     * Handle approved appeal.
     */
    private function handleApprovedAppeal($appeal)
    {
        // This would contain logic to reverse the original action
        // For example, if it was a ban appeal, unban the user
        // If it was a suspension appeal, unsuspend the user
        // Implementation depends on the specific appeal type
    }

    /**
     * Change user plan.
     */
    public function changeUserPlan(Request $request, $userId)
    {
        $user = User::findOrFail($userId);
        
        $request->validate([
            'plan_id' => 'required|exists:user_plans,id',
            'action' => 'required|in:upgrade,downgrade'
        ]);

        $plan = UserPlan::findOrFail($request->plan_id);

        $user->current_plan_id = $plan->id;
        $user->save();

        // Log the action
        $this->logUserAction($userId, $request->action, "Plan changed to {$plan->plan_name}", auth()->id());

        return response()->json([
            'success' => true,
            'message' => "User plan successfully changed to {$plan->plan_name}"
        ]);
    }

    /**
     * Send notification to user.
     */
    public function sendUserNotification(Request $request, $userId)
    {
        $user = User::findOrFail($userId);
        
        $request->validate([
            'type' => 'required|string',
            'message' => 'required|string|max:500'
        ]);

        Notification::create([
            'user_id' => $user->id,
            'type' => $request->type, // e.g., 'info', 'warning'
            'title' => 'Admin Notification',
            'message' => $request->message,
            'status' => 'sent',
            'sent_at' => now(),
            'target_type' => 'single_user',
            'target_value' => ['user_id' => $user->id]
        ]);

        $this->logUserAction($userId, 'notification', "Sent notification: {$request->type}", auth()->id());

        return response()->json([
            'success' => true,
            'message' => 'Notification sent successfully'
        ]);
    }
}
