<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\SupportSetting;
use App\Models\SupportTicket;
use App\Models\SupportTicketReply;
use App\Models\SupportChat;
use App\Models\SupportChatMessage;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class SupportController extends Controller
{
    // Settings Management
    public function settings()
    {
        $settings = SupportSetting::current();
        return view('admin.support.settings', compact('settings'));
    }

    public function updateSettings(Request $request)
    {
        $validated = $request->validate([
            'support_email' => 'required|email',
            'support_phone' => 'required|string',
            'ticket_enabled' => 'boolean',
            'chat_enabled' => 'boolean',
            'email_enabled' => 'boolean',
            'call_enabled' => 'boolean',
            'auto_reply_message' => 'nullable|string',
            'ticket_instructions' => 'nullable|string'
        ]);

        $settings = SupportSetting::current();
        $settings->update($validated);

        return redirect()->back()->with('success', 'Support settings updated successfully!');
    }

    // Tickets Management
    public function tickets(Request $request)
    {
        $query = SupportTicket::with(['user', 'assignedAdmin'])->latest();

        // Filters
        if ($request->status) {
            $query->where('status', $request->status);
        }
        if ($request->category) {
            $query->where('category', $request->category);
        }
        if ($request->search) {
            $query->where(function($q) use ($request) {
                $q->where('ticket_number', 'like', '%' . $request->search . '%')
                  ->orWhere('subject', 'like', '%' . $request->search . '%')
                  ->orWhereHas('user', function($uq) use ($request) {
                      $uq->where('name', 'like', '%' . $request->search . '%');
                  });
            });
        }

        $tickets = $query->paginate(20);
        $categories = $this->getCategories();
        $stats = $this->getTicketStats();

        return view('admin.support.tickets.index', compact('tickets', 'categories', 'stats'));
    }

    public function showTicket($id)
    {
        $ticket = SupportTicket::with(['user', 'assignedAdmin', 'replies.user'])->findOrFail($id);
        $admins = User::where('is_admin', true)->get();
        
        return view('admin.support.tickets.show', compact('ticket', 'admins'));
    }

    public function updateTicket(Request $request, $id)
    {
        $ticket = SupportTicket::findOrFail($id);
        
        $validated = $request->validate([
            'status' => 'required|in:open,in_progress,resolved,closed',
            'priority' => 'required|in:low,medium,high,urgent',
            'assigned_to' => 'nullable|exists:users,id'
        ]);

        $ticket->update($validated);

        if ($validated['status'] === 'resolved' && !$ticket->resolved_at) {
            $ticket->update(['resolved_at' => now()]);
        }
        if ($validated['status'] === 'closed' && !$ticket->closed_at) {
            $ticket->update(['closed_at' => now()]);
        }

        return redirect()->back()->with('success', 'Ticket updated successfully!');
    }

    public function replyTicket(Request $request, $id)
    {
        $ticket = SupportTicket::findOrFail($id);
        
        $validated = $request->validate([
            'message' => 'required|string',
            'attachments.*' => 'nullable|file|max:5120|mimes:jpg,jpeg,png,pdf'
        ]);

        $attachments = [];
        if ($request->hasFile('attachments')) {
            foreach ($request->file('attachments') as $file) {
                $path = $file->store('support_attachments', 'public');
                $attachments[] = $path;
            }
        }

        SupportTicketReply::create([
            'ticket_id' => $ticket->id,
            'user_id' => auth()->id(),
            'is_admin' => true,
            'message' => $validated['message'],
            'attachments' => $attachments
        ]);

        // Update ticket status if it was closed
        if ($ticket->status === 'closed') {
            $ticket->update(['status' => 'in_progress']);
        }

        return redirect()->back()->with('success', 'Reply sent successfully!');
    }

    // Chat Management
    public function chats(Request $request)
    {
        $query = SupportChat::with(['user', 'latestMessage'])->latest('updated_at');

        // Filters
        if ($request->status) {
            $query->where('status', $request->status);
        }
        if ($request->search) {
            $query->whereHas('user', function($q) use ($request) {
                $q->where('name', 'like', '%' . $request->search . '%')
                  ->orWhere('email', 'like', '%' . $request->search . '%');
            });
        }

        $chats = $query->paginate(20);
        $stats = $this->getChatStats();

        return view('admin.support.chats.index', compact('chats', 'stats'));
    }

    public function showChat($id)
    {
        $chat = SupportChat::with(['user', 'messages.user'])->findOrFail($id);
        
        // Mark chat as seen by admin
        $chat->update(['last_seen_by_admin' => now()]);

        // Fetch user's latest ads if user exists
        $userAds = [];
        if ($chat->user) {
            $userAds = \App\Models\Ad::where('user_id', $chat->user_id)
                        ->with('images')
                        ->latest()
                        ->take(5)
                        ->get();
        }
        
        return view('admin.support.chats.show', compact('chat', 'userAds'));
    }

    public function sendMessage(Request $request, $id)
    {
        $chat = SupportChat::findOrFail($id);
        
        $validated = $request->validate([
            'message' => 'required|string'
        ]);

        SupportChatMessage::create([
            'chat_id' => $chat->id,
            'user_id' => auth()->id(),
            'is_admin' => true,
            'message' => $validated['message']
        ]);

        // Update chat status and timestamp
        $chat->update([
            'status' => 'active',
            'updated_at' => now()
        ]);

        return redirect()->back()->with('success', 'Message sent successfully!');
    }

    public function closeChat($id)
    {
        $chat = SupportChat::findOrFail($id);
        $chat->update([
            'status' => 'closed',
            'closed_at' => now()
        ]);

        return redirect()->back()->with('success', 'Chat closed successfully!');
    }

    // Helper Methods
    private function getCategories()
    {
        return [
            'Account Issues',
            'Payment Problems',
            'Ad Posting Issues',
            'Technical Problems',
            'Report Abuse',
            'Feature Request',
            'Other'
        ];
    }

    private function getTicketStats()
    {
        return [
            'total' => SupportTicket::count(),
            'open' => SupportTicket::where('status', 'open')->count(),
            'in_progress' => SupportTicket::where('status', 'in_progress')->count(),
            'resolved' => SupportTicket::where('status', 'resolved')->count(),
            'closed' => SupportTicket::where('status', 'closed')->count()
        ];
    }

    private function getChatStats()
    {
        return [
            'total' => SupportChat::count(),
            'active' => SupportChat::where('status', 'active')->count(),
            'pending' => SupportChat::where('status', 'pending')->count(),
            'closed' => SupportChat::where('status', 'closed')->count()
        ];
    }
}
