<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\UserPlan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Database\QueryException;

class PlansController extends Controller
{
    /**
     * Display the admin plans management dashboard.
     */
    public function index()
    {
        $plans = UserPlan::orderBy('price', 'desc')->get();
        $stats = [
            'total_plans' => UserPlan::count(),
            'active_plans' => UserPlan::where('status', 'Active')->count(),
            'total_revenue' => $this->calculateTotalRevenue(),
            'premium_users' => $this->calculatePremiumUsers(),
        ];

        return view('admin.plans.index', compact('plans', 'stats'));
    }

    /**
     * Show the form for creating a new plan.
     */
    public function create()
    {
        return view('admin.plans.create');
    }

    /**
     * Store a newly created plan in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'plan_name' => 'required|string|max:100',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'post_limit' => 'nullable|integer|min:0',
            'lead_limit' => 'nullable|integer|min:0',
            'contact_limit' => 'nullable|integer|min:0',
            'analytics_tier' => 'required|string',
            'validity_days' => 'required|integer|min:1',
            'ad_free' => 'boolean',
            'has_analytics' => 'boolean',
            'verified_badge' => 'boolean',
            'vip_badge' => 'boolean',
            'badge_style' => 'nullable|string',
            'theme' => 'nullable|string',
            'halo_effect' => 'nullable|string',
            'ad_border' => 'nullable|string',
            'status' => 'required|string|in:Active,Inactive',
        ]);

        try {
            $plan = UserPlan::create($validated);
            Log::info('New plan created: ' . $plan->plan_name, ['plan_id' => $plan->id]);
            
            return redirect()->route('admin.admin.plans.index')
                ->with('success', 'Plan created successfully!');
        } catch (\Exception $e) {
            Log::error('Error creating plan: ' . $e->getMessage());
            return back()->with('error', 'Failed to create plan. Please try again.');
        }
    }

    /**
     * Toggle default plan (for new user registration).
     */
    public function toggleDefault($id)
    {
        try {
            $plan = UserPlan::findOrFail($id);
            
            // Ensure plan is active before making it default
            if ($plan->status !== 'Active' && !$plan->is_default) {
                return back()->with('error', 'Only active plans can be set as default.');
            }

            // Un-set all other plans as default
            UserPlan::where('id', '!=', $id)->update(['is_default' => false]);

            // Toggle this plan's default status
            $plan->is_default = !$plan->is_default;
            $plan->save();

            $message = $plan->is_default 
                ? "{$plan->plan_name} is now the default plan for new users." 
                : "{$plan->plan_name} is no longer the default plan.";

            Log::info("Default plan toggled for: {$plan->plan_name}", [
                'plan_id' => $plan->id,
                'is_default' => $plan->is_default,
            ]);

            return back()->with('success', $message);
        } catch (\Exception $e) {
            Log::error('Error toggling default plan: ' . $e->getMessage());
            return back()->with('error', 'Failed to set default plan.');
        }
    }

    /**
     * API endpoint to store a newly created plan in storage.
     */
    public function apiStore(Request $request)
    {
        $validated = $request->validate([
            'plan_name' => 'required|string|max:100',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'post_limit' => 'nullable|integer|min:0',
            'lead_limit' => 'nullable|integer|min:0',
            'contact_limit' => 'nullable|integer|min:0',
            'analytics_tier' => 'required|string',
            'validity_days' => 'required|integer|min:1',
            'ad_free' => 'boolean',
            'has_analytics' => 'boolean',
            'verified_badge' => 'boolean',
            'vip_badge' => 'boolean',
            'badge_style' => 'nullable|string',
            'theme' => 'nullable|string',
            'halo_effect' => 'nullable|string',
            'ad_border' => 'nullable|string',
            'status' => 'required|string|in:Active,Inactive',
        ]);

        try {
            $plan = UserPlan::create($validated);
            Log::info('New plan created via API: ' . $plan->plan_name, ['plan_id' => $plan->id]);
            
            return response()->json([
                'success' => true,
                'message' => 'Plan created successfully!',
                'plan' => $plan
            ]);
        } catch (\Exception $e) {
            Log::error('Error creating plan via API: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to create plan. Please try again.'
            ], 500);
        }
    }

    /**
     * API endpoint to update a plan in storage.
     */
    public function apiUpdate(Request $request, UserPlan $plan)
    {
        $validated = $request->validate([
            'plan_name' => 'required|string|max:100',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'post_limit' => 'nullable|integer|min:0',
            'lead_limit' => 'nullable|integer|min:0',
            'contact_limit' => 'nullable|integer|min:0',
            'analytics_tier' => 'required|string',
            'validity_days' => 'required|integer|min:1',
            'ad_free' => 'boolean',
            'has_analytics' => 'boolean',
            'verified_badge' => 'boolean',
            'vip_badge' => 'boolean',
            'badge_style' => 'nullable|string',
            'theme' => 'nullable|string',
            'halo_effect' => 'nullable|string',
            'ad_border' => 'nullable|string',
            'status' => 'required|string|in:Active,Inactive',
            // Discount Fields
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
            'discount_fixed_amount' => 'nullable|numeric|min:0',
            'discount_start_date' => 'nullable|date',
            'discount_end_date' => 'nullable|date|after_or_equal:discount_start_date',
        ]);

        try {
            $plan->update($validated);
            Log::info('Plan updated via API: ' . $plan->plan_name, ['plan_id' => $plan->id]);
            
            return response()->json([
                'success' => true,
                'message' => 'Plan updated successfully!',
                'plan' => $plan
            ]);
        } catch (\Exception $e) {
            Log::error('Error updating plan via API: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update plan. Please try again.'
            ], 500);
        }
    }

    /**
     * Show the form for editing the specified plan.
     */
    public function edit(UserPlan $plan)
    {
        return view('admin.plans.edit', compact('plan'));
    }

    /**
     * Update the specified plan in storage.
     */
    public function update(Request $request, UserPlan $plan)
    {
        $validated = $request->validate([
            'plan_name' => 'required|string|max:100',
            'price' => 'required|numeric|min:0',
            'duration_days' => 'required|integer|min:1',
            'post_limit' => 'nullable|integer|min:0',
            'lead_limit' => 'nullable|integer|min:0',
            'analytics_tier' => 'required|string',
            'validity_days' => 'required|integer|min:1',
            'ad_free' => 'boolean',
            'has_analytics' => 'boolean',
            'verified_badge' => 'boolean',
            'vip_badge' => 'boolean',
            'badge_style' => 'nullable|string',
            'theme' => 'nullable|string',
            'halo_effect' => 'nullable|string',
            'ad_border' => 'nullable|string',
            'status' => 'required|string|in:Active,Inactive',
            // Discount Fields
            'discount_percentage' => 'nullable|numeric|min:0|max:100',
            'discount_fixed_amount' => 'nullable|numeric|min:0',
            'discount_start_date' => 'nullable|date',
            'discount_end_date' => 'nullable|date|after_or_equal:discount_start_date',
        ]);

        try {
            $plan->update($validated);
            Log::info('Plan updated: ' . $plan->plan_name, ['plan_id' => $plan->id]);
            
            return redirect()->route('admin.plans.index')
                ->with('success', 'Plan updated successfully!');
        } catch (\Exception $e) {
            Log::error('Error updating plan: ' . $e->getMessage());
            return back()->with('error', 'Failed to update plan. Please try again.');
        }
    }

    /**
     * Set a plan as the default plan for new users
     */
    public function setDefault(UserPlan $plan)
    {
        try {
            Log::info('Setting default plan', [
                'plan_id' => $plan->id,
                'plan_name' => $plan->plan_name
            ]);

            // First, remove default from all plans
            UserPlan::query()->update(['is_default' => false]);
            
            // Set this plan as default
            $plan->is_default = true;
            $plan->save();

            Log::info('Default plan updated successfully', [
                'plan_id' => $plan->id,
                'plan_name' => $plan->plan_name
            ]);

            return redirect()->route('admin.plans.index')
                ->with('success', "'{$plan->plan_name}' is now the default plan for new users.");
        } catch (\Exception $e) {
            Log::error('Failed to set default plan', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'plan_id' => $plan->id ?? 'unknown'
            ]);
            return redirect()->back()->with('error', 'Failed to set default plan: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified plan from storage.
     */
    public function destroy($id) 
    {
        try {
            $plan = UserPlan::find($id);
            
            if (!$plan) {
                 Log::warning("Delete requested for non-existent plan ID: " . $id);
                 if (request()->wantsJson()) {
                    return response()->json([
                        'success' => false,
                        'message' => 'Plan not found.'
                    ], 404);
                }
                return back()->with('error', 'Plan not found.');
            }

            $planName = $plan->plan_name;
            $plan->delete();
            
            Log::info('Plan deleted: ' . $planName, ['plan_id' => $id]);
            
            if (request()->wantsJson()) {
                return response()->json([
                    'success' => true,
                    'message' => 'Plan deleted successfully!'
                ]);
            }
            
            return redirect()->route('admin.plans.index')
                ->with('success', 'Plan deleted successfully!');
        } catch (\Exception $e) {
            Log::error('Error deleting plan: ' . $e->getMessage());
            
            if (request()->wantsJson()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to delete plan: ' . $e->getMessage()
                ], 500);
            }
            
            return back()->with('error', 'Failed to delete plan. Please try again.');
        }
    }

    /**
     * Get analytics for a specific plan.
     */
    public function analytics($id)
    {
        try {
            $plan = UserPlan::find($id);
            
            if (!$plan) {
                Log::warning("Analytics requested for non-existent plan ID: " . $id);
                return response()->json([
                    'success' => false,
                    'message' => 'Plan not found'
                ], 404);
            }

            Log::info("Fetching analytics for plan: " . $plan->plan_name, ['plan_id' => $plan->id]);

            $analytics = [
                'plan_name' => $plan->plan_name,
                'total_users' => $plan->users()->count(),
                'monthly_revenue' => number_format($this->calculateTotalRevenueForPlan($plan), 2),
                'conversion_rate' => $this->calculateConversionRate($plan),
                'churn_rate' => $this->calculateRealChurnRate($plan),
                'retention_rate' => $this->calculateRetentionRate($plan),
                'growth_rate' => $this->calculateGrowthRate($plan),
            ];

            return response()->json([
                'success' => true,
                'analytics' => $analytics
            ]);
        } catch (\Exception $e) {
            Log::error('Analytics Error for plan ID ' . $id . ': ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Error loading analytics'
            ], 500);
        }
    }

    /**
     * Toggle plan status (Active/Inactive).
     */
    /**
     * Toggle plan status (Active/Inactive).
     */
    public function toggleStatus($id)
    {
        try {
            Log::info("Attempting to toggle status for Plan ID: " . $id);
            
            $plan = UserPlan::find($id);
            
            if (!$plan) {
                Log::error("Plan not found for ID: " . $id);
                return response()->json([
                    'success' => false,
                    'message' => 'Plan not found.'
                ], 404);
            }

            $plan->status = $plan->status === 'Active' ? 'Inactive' : 'Active';
            $plan->save();
            
            Log::info('Plan status changed: ' . $plan->plan_name, [
                'plan_id' => $plan->id,
                'status' => $plan->status
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Plan status updated successfully!',
                'status' => $plan->status
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling plan status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update plan status: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get all plans for API.
     */
    public function apiIndex()
    {
        $plans = UserPlan::where('status', 'Active')->get();
        return response()->json(['success' => true, 'plans' => $plans]);
    }

    /**
     * Get a specific plan for API.
     */
    public function apiShow(UserPlan $plan)
    {
        return response()->json(['success' => true, 'plan' => $plan]);
    }

    /**
     * API endpoint for plan analytics.
     */
    public function apiAnalytics(UserPlan $plan)
    {
        $analytics = [
            'plan_name' => $plan->plan_name,
            'total_users' => $plan->users()->count(),
            'monthly_revenue' => $this->calculatePlanRevenue($plan),
            'conversion_rate' => $this->calculateConversionRate($plan),
            'churn_rate' => $this->calculateChurnRate($plan),
            'user_retention' => $this->calculateUserRetention($plan),
        ];

        return response()->json(['success' => true, 'analytics' => $analytics]);
    }

    /**
     * API endpoint to toggle plan status.
     */
    public function apiToggleStatus(UserPlan $plan)
    {
        // Check if user is authenticated and is an admin
        if (!auth()->check()) {
            return response()->json([
                'success' => false,
                'message' => 'Authentication required.'
            ], 401);
        }

        // For now, allow any authenticated user (in production, check is_admin field)
        // if (!auth()->user()->is_admin) {
        //     return response()->json([
        //         'success' => false,
        //         'message' => 'Admin access required.'
        //     ], 403);
        // }

        try {
            $plan->status = $plan->status === 'Active' ? 'Inactive' : 'Active';
            $plan->save();
            
            Log::info('Plan status changed via API: ' . $plan->plan_name, [
                'plan_id' => $plan->id,
                'status' => $plan->status,
                'user_id' => auth()->id()
            ]);
            
            return response()->json([
                'success' => true,
                'message' => 'Plan status updated successfully!',
                'status' => $plan->status
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling plan status via API: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update plan status.'
            ], 500);
        }
    }

    /**
     * API endpoint for bulk actions.
     */
    public function apiBulkAction(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|string|in:activate,deactivate,delete',
            'plan_ids' => 'required|array',
            'plan_ids.*' => 'integer|exists:user_plans,id'
        ]);

        $action = $validated['action'];
        $planIds = $validated['plan_ids'];
        $count = 0;

        try {
            foreach ($planIds as $planId) {
                $plan = UserPlan::find($planId);
                if ($plan) {
                    switch ($action) {
                        case 'activate':
                            $plan->status = 'Active';
                            break;
                        case 'deactivate':
                            $plan->status = 'Inactive';
                            break;
                        case 'delete':
                            $plan->delete();
                            continue 2; // Skip the save for deleted plans
                    }
                    $plan->save();
                    $count++;
                }
            }

            return response()->json([
                'success' => true,
                'message' => "$count plans $action" . ($count > 1 ? 'ed' : 'd') . ' successfully!'
            ]);

        } catch (\Exception $e) {
            Log::error('Error in bulk action: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to perform bulk action.'
            ], 500);
        }
    }

    /**
     * Calculate total revenue for a specific plan.
     */
    private function calculateTotalRevenueForPlan(UserPlan $plan)
    {
        try {
            // Try to sum active subscriptions first
            $subRevenue = \App\Models\Subscription::where('plan_id', $plan->id)
                ->where('status', 'active')
                ->sum('amount');
                
            if ($subRevenue > 0) {
                return $subRevenue;
            }
        } catch (QueryException $e) {
            // Table might not exist or query failed, log and proceed to fallback
            Log::warning('QueryException in calculateTotalRevenueForPlan: ' . $e->getMessage());
        }

        // Fallback to simple calculation
        return $plan->price * $plan->users()->count();
    }

    /**
     * Calculate real churn rate based on subscriptions.
     */
    private function calculateRealChurnRate(UserPlan $plan)
    {
        try {
            $totalSubscriptions = \App\Models\Subscription::where('plan_id', $plan->id)->count();
            
            if ($totalSubscriptions === 0) {
                // Fallback: If no subscription records, check users who moved AWAY from this plan?
                // Without history, it's hard. We'll return 0% if no data.
                return '0%'; 
            }

            $cancelled = \App\Models\Subscription::where('plan_id', $plan->id)
                ->where('status', 'cancelled')
                ->count();

            return round(($cancelled / $totalSubscriptions) * 100, 1) . '%';
        } catch (QueryException $e) {
            Log::error('QueryException in calculateRealChurnRate: ' . $e->getMessage());
            return '0%'; // Fallback value
        }
    }

    /**
     * Calculate retention rate.
     */
    private function calculateRetentionRate(UserPlan $plan)
    {
        try {
            $totalSubscriptions = \App\Models\Subscription::where('plan_id', $plan->id)->count();
            
            if ($totalSubscriptions === 0) {
                 return '100%'; 
            }
            
            $active = \App\Models\Subscription::where('plan_id', $plan->id)
                ->where('status', 'active')
                ->count();
                
            return round(($active / $totalSubscriptions) * 100, 1) . '%';
        } catch (QueryException $e) {
            Log::error('QueryException in calculateRetentionRate: ' . $e->getMessage());
            return '100%'; // Fallback value
        }
    }

    /**
     * Calculate Month-over-Month growth rate.
     */
    private function calculateGrowthRate(UserPlan $plan)
    {
        try {
            $currentMonth = now()->startOfMonth();
            $lastMonth = now()->subMonth()->startOfMonth();
            
            $newSubsThisMonth = \App\Models\Subscription::where('plan_id', $plan->id)
                ->where('created_at', '>=', $currentMonth)
                ->count();
                
            $newSubsLastMonth = \App\Models\Subscription::where('plan_id', $plan->id)
                ->whereBetween('created_at', [$lastMonth, $currentMonth])
                ->count();
                
            if ($newSubsLastMonth == 0) {
                return $newSubsThisMonth > 0 ? '+100%' : '0%';
            }
            
            $growth = (($newSubsThisMonth - $newSubsLastMonth) / $newSubsLastMonth) * 100;
            $sign = $growth >= 0 ? '+' : '';
            
            return $sign . round($growth, 1) . '%';
        } catch (QueryException $e) {
            Log::error('QueryException in calculateGrowthRate: ' . $e->getMessage());
            return '0%'; // Fallback value
        }
    }

    /**
     * Calculate total revenue from all plans (Monthly Recurring Revenue Estimate).
     */
    private function calculateTotalRevenue()
    {
        $revenue = 0;
        $plans = UserPlan::where('status', 'Active')->withCount('users')->get();
        foreach ($plans as $plan) {
            $revenue += $this->calculateTotalRevenueForPlan($plan);
        }
        return $revenue;
    }

    /**
     * Calculate total number of premium users.
     */
    private function calculatePremiumUsers()
    {
        // Count users on any paid plan (price > 0)
        return \App\Models\User::whereHas('currentPlan', function($query) {
            $query->where('price', '>', 0)->where('status', 'Active');
        })->count();
    }

    /**
     * Calculate revenue for a specific plan.
     */
    private function calculatePlanRevenue(UserPlan $plan)
    {
        return $this->calculateTotalRevenueForPlan($plan);
    }

    /**
     * Calculate conversion rate for a plan.
     */
    private function calculateConversionRate(UserPlan $plan)
    {
        $totalUsers = \App\Models\User::count();
        $planUsers = $plan->users()->count();
        return $totalUsers > 0 ? round(($planUsers / $totalUsers) * 100, 2) : 0;
    }

    /**
     * Export plans to CSV.
     */
    public function export()
    {
        $fileName = 'user_plans_' . date('Y-m-d_H-i-s') . '.csv';
        $plans = UserPlan::all();

        $headers = array(
            "Content-type"        => "text/csv",
            "Content-Disposition" => "attachment; filename=$fileName",
            "Pragma"              => "no-cache",
            "Cache-Control"       => "must-revalidate, post-check=0, pre-check=0",
            "Expires"             => "0"
        );

        $columns = array('ID', 'Plan Name', 'Price', 'Duration (Days)', 'Users Count', 'Status', 'Created At');

        $callback = function() use($plans, $columns) {
            $file = fopen('php://output', 'w');
            fputcsv($file, $columns);

            foreach ($plans as $plan) {
                $row['ID']  = $plan->id;
                $row['Plan Name']    = $plan->plan_name;
                $row['Price']    = $plan->price;
                $row['Duration (Days)']  = $plan->duration_days;
                $row['Users Count']  = $plan->users()->count();
                $row['Status']  = $plan->status;
                $row['Created At']  = $plan->created_at;

                fputcsv($file, array($row['ID'], $row['Plan Name'], $row['Price'], $row['Duration (Days)'], $row['Users Count'], $row['Status'], $row['Created At']));
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }
}
