<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\PaymentGateway;
use App\Models\PaymentMethod;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class PaymentGatewayController extends Controller
{
    /**
     * Display a listing of payment gateways.
     */
    public function index()
    {
        $gateways = PaymentGateway::with('paymentMethods')->orderBy('sort_order')->get();
        return view('admin.payment_gateways.index', compact('gateways'));
    }

    /**
     * Show the form for creating a new payment gateway.
     */
    public function create()
    {
        return view('admin.payment_gateways.create');
    }

    /**
     * Store a newly created payment gateway in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'slug' => 'required|string|max:50|unique:payment_gateways,slug',
            'api_key' => 'nullable|string',
            'api_secret' => 'nullable|string',
            'webhook_secret' => 'nullable|string',
            'is_active' => 'boolean',
            'is_test_mode' => 'boolean',
            'supported_currencies' => 'nullable|array',
            'supported_methods' => 'nullable|array',
            'transaction_fee_percent' => 'nullable|numeric|min:0|max:100',
            'transaction_fee_fixed' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'sort_order' => 'integer|min:0',
        ]);

        try {
            $gateway = PaymentGateway::create($validated);
            
            Log::info('Payment gateway created', [
                'gateway_id' => $gateway->id,
                'gateway_name' => $gateway->name,
                'admin_id' => auth()->id(),
            ]);

            return redirect()->route('admin.payment-gateways.index')
                ->with('success', 'Payment gateway created successfully!');
        } catch (\Exception $e) {
            Log::error('Error creating payment gateway: ' . $e->getMessage());
            return back()->with('error', 'Failed to create payment gateway. Please try again.');
        }
    }

    /**
     * Show the form for editing the specified payment gateway.
     */
    public function edit(PaymentGateway $paymentGateway)
    {
        return view('admin.payment_gateways.edit', compact('paymentGateway'));
    }

    /**
     * Update the specified payment gateway in storage.
     */
    public function update(Request $request, PaymentGateway $paymentGateway)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'slug' => 'required|string|max:50|unique:payment_gateways,slug,' . $paymentGateway->id,
            'api_key' => 'nullable|string',
            'api_secret' => 'nullable|string',
            'webhook_secret' => 'nullable|string',
            'is_active' => 'boolean',
            'is_test_mode' => 'boolean',
            'supported_currencies' => 'nullable|array',
            'supported_methods' => 'nullable|array',
            'transaction_fee_percent' => 'nullable|numeric|min:0|max:100',
            'transaction_fee_fixed' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'sort_order' => 'integer|min:0',
        ]);

        try {
            $paymentGateway->update($validated);
            
            Log::info('Payment gateway updated', [
                'gateway_id' => $paymentGateway->id,
                'gateway_name' => $paymentGateway->name,
                'admin_id' => auth()->id(),
            ]);

            return redirect()->route('admin.payment-gateways.index')
                ->with('success', 'Payment gateway updated successfully!');
        } catch (\Exception $e) {
            Log::error('Error updating payment gateway: ' . $e->getMessage());
            return back()->with('error', 'Failed to update payment gateway. Please try again.');
        }
    }

    /**
     * Toggle the active status of the specified payment gateway.
     */
    public function toggleStatus(PaymentGateway $paymentGateway)
    {
        try {
            $paymentGateway->is_active = !$paymentGateway->is_active;
            $paymentGateway->save();

            Log::info('Payment gateway status toggled', [
                'gateway_id' => $paymentGateway->id,
                'gateway_name' => $paymentGateway->name,
                'is_active' => $paymentGateway->is_active,
                'admin_id' => auth()->id(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Payment gateway status updated successfully!',
                'status' => $paymentGateway->is_active ? 'Active' : 'Inactive',
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling payment gateway status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update payment gateway status.',
            ], 500);
        }
    }

    /**
     * Remove the specified payment gateway from storage.
     */
    public function destroy(PaymentGateway $paymentGateway)
    {
        try {
            $gatewayName = $paymentGateway->name;
            $paymentGateway->delete();

            Log::info('Payment gateway deleted', [
                'gateway_id' => $paymentGateway->id,
                'gateway_name' => $gatewayName,
                'admin_id' => auth()->id(),
            ]);

            return redirect()->route('admin.payment-gateways.index')
                ->with('success', 'Payment gateway deleted successfully!');
        } catch (\Exception $e) {
            Log::error('Error deleting payment gateway: ' . $e->getMessage());
            return back()->with('error', 'Failed to delete payment gateway. Please try again.');
        }
    }

    /**
     * Display payment methods for a specific gateway.
     */
    public function paymentMethods(PaymentGateway $paymentGateway)
    {
        $paymentMethods = $paymentGateway->paymentMethods()->orderBy('sort_order')->get();
        return view('admin.payment_gateways.payment_methods', compact('paymentGateway', 'paymentMethods'));
    }

    /**
     * Add a payment method to a gateway.
     */
    public function addPaymentMethod(Request $request, PaymentGateway $paymentGateway)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:100',
            'slug' => 'required|string|max:50|unique:payment_methods,slug',
            'icon' => 'nullable|string',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        try {
            $paymentMethod = PaymentMethod::create(array_merge($validated, [
                'gateway_slug' => $paymentGateway->slug,
            ]));

            Log::info('Payment method added to gateway', [
                'gateway_id' => $paymentGateway->id,
                'gateway_name' => $paymentGateway->name,
                'payment_method_id' => $paymentMethod->id,
                'payment_method_name' => $paymentMethod->name,
                'admin_id' => auth()->id(),
            ]);

            return redirect()->route('admin.payment-gateways.payment-methods', $paymentGateway)
                ->with('success', 'Payment method added successfully!');
        } catch (\Exception $e) {
            Log::error('Error adding payment method: ' . $e->getMessage());
            return back()->with('error', 'Failed to add payment method. Please try again.');
        }
    }

    /**
     * Toggle payment method status.
     */
    public function togglePaymentMethod(PaymentMethod $paymentMethod)
    {
        try {
            $paymentMethod->is_active = !$paymentMethod->is_active;
            $paymentMethod->save();

            return response()->json([
                'success' => true,
                'message' => 'Payment method status updated successfully!',
                'status' => $paymentMethod->is_active ? 'Active' : 'Inactive',
            ]);
        } catch (\Exception $e) {
            Log::error('Error toggling payment method status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'message' => 'Failed to update payment method status.',
            ], 500);
        }
    }

    /**
     * Display payment gateway analytics.
     */
    public function analytics(PaymentGateway $paymentGateway)
    {
        $transactions = $paymentGateway->transactions()->latest()->paginate(50);
        $stats = [
            'total_transactions' => $paymentGateway->transactions()->count(),
            'successful_transactions' => $paymentGateway->transactions()->where('status', 'success')->count(),
            'failed_transactions' => $paymentGateway->transactions()->where('status', 'failed')->count(),
            'total_revenue' => $paymentGateway->transactions()->where('status', 'success')->sum('amount'),
            'total_gateway_fees' => $paymentGateway->transactions()->sum('gateway_fee'),
        ];

        return view('admin.payment_gateways.analytics', compact('paymentGateway', 'transactions', 'stats'));
    }
}
