<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Page;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class PagesController extends Controller
{
    /**
     * Display a listing of pages.
     */
    public function index()
    {
        $pages = Page::ordered()->get();
        return view('admin.pages.index', compact('pages'));
    }

    /**
     * Show the form for creating a new page.
     */
    public function create()
    {
        return view('admin.pages.create');
    }

    /**
     * Store a newly created page.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'nullable|string|max:255|unique:pages,slug',
            'content' => 'nullable|string',
            'is_active' => 'boolean',
            'show_in_app' => 'boolean',
            'order' => 'nullable|integer',
        ]);

        // Auto-generate slug if not provided
        if (empty($validated['slug'])) {
            $validated['slug'] = Str::slug($validated['title']);
        }

        // Set defaults
        $validated['is_active'] = $request->has('is_active');
        $validated['show_in_app'] = $request->has('show_in_app') ? true : false;
        $validated['is_system_page'] = false; // Only seeded pages are system pages
        $validated['order'] = $validated['order'] ?? 0;

        Page::create($validated);

        return redirect()->route('admin.pages.index')
            ->with('success', 'Page created successfully.');
    }

    /**
     * Show the form for editing the specified page.
     */
    public function edit(Page $page)
    {
        return view('admin.pages.edit', compact('page'));
    }

    /**
     * Update the specified page.
     */
    public function update(Request $request, Page $page)
    {
        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:pages,slug,' . $page->id,
            'content' => 'nullable|string',
            'is_active' => 'boolean',
            'show_in_app' => 'boolean',
            'order' => 'nullable|integer',
        ]);

        $validated['is_active'] = $request->has('is_active');
        $validated['show_in_app'] = $request->has('show_in_app') ? true : false;
        $validated['order'] = $validated['order'] ?? $page->order;
        // Note: is_system_page cannot be changed via admin UI

        $page->update($validated);

        return redirect()->route('admin.pages.index')
            ->with('success', 'Page updated successfully.');
    }

    /**
     * Remove the specified page.
     */
    public function destroy(Page $page)
    {
        // Prevent deletion of system pages
        if ($page->is_system_page) {
            return redirect()->route('admin.pages.index')
                ->with('error', 'System pages cannot be deleted. You can disable them instead.');
        }

        $page->delete();

        return redirect()->route('admin.pages.index')
            ->with('success', 'Page deleted successfully.');
    }

    /**
     * Toggle page visibility in app via AJAX
     */
    public function toggleAppVisibility(Request $request, Page $page)
    {
        $validated = $request->validate([
            'show_in_app' => 'required|boolean',
        ]);

        $page->update(['show_in_app' => $validated['show_in_app']]);

        return response()->json([
            'success' => true,
            'message' => 'Page visibility updated successfully',
            'show_in_app' => $page->show_in_app
        ]);
    }
}
