<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class PackagesController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $packages = \App\Models\AdPackage::ordered()->get();
        return view('admin.packages.index', compact('packages'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.packages.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:ad_packages,slug',
            'type' => 'required|in:multiple_ads,featured,boost,combo',
            'description' => 'nullable|string',
            'features' => 'array',
            'algorithm_params' => 'array',
            'sort_order' => 'integer',
            'is_active' => 'boolean',
            'icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        // Default empty arrays if null
        $validated['features'] = $validated['features'] ?? [];
        $validated['algorithm_params'] = $validated['algorithm_params'] ?? [];
        $validated['is_active'] = $request->has('is_active');

        if ($request->hasFile('icon')) {
            $path = $request->file('icon')->store('packages', 'public');
            $validated['icon'] = $path;
        }

        \App\Models\AdPackage::create($validated);

        return redirect()->route('admin.packages.index')
            ->with('success', 'Package created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $package = \App\Models\AdPackage::findOrFail($id);
        return view('admin.packages.edit', compact('package'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $package = \App\Models\AdPackage::findOrFail($id);

        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'slug' => 'required|string|max:255|unique:ad_packages,slug,' . $id,
            'type' => 'required|in:multiple_ads,featured,boost,combo',
            'description' => 'nullable|string',
            'features' => 'array',
            'algorithm_params' => 'array',
            'sort_order' => 'integer',
            'is_active' => 'boolean',
            'icon' => 'nullable|image|mimes:jpeg,png,jpg,gif,svg|max:2048',
        ]);

        $validated['is_active'] = $request->has('is_active');
        // Ensure arrays
        $validated['features'] = $validated['features'] ?? [];
        $validated['algorithm_params'] = $validated['algorithm_params'] ?? [];

        if ($request->hasFile('icon')) {
            // Delete old icon
            if ($package->icon) {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($package->icon);
            }
            $path = $request->file('icon')->store('packages', 'public');
            $validated['icon'] = $path;
        }

        $package->update($validated);

        return redirect()->route('admin.packages.index')
            ->with('success', 'Package updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $package = \App\Models\AdPackage::findOrFail($id);
        $package->delete();

        return redirect()->route('admin.packages.index')
            ->with('success', 'Package deleted successfully.');
    }
}
