<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\GlobalField;
use Illuminate\Http\Request;

class GlobalFieldController extends Controller
{
    /**
     * Display the Global Fields Library page.
     */
    public function index()
    {
        return view('admin.fields.index');
    }

    /**
     * Get all global fields (API).
     */
    public function list()
    {
        $fields = GlobalField::orderBy('name')->get();
        return response()->json($fields);
    }

    /**
     * Store a new global field.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:text,number,dropdown,radio,checkbox,date',
            'options' => 'nullable|array',
            'icon' => 'nullable|string',
        ]);

        $field = GlobalField::create([
            'name' => $validated['name'],
            'slug' => str()->slug($validated['name']),
            'type' => $validated['type'],
            'options' => $request->options,
            'icon' => $request->icon,
            'is_active' => true,
        ]);

        return response()->json(['message' => 'Field Definition created successfully', 'field' => $field]);
    }

    /**
     * Update a global field.
     */
    public function update(Request $request, $id)
    {
        $field = GlobalField::findOrFail($id);
        
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:text,number,dropdown,radio,checkbox,date',
            'options' => 'nullable|array',
            'icon' => 'nullable|string',
        ]);

        $field->update([
            'name' => $validated['name'],
            // 'slug' => str()->slug($validated['name']), // Keep slug stable usually? Or update? Let's update.
            'type' => $validated['type'],
            'options' => $request->options,
            'icon' => $request->icon,
        ]);

        return response()->json(['message' => 'Field Definition updated successfully', 'field' => $field]);
    }

    /**
     * Delete a global field.
     */
    public function destroy($id)
    {
        $field = GlobalField::findOrFail($id);
        // Optional: Check if used in categories? 
        // For now, let's allow delete, cascading set null on CategoryFields (defined in migration)
        $field->delete();
        return response()->json(['message' => 'Field Definition deleted successfully']);
    }
}
